/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtgui_python.h"

#include "qimage_wrapper.h"

// Extra includes
#include <QMatrix>
#include <QStringList>
#include <QVector>
#include <qbytearray.h>
#include <qdatastream.h>
#include <qimage.h>
#include <qiodevice.h>
#include <qmatrix.h>
#include <qpaintdevice.h>
#include <qpaintengine.h>
#include <qpoint.h>
#include <qrect.h>
#include <qsize.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qtransform.h>
#include <qvariant.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QImageWrapper::QImageWrapper() : QImage() {
    // ... middle
}

QImageWrapper::QImageWrapper(const QSize & size, QImage::Format format) : QImage(size, format) {
    // ... middle
}

QImageWrapper::QImageWrapper(const QString & fileName, const char * format) : QImage(fileName, format) {
    // ... middle
}

QImageWrapper::QImageWrapper(int width, int height, QImage::Format format) : QImage(width, height, format) {
    // ... middle
}

int QImageWrapper::devType() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "devType"));
    if (py_override.isNull()) {
        return this->QImage::devType();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImage.devType", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

int QImageWrapper::metric(QPaintDevice::PaintDeviceMetric metric) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "metric"));
    if (py_override.isNull()) {
        return this->QImage::metric(metric);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QPaintDevice::PaintDeviceMetric >::toPython(metric)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImage.metric", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

QPaintEngine * QImageWrapper::paintEngine() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "paintEngine"));
    if (py_override.isNull()) {
        return this->QImage::paintEngine();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return 0;
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QPaintEngine * >::isConvertible(py_result);
    typeIsValid = typeIsValid || (py_result == Py_None);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImage.paintEngine", SbkType<QPaintEngine * >()->tp_name, py_result->ob_type->tp_name);
        return 0;
    }
    QPaintEngine * cpp_result(Shiboken::Converter<QPaintEngine * >::toCpp(py_result));
    return cpp_result;
}

QImageWrapper::~QImageWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQImage_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QImageWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "QImage", 0, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return -1;

    if (numArgs == 0) {
        // QImage()
        cptr = new QImageWrapper();
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 3 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && Shiboken::Converter<QImage::Format >::isConvertible(pyargs[2])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        QImage::Format cpp_arg2 = Shiboken::Converter<QImage::Format >::toCpp(pyargs[2]);
        // QImage(int,int,QImage::Format)
        cptr = new QImageWrapper(cpp_arg0, cpp_arg1, cpp_arg2);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // QImage(QString,const char*)
            cptr = new QImageWrapper(*cpp_arg0);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            // QImage(QString,const char*)
            cptr = new QImageWrapper(*cpp_arg0, cpp_arg1);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImage_Init_TypeError;
    } else if (numArgs == 2 && Shiboken::Converter<QSize & >::isConvertible(pyargs[0]) && Shiboken::Converter<QImage::Format >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QSize* cpp_arg0 = Shiboken::Converter<QSize* >::toCpp(pyargs[0]);
        QImage::Format cpp_arg1 = Shiboken::Converter<QImage::Format >::toCpp(pyargs[1]);
        // QImage(QSize,QImage::Format)
        cptr = new QImageWrapper(*cpp_arg0, cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QImage & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QImage* cpp_arg0 = Shiboken::Converter<QImage* >::toCpp(pyargs[0]);
        // QImage(QImage)
        cptr = new QImageWrapper(*reinterpret_cast<QImageWrapper*>(cpp_arg0));
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImage_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QImage >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQImage_Init_TypeError;

    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    sbkSelf->referredObjects = new Shiboken::RefCountMap;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQImage_Init_TypeError:
        const char* overloads[] = {"", "PySide.QtGui.QImage", "PySide.QtCore.QSize, PySide.QtGui.QImage.Format", "PySide.QtCore.QString, str = None", "int, int, PySide.QtGui.QImage.Format", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage", overloads);
        return -1;
}

static PyObject* SbkQImageFunc_allGray(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // allGray()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QImage::allGray();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_alphaChannel(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // alphaChannel()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QImage cpp_result = cppSelf->QImage::alphaChannel();
    py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_byteCount(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // byteCount()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::byteCount();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_bytesPerLine(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // bytesPerLine()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::bytesPerLine();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_cacheKey(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // cacheKey()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    qint64 cpp_result = cppSelf->QImage::cacheKey();
    py_result = Shiboken::Converter<long long >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_color(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // color(int)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        unsigned int cpp_result = cppSelf->QImage::color(cpp_arg0);
        py_result = Shiboken::Converter<unsigned int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_color_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_color_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.color", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_colorCount(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // colorCount()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::colorCount();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_colorTable(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // colorTable()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QVector<unsigned int > cpp_result = cppSelf->QImage::colorTable();
    py_result = Shiboken::Converter<QVector<unsigned int > >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_convertToFormat(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "convertToFormat", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<QImage::Format >::isConvertible(pyargs[0])) {
        QImage::Format cpp_arg0 = Shiboken::Converter<QImage::Format >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // convertToFormat(QImage::Format,QFlags<Qt::ImageConversionFlag>)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::convertToFormat(cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QVector<unsigned int >  >::isConvertible(pyargs[1])) {
            QVector<unsigned int >  cpp_arg1 = Shiboken::Converter<QVector<unsigned int >  >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // convertToFormat(QImage::Format,QVector<uint>,QFlags<Qt::ImageConversionFlag>)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                QImage cpp_result = cppSelf->QImage::convertToFormat(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (SbkPySide_QtCore_Qt_ImageConversionFlag_Check(pyargs[2])) {
                QFlags<Qt::ImageConversionFlag> cpp_arg2 = Shiboken::Converter<QFlags<Qt::ImageConversionFlag> >::toCpp(pyargs[2]);
                // convertToFormat(QImage::Format,QVector<uint>,QFlags<Qt::ImageConversionFlag>)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                QImage cpp_result = cppSelf->QImage::convertToFormat(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQImageFunc_convertToFormat_TypeError;
        } else if (SbkPySide_QtCore_Qt_ImageConversionFlag_Check(pyargs[1])) {
            QFlags<Qt::ImageConversionFlag> cpp_arg1 = Shiboken::Converter<QFlags<Qt::ImageConversionFlag> >::toCpp(pyargs[1]);
            // convertToFormat(QImage::Format,QFlags<Qt::ImageConversionFlag>)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::convertToFormat(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_convertToFormat_TypeError;
    } else goto SbkQImageFunc_convertToFormat_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_convertToFormat_TypeError:
        const char* overloads[] = {"PySide.QtGui.QImage.Format, PySide.QtCore.Qt.ImageConversionFlags = Qt.AutoColor", "PySide.QtGui.QImage.Format, list, PySide.QtCore.Qt.ImageConversionFlags = Qt.AutoColor", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.convertToFormat", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_copy(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 2 || numArgs == 3)
        goto SbkQImageFunc_copy_TypeError;

    if (!PyArg_UnpackTuple(args, "copy", 0, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 0) {
        // copy(QRect)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QImage cpp_result = cppSelf->QImage::copy();
        py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 4 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
        int cpp_arg3 = Shiboken::Converter<int >::toCpp(pyargs[3]);
        // copy(int,int,int,int)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QImage cpp_result = cppSelf->QImage::copy(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
        py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QRect & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRect* cpp_arg0 = Shiboken::Converter<QRect* >::toCpp(pyargs[0]);
        // copy(QRect)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QImage cpp_result = cppSelf->QImage::copy(*cpp_arg0);
        py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_copy_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_copy_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRect = QRect()", "int, int, int, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.copy", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_createAlphaMask(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // createAlphaMask(QFlags<Qt::ImageConversionFlag>)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QImage cpp_result = cppSelf->QImage::createAlphaMask();
        py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (SbkPySide_QtCore_Qt_ImageConversionFlag_Check(arg)) {
        QFlags<Qt::ImageConversionFlag> cpp_arg0 = Shiboken::Converter<QFlags<Qt::ImageConversionFlag> >::toCpp(arg);
        // createAlphaMask(QFlags<Qt::ImageConversionFlag>)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QImage cpp_result = cppSelf->QImage::createAlphaMask(cpp_arg0);
        py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_createAlphaMask_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_createAlphaMask_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.ImageConversionFlags = Qt.AutoColor", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.createAlphaMask", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_createHeuristicMask(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // createHeuristicMask(bool)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QImage cpp_result = cppSelf->QImage::createHeuristicMask();
        py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // createHeuristicMask(bool)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QImage cpp_result = cppSelf->QImage::createHeuristicMask(cpp_arg0);
        py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_createHeuristicMask_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_createHeuristicMask_TypeError:
        const char* overloads[] = {"bool = true", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.createHeuristicMask", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_createMaskFromColor(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "createMaskFromColor", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        unsigned int cpp_arg0 = Shiboken::Converter<unsigned int >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // createMaskFromColor(uint,Qt::MaskMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::createMaskFromColor(cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::MaskMode >::isConvertible(pyargs[1])) {
            Qt::MaskMode cpp_arg1 = Shiboken::Converter<Qt::MaskMode >::toCpp(pyargs[1]);
            // createMaskFromColor(uint,Qt::MaskMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::createMaskFromColor(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_createMaskFromColor_TypeError;
    } else goto SbkQImageFunc_createMaskFromColor_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_createMaskFromColor_TypeError:
        const char* overloads[] = {"unsigned int, PySide.QtCore.Qt.MaskMode = Qt.MaskInColor", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.createMaskFromColor", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_depth(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // depth()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::depth();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_devType(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // devType()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::devType();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_dotsPerMeterX(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // dotsPerMeterX()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::dotsPerMeterX();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_dotsPerMeterY(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // dotsPerMeterY()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::dotsPerMeterY();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_fill(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        uint cpp_arg0 = Shiboken::Converter<uint >::toCpp(arg);
        // fill(uint)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::fill(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_fill_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_fill_TypeError:
        const char* overloads[] = {"unsigned int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.fill", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_format(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // format()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QImage::Format cpp_result = cppSelf->QImage::format();
    py_result = Shiboken::Converter<QImage::Format >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_fromData(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "fromData", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QByteArray & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(pyargs[0]);
        if (!SbkQByteArray_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        if (numArgs == 1) {
            // fromData(QByteArray,const char*)
            QImage cpp_result = QImage::fromData(*cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            // fromData(QByteArray,const char*)
            QImage cpp_result = QImage::fromData(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_fromData_TypeError;
    } else goto SbkQImageFunc_fromData_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_fromData_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray, str = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.fromData", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_hasAlphaChannel(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // hasAlphaChannel()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QImage::hasAlphaChannel();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_height(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // height()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::height();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_invertPixels(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // invertPixels(QImage::InvertMode)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::invertPixels();
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QImage::InvertMode >::isConvertible(arg)) {
        QImage::InvertMode cpp_arg0 = Shiboken::Converter<QImage::InvertMode >::toCpp(arg);
        // invertPixels(QImage::InvertMode)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::invertPixels(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_invertPixels_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_invertPixels_TypeError:
        const char* overloads[] = {"PySide.QtGui.QImage.InvertMode = InvertRgb", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.invertPixels", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_isGrayscale(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isGrayscale()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QImage::isGrayscale();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_isNull(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isNull()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QImage::isNull();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_load(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "load", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // load(QString,const char*)
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QImage::load(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            // load(QString,const char*)
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QImage::load(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_load_TypeError;
    } else if (numArgs == 2 && Shiboken::Converter<QIODevice * >::isConvertible(pyargs[0]) && Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QIODevice* cpp_arg0 = Shiboken::Converter<QIODevice* >::toCpp(pyargs[0]);
        const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
        // load(QIODevice*,const char*)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QImage::load(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_load_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_load_TypeError:
        const char* overloads[] = {"PySide.QtCore.QIODevice, str", "PySide.QtCore.QString, str = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.load", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_loadFromData(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "loadFromData", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QByteArray & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(pyargs[0]);
        if (!SbkQByteArray_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        if (numArgs == 1) {
            // loadFromData(QByteArray,const char*)
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QImage::loadFromData(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            // loadFromData(QByteArray,const char*)
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QImage::loadFromData(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_loadFromData_TypeError;
    } else goto SbkQImageFunc_loadFromData_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_loadFromData_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray, str = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.loadFromData", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_metric(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QPaintDevice::PaintDeviceMetric >::isConvertible(arg)) {
        QPaintDevice::PaintDeviceMetric cpp_arg0 = Shiboken::Converter<QPaintDevice::PaintDeviceMetric >::toCpp(arg);
        // metric(QPaintDevice::PaintDeviceMetric)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        int cpp_result = cppSelf->QImage::metric(cpp_arg0);
        py_result = Shiboken::Converter<int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_metric_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_metric_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPaintDevice.PaintDeviceMetric", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.metric", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_mirrored(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "mirrored", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 0) {
        // mirrored(bool,bool)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QImage cpp_result = cppSelf->QImage::mirrored();
        py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(pyargs[0])) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // mirrored(bool,bool)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::mirrored(cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            bool cpp_arg1 = Shiboken::Converter<bool >::toCpp(pyargs[1]);
            // mirrored(bool,bool)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::mirrored(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_mirrored_TypeError;
    } else goto SbkQImageFunc_mirrored_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_mirrored_TypeError:
        const char* overloads[] = {"bool = false, bool = true", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.mirrored", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_numBytes(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // numBytes()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::numBytes();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_numColors(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // numColors()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::numColors();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_offset(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // offset()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QPoint cpp_result = cppSelf->QImage::offset();
    py_result = Shiboken::Converter<QPoint >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_paintEngine(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // paintEngine()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QPaintEngine * cpp_result = cppSelf->QImage::paintEngine();
    py_result = Shiboken::Converter<QPaintEngine * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_pixel(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "pixel", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // pixel(int,int)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        unsigned int cpp_result = cppSelf->QImage::pixel(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<unsigned int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QPoint & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(pyargs[0]);
        // pixel(QPoint)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        unsigned int cpp_result = cppSelf->QImage::pixel(*cpp_arg0);
        py_result = Shiboken::Converter<unsigned int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_pixel_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_pixel_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPoint", "int, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.pixel", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_pixelIndex(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "pixelIndex", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // pixelIndex(int,int)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        int cpp_result = cppSelf->QImage::pixelIndex(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QPoint & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(pyargs[0]);
        // pixelIndex(QPoint)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        int cpp_result = cppSelf->QImage::pixelIndex(*cpp_arg0);
        py_result = Shiboken::Converter<int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_pixelIndex_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_pixelIndex_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPoint", "int, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.pixelIndex", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_rect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // rect()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QRect cpp_result = cppSelf->QImage::rect();
    py_result = Shiboken::Converter<QRect >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_rgbSwapped(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // rgbSwapped()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QImage cpp_result = cppSelf->QImage::rgbSwapped();
    py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_save(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "save", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // save(QString,const char*,int)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QImage::save(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // save(QString,const char*,int)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QImage::save(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                // save(QString,const char*,int)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QImage::save(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQImageFunc_save_TypeError;
        } else goto SbkQImageFunc_save_TypeError;
    } else if (Shiboken::Converter<QIODevice * >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QIODevice* cpp_arg0 = Shiboken::Converter<QIODevice* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // save(QIODevice*,const char*,int)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QImage::save(cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // save(QIODevice*,const char*,int)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QImage::save(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                // save(QIODevice*,const char*,int)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QImage::save(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQImageFunc_save_TypeError;
        } else goto SbkQImageFunc_save_TypeError;
    } else goto SbkQImageFunc_save_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_save_TypeError:
        const char* overloads[] = {"PySide.QtCore.QIODevice, str = None, int = -1", "PySide.QtCore.QString, str = None, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.save", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_scaled(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "scaled", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // scaled(int,int,Qt::AspectRatioMode,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::scaled(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::AspectRatioMode >::isConvertible(pyargs[2])) {
            Qt::AspectRatioMode cpp_arg2 = Shiboken::Converter<Qt::AspectRatioMode >::toCpp(pyargs[2]);
            if (numArgs == 3) {
                // scaled(int,int,Qt::AspectRatioMode,Qt::TransformationMode)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                QImage cpp_result = cppSelf->QImage::scaled(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::TransformationMode >::isConvertible(pyargs[3])) {
                Qt::TransformationMode cpp_arg3 = Shiboken::Converter<Qt::TransformationMode >::toCpp(pyargs[3]);
                // scaled(int,int,Qt::AspectRatioMode,Qt::TransformationMode)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                QImage cpp_result = cppSelf->QImage::scaled(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
                py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQImageFunc_scaled_TypeError;
        } else goto SbkQImageFunc_scaled_TypeError;
    } else if (Shiboken::Converter<QSize & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QSize* cpp_arg0 = Shiboken::Converter<QSize* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // scaled(QSize,Qt::AspectRatioMode,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::scaled(*cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::AspectRatioMode >::isConvertible(pyargs[1])) {
            Qt::AspectRatioMode cpp_arg1 = Shiboken::Converter<Qt::AspectRatioMode >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // scaled(QSize,Qt::AspectRatioMode,Qt::TransformationMode)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                QImage cpp_result = cppSelf->QImage::scaled(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::TransformationMode >::isConvertible(pyargs[2])) {
                Qt::TransformationMode cpp_arg2 = Shiboken::Converter<Qt::TransformationMode >::toCpp(pyargs[2]);
                // scaled(QSize,Qt::AspectRatioMode,Qt::TransformationMode)const
                QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
                QImage cpp_result = cppSelf->QImage::scaled(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQImageFunc_scaled_TypeError;
        } else goto SbkQImageFunc_scaled_TypeError;
    } else goto SbkQImageFunc_scaled_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_scaled_TypeError:
        const char* overloads[] = {"PySide.QtCore.QSize, PySide.QtCore.Qt.AspectRatioMode = Qt.IgnoreAspectRatio, PySide.QtCore.Qt.TransformationMode = Qt.FastTransformation", "int, int, PySide.QtCore.Qt.AspectRatioMode = Qt.IgnoreAspectRatio, PySide.QtCore.Qt.TransformationMode = Qt.FastTransformation", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.scaled", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_scaledToHeight(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "scaledToHeight", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // scaledToHeight(int,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::scaledToHeight(cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::TransformationMode >::isConvertible(pyargs[1])) {
            Qt::TransformationMode cpp_arg1 = Shiboken::Converter<Qt::TransformationMode >::toCpp(pyargs[1]);
            // scaledToHeight(int,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::scaledToHeight(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_scaledToHeight_TypeError;
    } else goto SbkQImageFunc_scaledToHeight_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_scaledToHeight_TypeError:
        const char* overloads[] = {"int, PySide.QtCore.Qt.TransformationMode = Qt.FastTransformation", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.scaledToHeight", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_scaledToWidth(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "scaledToWidth", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // scaledToWidth(int,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::scaledToWidth(cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::TransformationMode >::isConvertible(pyargs[1])) {
            Qt::TransformationMode cpp_arg1 = Shiboken::Converter<Qt::TransformationMode >::toCpp(pyargs[1]);
            // scaledToWidth(int,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::scaledToWidth(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_scaledToWidth_TypeError;
    } else goto SbkQImageFunc_scaledToWidth_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_scaledToWidth_TypeError:
        const char* overloads[] = {"int, PySide.QtCore.Qt.TransformationMode = Qt.FastTransformation", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.scaledToWidth", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setAlphaChannel(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QImage & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QImage* cpp_arg0 = Shiboken::Converter<QImage* >::toCpp(arg);
        // setAlphaChannel(QImage)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setAlphaChannel(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setAlphaChannel_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setAlphaChannel_TypeError:
        const char* overloads[] = {"PySide.QtGui.QImage", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.setAlphaChannel", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setColor(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setColor", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        unsigned int cpp_arg1 = Shiboken::Converter<unsigned int >::toCpp(pyargs[1]);
        // setColor(int,uint)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setColor(cpp_arg0, cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setColor_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setColor_TypeError:
        const char* overloads[] = {"int, unsigned int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.setColor", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setColorCount(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // setColorCount(int)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setColorCount(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setColorCount_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setColorCount_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.setColorCount", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setDotsPerMeterX(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // setDotsPerMeterX(int)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setDotsPerMeterX(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setDotsPerMeterX_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setDotsPerMeterX_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.setDotsPerMeterX", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setDotsPerMeterY(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // setDotsPerMeterY(int)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setDotsPerMeterY(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setDotsPerMeterY_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setDotsPerMeterY_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.setDotsPerMeterY", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setNumColors(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // setNumColors(int)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setNumColors(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setNumColors_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setNumColors_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.setNumColors", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setOffset(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QPoint & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(arg);
        // setOffset(QPoint)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setOffset(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setOffset_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setOffset_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPoint", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.setOffset", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setPixel(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "setPixel", 2, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        uint cpp_arg2 = Shiboken::Converter<uint >::toCpp(pyargs[2]);
        // setPixel(int,int,uint)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setPixel(cpp_arg0, cpp_arg1, cpp_arg2);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 2 && Shiboken::Converter<QPoint & >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(pyargs[0]);
        uint cpp_arg1 = Shiboken::Converter<uint >::toCpp(pyargs[1]);
        // setPixel(QPoint,uint)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setPixel(*cpp_arg0, cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setPixel_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setPixel_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPoint, unsigned int", "int, int, unsigned int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.setPixel", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_setText(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setText", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QString & >::isConvertible(pyargs[0]) && Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QString > cpp_arg1_auto_ptr;
        QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
        if (!SbkQString_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
        // setText(QString,QString)
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        cppSelf->QImage::setText(*cpp_arg0, *cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_setText_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageFunc_setText_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString, PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.setText", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_size(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // size()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QSize cpp_result = cppSelf->QImage::size();
    py_result = Shiboken::Converter<QSize >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_text(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // text(QString)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QImage::text();
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // text(QString)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QImage::text(*cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_text_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_text_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString = QString()", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.text", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_textKeys(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // textKeys()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    QStringList cpp_result = cppSelf->QImage::textKeys();
    py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageFunc_transformed(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "transformed", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QTransform & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QTransform* cpp_arg0 = Shiboken::Converter<QTransform* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // transformed(QTransform,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::transformed(*cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::TransformationMode >::isConvertible(pyargs[1])) {
            Qt::TransformationMode cpp_arg1 = Shiboken::Converter<Qt::TransformationMode >::toCpp(pyargs[1]);
            // transformed(QTransform,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::transformed(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_transformed_TypeError;
    } else if (Shiboken::Converter<QMatrix & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QMatrix* cpp_arg0 = Shiboken::Converter<QMatrix* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // transformed(QMatrix,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::transformed(*cpp_arg0);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::TransformationMode >::isConvertible(pyargs[1])) {
            Qt::TransformationMode cpp_arg1 = Shiboken::Converter<Qt::TransformationMode >::toCpp(pyargs[1]);
            // transformed(QMatrix,Qt::TransformationMode)const
            QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
            QImage cpp_result = cppSelf->QImage::transformed(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QImage >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQImageFunc_transformed_TypeError;
    } else goto SbkQImageFunc_transformed_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_transformed_TypeError:
        const char* overloads[] = {"PySide.QtGui.QMatrix, PySide.QtCore.Qt.TransformationMode = Qt.FastTransformation", "PySide.QtGui.QTransform, PySide.QtCore.Qt.TransformationMode = Qt.FastTransformation", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.transformed", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_trueMatrix(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "trueMatrix", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && Shiboken::Converter<QTransform & >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QTransform* cpp_arg0 = Shiboken::Converter<QTransform* >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
        // trueMatrix(QTransform,int,int)
        QTransform cpp_result = QImage::trueMatrix(*cpp_arg0, cpp_arg1, cpp_arg2);
        py_result = Shiboken::Converter<QTransform >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 3 && Shiboken::Converter<QMatrix & >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QMatrix* cpp_arg0 = Shiboken::Converter<QMatrix* >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
        // trueMatrix(QMatrix,int,int)
        QMatrix cpp_result = QImage::trueMatrix(*cpp_arg0, cpp_arg1, cpp_arg2);
        py_result = Shiboken::Converter<QMatrix >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_trueMatrix_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_trueMatrix_TypeError:
        const char* overloads[] = {"PySide.QtGui.QMatrix, int, int", "PySide.QtGui.QTransform, int, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.trueMatrix", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_valid(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "valid", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // valid(int,int)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QImage::valid(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QPoint & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(pyargs[0]);
        // valid(QPoint)const
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QImage::valid(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageFunc_valid_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc_valid_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPoint", "int, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImage.valid", overloads);
        return 0;
}

static PyObject* SbkQImageFunc_width(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // width()const
    QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImage::width();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyMethodDef SbkQImage_methods[] = {
    {"allGray", (PyCFunction)SbkQImageFunc_allGray, METH_NOARGS},
    {"alphaChannel", (PyCFunction)SbkQImageFunc_alphaChannel, METH_NOARGS},
    {"byteCount", (PyCFunction)SbkQImageFunc_byteCount, METH_NOARGS},
    {"bytesPerLine", (PyCFunction)SbkQImageFunc_bytesPerLine, METH_NOARGS},
    {"cacheKey", (PyCFunction)SbkQImageFunc_cacheKey, METH_NOARGS},
    {"color", (PyCFunction)SbkQImageFunc_color, METH_O},
    {"colorCount", (PyCFunction)SbkQImageFunc_colorCount, METH_NOARGS},
    {"colorTable", (PyCFunction)SbkQImageFunc_colorTable, METH_NOARGS},
    {"convertToFormat", (PyCFunction)SbkQImageFunc_convertToFormat, METH_VARARGS},
    {"copy", (PyCFunction)SbkQImageFunc_copy, METH_VARARGS},
    {"createAlphaMask", (PyCFunction)SbkQImageFunc_createAlphaMask, METH_NOARGS|METH_O},
    {"createHeuristicMask", (PyCFunction)SbkQImageFunc_createHeuristicMask, METH_NOARGS|METH_O},
    {"createMaskFromColor", (PyCFunction)SbkQImageFunc_createMaskFromColor, METH_VARARGS},
    {"depth", (PyCFunction)SbkQImageFunc_depth, METH_NOARGS},
    {"devType", (PyCFunction)SbkQImageFunc_devType, METH_NOARGS},
    {"dotsPerMeterX", (PyCFunction)SbkQImageFunc_dotsPerMeterX, METH_NOARGS},
    {"dotsPerMeterY", (PyCFunction)SbkQImageFunc_dotsPerMeterY, METH_NOARGS},
    {"fill", (PyCFunction)SbkQImageFunc_fill, METH_O},
    {"format", (PyCFunction)SbkQImageFunc_format, METH_NOARGS},
    {"fromData", (PyCFunction)SbkQImageFunc_fromData, METH_VARARGS|METH_STATIC},
    {"hasAlphaChannel", (PyCFunction)SbkQImageFunc_hasAlphaChannel, METH_NOARGS},
    {"height", (PyCFunction)SbkQImageFunc_height, METH_NOARGS},
    {"invertPixels", (PyCFunction)SbkQImageFunc_invertPixels, METH_NOARGS|METH_O},
    {"isGrayscale", (PyCFunction)SbkQImageFunc_isGrayscale, METH_NOARGS},
    {"isNull", (PyCFunction)SbkQImageFunc_isNull, METH_NOARGS},
    {"load", (PyCFunction)SbkQImageFunc_load, METH_VARARGS},
    {"loadFromData", (PyCFunction)SbkQImageFunc_loadFromData, METH_VARARGS},
    {"metric", (PyCFunction)SbkQImageFunc_metric, METH_O},
    {"mirrored", (PyCFunction)SbkQImageFunc_mirrored, METH_VARARGS},
    {"numBytes", (PyCFunction)SbkQImageFunc_numBytes, METH_NOARGS},
    {"numColors", (PyCFunction)SbkQImageFunc_numColors, METH_NOARGS},
    {"offset", (PyCFunction)SbkQImageFunc_offset, METH_NOARGS},
    {"paintEngine", (PyCFunction)SbkQImageFunc_paintEngine, METH_NOARGS},
    {"pixel", (PyCFunction)SbkQImageFunc_pixel, METH_VARARGS},
    {"pixelIndex", (PyCFunction)SbkQImageFunc_pixelIndex, METH_VARARGS},
    {"rect", (PyCFunction)SbkQImageFunc_rect, METH_NOARGS},
    {"rgbSwapped", (PyCFunction)SbkQImageFunc_rgbSwapped, METH_NOARGS},
    {"save", (PyCFunction)SbkQImageFunc_save, METH_VARARGS},
    {"scaled", (PyCFunction)SbkQImageFunc_scaled, METH_VARARGS},
    {"scaledToHeight", (PyCFunction)SbkQImageFunc_scaledToHeight, METH_VARARGS},
    {"scaledToWidth", (PyCFunction)SbkQImageFunc_scaledToWidth, METH_VARARGS},
    {"setAlphaChannel", (PyCFunction)SbkQImageFunc_setAlphaChannel, METH_O},
    {"setColor", (PyCFunction)SbkQImageFunc_setColor, METH_VARARGS},
    {"setColorCount", (PyCFunction)SbkQImageFunc_setColorCount, METH_O},
    {"setDotsPerMeterX", (PyCFunction)SbkQImageFunc_setDotsPerMeterX, METH_O},
    {"setDotsPerMeterY", (PyCFunction)SbkQImageFunc_setDotsPerMeterY, METH_O},
    {"setNumColors", (PyCFunction)SbkQImageFunc_setNumColors, METH_O},
    {"setOffset", (PyCFunction)SbkQImageFunc_setOffset, METH_O},
    {"setPixel", (PyCFunction)SbkQImageFunc_setPixel, METH_VARARGS},
    {"setText", (PyCFunction)SbkQImageFunc_setText, METH_VARARGS},
    {"size", (PyCFunction)SbkQImageFunc_size, METH_NOARGS},
    {"text", (PyCFunction)SbkQImageFunc_text, METH_NOARGS|METH_O},
    {"textKeys", (PyCFunction)SbkQImageFunc_textKeys, METH_NOARGS},
    {"transformed", (PyCFunction)SbkQImageFunc_transformed, METH_VARARGS},
    {"trueMatrix", (PyCFunction)SbkQImageFunc_trueMatrix, METH_VARARGS|METH_STATIC},
    {"valid", (PyCFunction)SbkQImageFunc_valid, METH_VARARGS},
    {"width", (PyCFunction)SbkQImageFunc_width, METH_NOARGS},
    {0} // Sentinel
};

static PyObject* SbkQImageFunc___lshift__(PyObject* self, PyObject* arg)
{
    bool isReverse = SbkQImage_Check(arg) && !SbkQImage_Check(self);
    if (isReverse)
        std::swap(self, arg);

    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (!isReverse
        && SbkBaseWrapper_Check(arg)
        && !PyObject_TypeCheck(arg, self->ob_type)
        && PyObject_HasAttrString(arg, const_cast<char*>("__rlshift__"))) {
        PyObject* revOpMethod = PyObject_GetAttrString(arg, const_cast<char*>("__rlshift__"));
        if (revOpMethod && PyCallable_Check(revOpMethod)) {
            py_result = PyObject_CallFunction(revOpMethod, const_cast<char*>("O"), self);
            if (PyErr_Occurred() && (PyErr_ExceptionMatches(PyExc_NotImplementedError) || PyErr_ExceptionMatches(PyExc_AttributeError))) {
                PyErr_Clear();
                Py_XDECREF(py_result);
                py_result = 0;
            }
        }
        Py_XDECREF(revOpMethod);

    }
    // Do not enter here if other object has implemented a reverse operator.
    if (!py_result) {

    if (isReverse && Shiboken::Converter<QDataStream & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QDataStream* cpp_arg0 = Shiboken::Converter<QDataStream* >::toCpp(arg);
        // operator<<(QDataStream&,QImage) [reverse operator]
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QDataStream & cpp_result = (*cpp_arg0) << (*cppSelf);
        py_result = Shiboken::Converter<QDataStream & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, py_result);
    } else goto SbkQImageFunc___lshift___TypeError;

    } // End of "if (!py_result)"


    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc___lshift___TypeError:
        const char* overloads[] = {"PySide.QtCore.QDataStream", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.__lshift__", overloads);
        return 0;
}

static PyObject* SbkQImageFunc___rshift__(PyObject* self, PyObject* arg)
{
    bool isReverse = SbkQImage_Check(arg) && !SbkQImage_Check(self);
    if (isReverse)
        std::swap(self, arg);

    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (!isReverse
        && SbkBaseWrapper_Check(arg)
        && !PyObject_TypeCheck(arg, self->ob_type)
        && PyObject_HasAttrString(arg, const_cast<char*>("__rrshift__"))) {
        PyObject* revOpMethod = PyObject_GetAttrString(arg, const_cast<char*>("__rrshift__"));
        if (revOpMethod && PyCallable_Check(revOpMethod)) {
            py_result = PyObject_CallFunction(revOpMethod, const_cast<char*>("O"), self);
            if (PyErr_Occurred() && (PyErr_ExceptionMatches(PyExc_NotImplementedError) || PyErr_ExceptionMatches(PyExc_AttributeError))) {
                PyErr_Clear();
                Py_XDECREF(py_result);
                py_result = 0;
            }
        }
        Py_XDECREF(revOpMethod);

    }
    // Do not enter here if other object has implemented a reverse operator.
    if (!py_result) {

    if (isReverse && Shiboken::Converter<QDataStream & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QDataStream* cpp_arg0 = Shiboken::Converter<QDataStream* >::toCpp(arg);
        // operator>>(QDataStream&,QImage&) [reverse operator]
        QImage* cppSelf = Shiboken::Converter<QImage* >::toCpp((PyObject*)self);
        QDataStream & cpp_result = (*cpp_arg0) >> (*cppSelf);
        py_result = Shiboken::Converter<QDataStream & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, py_result);
    } else goto SbkQImageFunc___rshift___TypeError;

    } // End of "if (!py_result)"


    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageFunc___rshift___TypeError:
        const char* overloads[] = {"PySide.QtCore.QDataStream", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImage.__rshift__", overloads);
        return 0;
}

// type has number operators
static PyNumberMethods SbkQImage_as_number = {
    /*nb_add*/                  (binaryfunc)0,
    /*nb_subtract*/             (binaryfunc)0,
    /*nb_multiply*/             (binaryfunc)0,
    /*nb_divide*/               (binaryfunc)0,
    /*nb_remainder*/            (binaryfunc)0,
    /*nb_divmod*/               0,
    /*nb_power*/                0,
    /*nb_negative*/             (unaryfunc)0,
    /*nb_positive*/             (unaryfunc)0,
    /*nb_absolute*/             0,
    /*nb_nonzero*/              0,
    /*nb_invert*/               (unaryfunc)0,
    /*nb_lshift*/               (binaryfunc)SbkQImageFunc___lshift__,
    /*nb_rshift*/               (binaryfunc)SbkQImageFunc___rshift__,
    /*nb_and*/                  (binaryfunc)0,
    /*nb_xor*/                  (binaryfunc)0,
    /*nb_or*/                   (binaryfunc)0,
    /*nb_coerce*/               0,
    /*nb_int*/                  0,
    /*nb_long*/                 0,
    /*nb_float*/                0,
    /*nb_oct*/                  0,
    /*nb_hex*/                  0,
    /*nb_inplace_add*/          (binaryfunc)0,
    /*nb_inplace_subtract*/     (binaryfunc)0,
    /*nb_inplace_multiply*/     (binaryfunc)0,
    /*nb_inplace_divide*/       (binaryfunc)0,
    /*nb_inplace_remainder*/    (binaryfunc)0,
    /*nb_inplace_power*/        0,
    /*nb_inplace_lshift*/       (binaryfunc)0,
    /*nb_inplace_rshift*/       (binaryfunc)0,
    /*nb_inplace_and*/          (binaryfunc)0,
    /*nb_inplace_xor*/          (binaryfunc)0,
    /*nb_inplace_or*/           (binaryfunc)0,
    /*nb_floor_divide*/         0,
    /*nb_true_divide*/          0,
    /*nb_inplace_floor_divide*/ 0,
    /*nb_inplace_true_divide*/  0,
    /*nb_index*/                0
};

// Rich comparison
static PyObject* SbkQImage_richcompare(PyObject* self, PyObject* other, int op)
{
    bool result = false;
    QImage& cpp_self = *Shiboken::Converter<QImage* >::toCpp((PyObject*)self);

    switch (op) {
        case Py_NE:
            if (Shiboken::Converter<QImage & >::isConvertible(other)) {
                // operator!=(const QImage & arg__1) const
                QImage* cpp_other = Shiboken::Converter<QImage* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QImage  >::toCpp(other) : (cpp_self != (*cpp_other));
            } else goto SbkQImage_RichComparison_TypeError;

            break;
        case Py_EQ:
            if (Shiboken::Converter<QImage & >::isConvertible(other)) {
                // operator==(const QImage & arg__1) const
                QImage* cpp_other = Shiboken::Converter<QImage* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QImage  >::toCpp(other) : (cpp_self == (*cpp_other));
            } else goto SbkQImage_RichComparison_TypeError;

            break;
        default:
            PyErr_SetString(PyExc_NotImplementedError, "operator not implemented.");
            return 0;
    }

    if (result)
        Py_RETURN_TRUE;
    SbkQImage_RichComparison_TypeError:
    Py_RETURN_FALSE;

}

extern "C"
{

static void* SbkQImage_ObjCopierFunc(const void* ptr){
    return new QImageWrapper(*reinterpret_cast<const QImage*>(ptr));
}

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQImage_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtGui.QImage",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        &SbkQImage_as_number,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      SbkQImage_richcompare,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQImage_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             0,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQImage_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          &SbkQImage_ObjCopierFunc,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QImage >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static PyGetSetDef SbkPySide_QtGui_QImage_InvertMode_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QImage_InvertMode_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QImage_InvertMode_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "InvertMode",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QImage_InvertMode_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QImage_InvertMode_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtGui_QImage_InvertMode_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtGui_QImage_Format_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QImage_Format_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QImage_Format_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "Format",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QImage_Format_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QImage_Format_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtGui_QImage_Format_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


PyAPI_FUNC(void) init_QImage(PyObject* module)
{
    SbkPySide_QtGuiTypes[SBK_QIMAGE_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQImage_Type);

    SbkQImage_Type.super.ht_type.tp_base = SbkPySide_QtGuiTypes[SBK_QPAINTDEVICE_IDX];
    // Fill type discovery information
    SbkQImage_Type.type_discovery = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_QtGuiTypes[SBK_QPAINTDEVICE_IDX])->type_discovery;

    if (PyType_Ready((PyTypeObject*)&SbkQImage_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQImage_Type));
    PyModule_AddObject(module, "QImage",
        ((PyObject*)&SbkQImage_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: InvertMode
    SbkPySide_QtGuiTypes[SBK_QIMAGE_INVERTMODE_IDX] = &SbkPySide_QtGui_QImage_InvertMode_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QImage_InvertMode_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QImage_InvertMode_Type);
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
            "InvertMode",((PyObject*)&SbkPySide_QtGui_QImage_InvertMode_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_InvertMode_Type,
        (long) QImage::InvertRgb, "InvertRgb");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "InvertRgb", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_InvertMode_Type.tp_dict,
        "InvertRgb", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_InvertMode_Type,
        (long) QImage::InvertRgba, "InvertRgba");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "InvertRgba", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_InvertMode_Type.tp_dict,
        "InvertRgba", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QImage::InvertMode");

    // init enum class: Format
    SbkPySide_QtGuiTypes[SBK_QIMAGE_FORMAT_IDX] = &SbkPySide_QtGui_QImage_Format_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QImage_Format_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QImage_Format_Type);
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
            "Format",((PyObject*)&SbkPySide_QtGui_QImage_Format_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_Invalid, "Format_Invalid");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_Invalid", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_Invalid", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_Mono, "Format_Mono");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_Mono", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_Mono", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_MonoLSB, "Format_MonoLSB");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_MonoLSB", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_MonoLSB", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_Indexed8, "Format_Indexed8");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_Indexed8", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_Indexed8", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_RGB32, "Format_RGB32");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_RGB32", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_RGB32", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_ARGB32, "Format_ARGB32");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_ARGB32", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_ARGB32", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_ARGB32_Premultiplied, "Format_ARGB32_Premultiplied");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_ARGB32_Premultiplied", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_ARGB32_Premultiplied", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_RGB16, "Format_RGB16");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_RGB16", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_RGB16", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_ARGB8565_Premultiplied, "Format_ARGB8565_Premultiplied");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_ARGB8565_Premultiplied", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_ARGB8565_Premultiplied", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_RGB666, "Format_RGB666");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_RGB666", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_RGB666", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_ARGB6666_Premultiplied, "Format_ARGB6666_Premultiplied");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_ARGB6666_Premultiplied", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_ARGB6666_Premultiplied", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_RGB555, "Format_RGB555");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_RGB555", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_RGB555", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_ARGB8555_Premultiplied, "Format_ARGB8555_Premultiplied");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_ARGB8555_Premultiplied", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_ARGB8555_Premultiplied", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_RGB888, "Format_RGB888");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_RGB888", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_RGB888", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_RGB444, "Format_RGB444");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_RGB444", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_RGB444", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImage_Format_Type,
        (long) QImage::Format_ARGB4444_Premultiplied, "Format_ARGB4444_Premultiplied");
    PyDict_SetItemString(SbkQImage_Type.super.ht_type.tp_dict,
        "Format_ARGB4444_Premultiplied", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImage_Format_Type.tp_dict,
        "Format_ARGB4444_Premultiplied", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QImage::Format");

    Shiboken::TypeResolver::createValueTypeResolver<QImage >("QImage");
    Shiboken::TypeResolver::createValueTypeResolver<QImage >(typeid(QImage).name());
}


} // extern "C"

