/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include <signalmanager.h>
#include <dynamicqmetaobject.h>
#include "pyside_qtcore_python.h"

#include "qobject_wrapper.h"

// Extra includes
#include <QCoreApplication>
#include <QList>
#include <QThread>
#include <qbytearray.h>
#include <qcoreevent.h>
#include <qobject.h>
#include <qobjectdefs.h>
#include <qstring.h>
#include <qthread.h>
#include <qvariant.h>
#include "signalmanager.h"

using namespace Shiboken;

// Begin code injection
// ========================================================================
// START of custom code block [file: glue/qobject_connect.cpp]
static bool getReceiver(PyObject *callback, QObject **receiver, PyObject **self)
{
    if (PyMethod_Check(callback)) {
        *self = PyMethod_GET_SELF(callback);
        if (SbkQObject_Check(*self))
            *receiver = Converter<QObject*>::toCpp(*self);
    } else if (PyCFunction_Check(callback)) {
        *self = PyCFunction_GET_SELF(callback);
        if (*self && SbkQObject_Check(*self))
            *receiver = Converter<QObject*>::toCpp(*self);
    } else if (!PyFunction_Check(callback)) {
        *receiver = 0;
        *self = 0;
        qWarning() << "Invalid callback object.";
        return false;
    }

    bool usingGlobalReceiver = !*receiver;
    if (usingGlobalReceiver) {
        PySide::SignalManager& signalManager = PySide::SignalManager::instance();
        *receiver = signalManager.globalReceiver();
    }

    return usingGlobalReceiver;
}

static bool qobjectConnect(QObject* source, const char* signal, QObject* receiver, const char* slot, Qt::ConnectionType type)
{
    if (!PySide::checkSignal(signal))
        return false;
    signal++;

    PySide::SignalManager::registerMetaMethod(source, signal, QMetaMethod::Signal);

    bool isSignal = PySide::isSignal(slot);
    slot++;
    PySide::SignalManager::registerMetaMethod(receiver, slot, isSignal ? QMetaMethod::Signal : QMetaMethod::Slot);
    return QObject::connect(source, signal - 1, receiver, slot - 1, type);
}

static bool qobjectConnectCallback(QObject* source, const char* signal, PyObject* callback, Qt::ConnectionType type)
{
    if (!PySide::checkSignal(signal))
        return false;
    signal++;

    PySide::SignalManager::registerMetaMethod(source, signal, QMetaMethod::Signal);
    int signalIndex = source->metaObject()->indexOfMethod(signal);

    PySide::SignalManager& signalManager = PySide::SignalManager::instance();

    // Extract receiver from callback
    QObject* receiver = 0;
    PyObject* self = 0;
    bool usingGlobalReceiver = getReceiver(callback, &receiver, &self);
    if (receiver == 0 and self == 0)
        return false;

    const QMetaObject* metaObject = receiver->metaObject();
    const QByteArray callbackSig = PySide::getCallbackSignature(signal, callback, usingGlobalReceiver).toAscii();
    const char* slot = callbackSig.constData();
    int slotIndex = metaObject->indexOfSlot(slot);
    if (slotIndex == -1) {
        if (!usingGlobalReceiver and self and !((SbkBaseWrapper*)self)->containsCppWrapper) {
            qWarning() << "You can't add dynamic slots on an object originated from C++.";
            return false;
        }
        if (usingGlobalReceiver) {
            signalManager.addGlobalSlot(slot, callback);
        } else {
            PySide::SignalManager::registerMetaMethod(receiver, slot, QMetaMethod::Slot);
        }
        slotIndex = metaObject->indexOfSlot(slot);
    }
    if (QMetaObject::connect(source, signalIndex, receiver, slotIndex, type)) {
        // FIXME: Need to cast to QObjectWrapper* and call the public version of connectNotify
        //        when avoiding the protected hack.
        #ifndef AVOID_PROTECTED_HACK
            source->connectNotify(signal);
        #else
            reinterpret_cast<QObjectWrapper*>(source)->connectNotify_protected(signal);
        #endif
        if (usingGlobalReceiver)
            signalManager.globalReceiverConnectNotify(source, slotIndex);

        return true;
    }
    return false;
}


static bool qobjectDisconnectCallback(QObject* source, const char* signal, PyObject* callback)
{
    if (!PySide::checkSignal(signal))
        return false;

    PySide::SignalManager& signalManager = PySide::SignalManager::instance();

    // Extract receiver from callback
    QObject* receiver = 0;
    PyObject* self = 0;
    bool usingGlobalReceiver = getReceiver(callback, &receiver, &self);
    if (receiver == 0 and self == 0)
        return false;

    const QMetaObject* metaObject = receiver->metaObject();
    const QByteArray callbackSig = PySide::getCallbackSignature(signal, callback, usingGlobalReceiver).toAscii();
    QByteArray qtSlotName(callbackSig);
    qtSlotName = qtSlotName.prepend('1');

    if (QObject::disconnect(source, signal, receiver, qtSlotName.constData())) {
        if (usingGlobalReceiver) {
            int slotIndex = metaObject->indexOfSlot(callbackSig.constData());
            signalManager.globalReceiverDisconnectNotify(source, slotIndex);
        }
        return true;
    }
    return false;
}

// END of custom code block [file: glue/qobject_connect.cpp]
// ========================================================================

// End of code injection
// Begin code injection

static bool
qobjectInheritsInternal(PyTypeObject *objType, const char *class_name)
{
    if (strcmp(objType->tp_name, class_name) == 0)
        return true;

    PyTypeObject* base = (objType)->tp_base;
    if (base == 0)
        return false;
    return qobjectInheritsInternal(base, class_name);
}

// End of code injection

// Native ---------------------------------------------------------

QObjectWrapper::QObjectWrapper(QObject * parent) : QObject(parent), m_metaObject(0) {
    // ... middle
}

void QObjectWrapper::childEvent(QChildEvent * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "childEvent"));
    if (py_override.isNull()) {
        return this->QObject::childEvent(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QChildEvent* >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QObjectWrapper::connectNotify(const char * signal)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "connectNotify"));
    if (py_override.isNull()) {
        return this->QObject::connectNotify(signal);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(z)",
        signal
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QObjectWrapper::customEvent(QEvent * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "customEvent"));
    if (py_override.isNull()) {
        return this->QObject::customEvent(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QEvent* >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QObjectWrapper::disconnectNotify(const char * signal)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "disconnectNotify"));
    if (py_override.isNull()) {
        return this->QObject::disconnectNotify(signal);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(z)",
        signal
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

bool QObjectWrapper::event(QEvent * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "event"));
    if (py_override.isNull()) {
        return this->QObject::event(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QEvent* >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QObject.event");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

bool QObjectWrapper::eventFilter(QObject * arg__1, QEvent * arg__2)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "eventFilter"));
    if (py_override.isNull()) {
        return this->QObject::eventFilter(arg__1, arg__2);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QObject* >::toPython(arg__1),
        Shiboken::Converter<QEvent* >::toPython(arg__2)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QObject.eventFilter");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 1));
    return cpp_result;
}

void QObjectWrapper::timerEvent(QTimerEvent * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "timerEvent"));
    if (py_override.isNull()) {
        return this->QObject::timerEvent(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QTimerEvent* >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

const QMetaObject* QObjectWrapper::metaObject() const
{
    if (!m_metaObject) {
        PyObject *pySelf = BindingManager::instance().retrieveWrapper(this);
        QString className(pySelf->ob_type->tp_name);
        className = className.mid(className.lastIndexOf(".")+1);
        m_metaObject = new PySide::DynamicQMetaObject(className.toAscii(), &QObject::staticMetaObject);
}
    return m_metaObject;
}

int QObjectWrapper::qt_metacall(QMetaObject::Call call, int id, void** args)
{
    int result = QObject::qt_metacall(call, id, args);
    return result < 0 ? result : PySide::SignalManager::qt_metacall(this, call, id, args);
}

QObjectWrapper::~QObjectWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
    delete m_metaObject;
}

// Target ---------------------------------------------------------

static int
SbkQObject_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QObjectWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);
    assert(!sbkSelf->cptr);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0};

    if (!PyArg_UnpackTuple(args, "QObject", 0, 1, &(pyargs[0])))
        return -1;

    if (numArgs == 0) {
        // QObject(QObject*)
        cptr = new QObjectWrapper();
        Shiboken::setParent(pyargs[0], self);
    } else if (Shiboken::Converter<QObject* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QObject* cpp_arg0 = Shiboken::Converter<QObject* >::toCpp(pyargs[0]);
        // QObject(QObject*)
        cptr = new QObjectWrapper(cpp_arg0);
        Shiboken::setParent(pyargs[0], self);
    } else goto SbkQObject_Init_TypeError;

    if (PyErr_Occurred()) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQObject_Init_TypeError;

    sbkSelf->cptr = cptr;
    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf);

    return 1;

    SbkQObject_Init_TypeError:
        const char* overloads[] = {"PySide.QtCore.QObject = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject", overloads);
        return -1;
}

static PyObject*
SbkQObjectFunc_blockSignals(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // blockSignals(bool)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QObject::blockSignals(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQObjectFunc_blockSignals_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_blockSignals_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.blockSignals", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_childEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QChildEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QChildEvent* cpp_arg0 = Shiboken::Converter<QChildEvent* >::toCpp(arg);
        // childEvent(QChildEvent*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::childEvent(cpp_arg0);
    } else goto SbkQObjectFunc_childEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_childEvent_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChildEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.childEvent", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_children(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // children()const
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    const QList<QObject * > & cpp_result = cppSelf->QObject::children();
    py_result = Shiboken::Converter<QList<QObject * >  >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_connect(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "connect", 2, 5, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (Shiboken::Converter<QObject* >::isConvertible(pyargs[1]) && Shiboken::Converter<const char * >::isConvertible(pyargs[2])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QObject* cpp_arg1 = Shiboken::Converter<QObject* >::toCpp(pyargs[1]);
            const char * cpp_arg2 = Shiboken::Converter<const char * >::toCpp(pyargs[2]);
            if (numArgs == 3) {
                // connect(const char*,const QObject*,const char*,Qt::ConnectionType)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                (void)cppSelf; // avoid warnings about unused variables
                // Begin code injection
                
                // connect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectConnect(cppSelf, cpp_arg0, cpp_arg1, cpp_arg2, Qt::AutoConnection));
                
                // End of code injection


            } else if (Shiboken::Converter<Qt::ConnectionType >::isConvertible(pyargs[3])) {
                Qt::ConnectionType cpp_arg3 = Shiboken::Converter<Qt::ConnectionType >::toCpp(pyargs[3]);
                // connect(const char*,const QObject*,const char*,Qt::ConnectionType)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                (void)cppSelf; // avoid warnings about unused variables
                // Begin code injection
                
                // connect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectConnect(cppSelf, cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3));
                
                // End of code injection


            } else goto SbkQObjectFunc_connect_TypeError;
        } else if (PyCallable_Check(pyargs[1])) {
            if (numArgs == 2) {
                // connect(const char*,PyCallable*,Qt::ConnectionType)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                (void)cppSelf; // avoid warnings about unused variables
                // Begin code injection
                
                // connect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectConnectCallback(cppSelf, cpp_arg0, pyargs[2-1], Qt::AutoConnection));
                
                // End of code injection


            } else if (Shiboken::Converter<Qt::ConnectionType >::isConvertible(pyargs[2])) {
                Qt::ConnectionType cpp_arg2 = Shiboken::Converter<Qt::ConnectionType >::toCpp(pyargs[2]);
                // connect(const char*,PyCallable*,Qt::ConnectionType)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                (void)cppSelf; // avoid warnings about unused variables
                // Begin code injection
                
                // connect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectConnectCallback(cppSelf, cpp_arg0, pyargs[2-1], cpp_arg2));
                
                // End of code injection


            } else goto SbkQObjectFunc_connect_TypeError;
        } else goto SbkQObjectFunc_connect_TypeError;
    } else if (Shiboken::Converter<QObject* >::isConvertible(pyargs[0]) && Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QObject* cpp_arg0 = Shiboken::Converter<QObject* >::toCpp(pyargs[0]);
        const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
        if (Shiboken::Converter<const char * >::isConvertible(pyargs[2])) {
            const char * cpp_arg2 = Shiboken::Converter<const char * >::toCpp(pyargs[2]);
            if (numArgs == 3) {
                // connect(const QObject*,const char*,const char*,Qt::ConnectionType)const
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QObject::connect(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            } else if (Shiboken::Converter<Qt::ConnectionType >::isConvertible(pyargs[3])) {
                Qt::ConnectionType cpp_arg3 = Shiboken::Converter<Qt::ConnectionType >::toCpp(pyargs[3]);
                // connect(const QObject*,const char*,const char*,Qt::ConnectionType)const
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QObject::connect(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            } else goto SbkQObjectFunc_connect_TypeError;
        } else if (Shiboken::Converter<QObject* >::isConvertible(pyargs[2]) && Shiboken::Converter<const char * >::isConvertible(pyargs[3])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                return 0;
            QObject* cpp_arg2 = Shiboken::Converter<QObject* >::toCpp(pyargs[2]);
            const char * cpp_arg3 = Shiboken::Converter<const char * >::toCpp(pyargs[3]);
            if (numArgs == 4) {
                // connect(const QObject*,const char*,const QObject*,const char*,Qt::ConnectionType)
                // Begin code injection
                
                // connect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectConnect(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, Qt::AutoConnection));
                
                // End of code injection


            } else if (Shiboken::Converter<Qt::ConnectionType >::isConvertible(pyargs[4])) {
                Qt::ConnectionType cpp_arg4 = Shiboken::Converter<Qt::ConnectionType >::toCpp(pyargs[4]);
                // connect(const QObject*,const char*,const QObject*,const char*,Qt::ConnectionType)
                // Begin code injection
                
                // connect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectConnect(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4));
                
                // End of code injection


            } else goto SbkQObjectFunc_connect_TypeError;
        } else if (PyCallable_Check(pyargs[2])) {
            if (numArgs == 3) {
                // connect(const QObject*,const char*,PyCallable*,Qt::ConnectionType)
                // Begin code injection
                
                // connect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectConnectCallback(cpp_arg0, cpp_arg1, pyargs[3-1], Qt::AutoConnection));
                
                // End of code injection


            } else if (Shiboken::Converter<Qt::ConnectionType >::isConvertible(pyargs[3])) {
                Qt::ConnectionType cpp_arg3 = Shiboken::Converter<Qt::ConnectionType >::toCpp(pyargs[3]);
                // connect(const QObject*,const char*,PyCallable*,Qt::ConnectionType)
                // Begin code injection
                
                // connect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectConnectCallback(cpp_arg0, cpp_arg1, pyargs[3-1], cpp_arg3));
                
                // End of code injection


            } else goto SbkQObjectFunc_connect_TypeError;
        } else goto SbkQObjectFunc_connect_TypeError;
    } else goto SbkQObjectFunc_connect_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_connect_TypeError:
        const char* overloads[] = {"PySide.QtCore.QObject, str, PyCallable, PySide.QtCore.Qt.ConnectionType = Qt.AutoConnection", "PySide.QtCore.QObject, str, PySide.QtCore.QObject, str, PySide.QtCore.Qt.ConnectionType = Qt.AutoConnection", "PySide.QtCore.QObject, str, str, PySide.QtCore.Qt.ConnectionType = Qt.AutoConnection", "str, PyCallable, PySide.QtCore.Qt.ConnectionType = Qt.AutoConnection", "str, PySide.QtCore.QObject, str, PySide.QtCore.Qt.ConnectionType = Qt.AutoConnection", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.connect", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_connectNotify(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // connectNotify(const char*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::connectNotify(cpp_arg0);
    } else goto SbkQObjectFunc_connectNotify_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_connectNotify_TypeError:
        const char* overloads[] = {"str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.connectNotify", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_customEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QEvent* cpp_arg0 = Shiboken::Converter<QEvent* >::toCpp(arg);
        // customEvent(QEvent*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::customEvent(cpp_arg0);
    } else goto SbkQObjectFunc_customEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_customEvent_TypeError:
        const char* overloads[] = {"PySide.QtCore.QEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.customEvent", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_deleteLater(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // deleteLater()
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    cppSelf->QObject::deleteLater();

    // Ownership transferences.
    BindingManager::instance().transferOwnershipToCpp(self);

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQObjectFunc_disconnect(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "disconnect", 0, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 0) {
        // disconnect(const char*,const QObject*,const char*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QObject::disconnect();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // disconnect(const char*,const QObject*,const char*)
            QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QObject::disconnect(cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        } else if (Shiboken::Converter<QObject* >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QObject* cpp_arg1 = Shiboken::Converter<QObject* >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // disconnect(const char*,const QObject*,const char*)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QObject::disconnect(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[2])) {
                const char * cpp_arg2 = Shiboken::Converter<const char * >::toCpp(pyargs[2]);
                // disconnect(const char*,const QObject*,const char*)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QObject::disconnect(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            } else goto SbkQObjectFunc_disconnect_TypeError;
        } else if (numArgs == 2 && PyCallable_Check(pyargs[1])) {
            // disconnect(const char*,PyCallable*)
            QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
            (void)cppSelf; // avoid warnings about unused variables
            // Begin code injection
            
            // disconnect() - disable generation of function call.
            py_result = Shiboken::Converter<bool >::toPython(qobjectDisconnectCallback(cppSelf, cpp_arg0, pyargs[1]));
            
            // End of code injection


        } else goto SbkQObjectFunc_disconnect_TypeError;
    } else if (Shiboken::Converter<QObject* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QObject* cpp_arg0 = Shiboken::Converter<QObject* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // disconnect(const QObject*,const char*)
            QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QObject::disconnect(cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // disconnect(const QObject*,const char*)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                bool cpp_result = cppSelf->QObject::disconnect(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            } else if (numArgs == 4 && Shiboken::Converter<QObject* >::isConvertible(pyargs[2]) && Shiboken::Converter<const char * >::isConvertible(pyargs[3])) {
                if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                    return 0;
                QObject* cpp_arg2 = Shiboken::Converter<QObject* >::toCpp(pyargs[2]);
                const char * cpp_arg3 = Shiboken::Converter<const char * >::toCpp(pyargs[3]);
                // disconnect(const QObject*,const char*,const QObject*,const char*)
                bool cpp_result = QObject::disconnect(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
                py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            } else if (numArgs == 3 && PyCallable_Check(pyargs[2])) {
                // disconnect(const QObject*,const char*,PyCallable*)
                // Begin code injection
                
                // disconnect() - disable generation of function call.
                py_result = Shiboken::Converter<bool >::toPython(qobjectDisconnectCallback(cpp_arg0, cpp_arg1, pyargs[2]));
                
                // End of code injection


            } else goto SbkQObjectFunc_disconnect_TypeError;
        } else goto SbkQObjectFunc_disconnect_TypeError;
    } else goto SbkQObjectFunc_disconnect_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_disconnect_TypeError:
        const char* overloads[] = {"PySide.QtCore.QObject, str, PyCallable", "PySide.QtCore.QObject, str = None", "PySide.QtCore.QObject, str, PySide.QtCore.QObject, str", "str, PyCallable", "str = None, PySide.QtCore.QObject = None, str = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.disconnect", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_disconnectNotify(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // disconnectNotify(const char*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::disconnectNotify(cpp_arg0);
    } else goto SbkQObjectFunc_disconnectNotify_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_disconnectNotify_TypeError:
        const char* overloads[] = {"str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.disconnectNotify", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_dumpObjectInfo(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // dumpObjectInfo()
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    cppSelf->QObject::dumpObjectInfo();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQObjectFunc_dumpObjectTree(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // dumpObjectTree()
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    cppSelf->QObject::dumpObjectTree();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQObjectFunc_dynamicPropertyNames(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // dynamicPropertyNames()const
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    QList<QByteArray > cpp_result = cppSelf->QObject::dynamicPropertyNames();
    py_result = Shiboken::Converter<QList<QByteArray > >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_emit(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    PyObject* nonvarargs = PyTuple_GetSlice(args, 0, 1);
    Shiboken::AutoDecRef auto_nonvarargs(nonvarargs);
    pyargs[1] = PyTuple_GetSlice(args, 1, numArgs);
    Shiboken::AutoDecRef auto_varargs(pyargs[1]);

    if (!PyArg_UnpackTuple(nonvarargs, "emit", 1, 1, &(pyargs[0])))
        return 0;

    if (numArgs >= 1 && Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        // emit(const char*,...)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        (void)cppSelf; // avoid warnings about unused variables
        // Begin code injection
        
        py_result = Shiboken::Converter<bool >::toPython(PySide::SignalManager::instance().emitSignal(cppSelf, cpp_arg0, pyargs[2-1]));
        
        // End of code injection


    } else goto SbkQObjectFunc_emit_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_emit_TypeError:
        const char* overloads[] = {"str, ...", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.emit", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_event(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QEvent* cpp_arg0 = Shiboken::Converter<QEvent* >::toCpp(arg);
        // event(QEvent*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QObject::event(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQObjectFunc_event_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_event_TypeError:
        const char* overloads[] = {"PySide.QtCore.QEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.event", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_eventFilter(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "eventFilter", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QObject* >::isConvertible(pyargs[0]) && Shiboken::Converter<QEvent* >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QObject* cpp_arg0 = Shiboken::Converter<QObject* >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        QEvent* cpp_arg1 = Shiboken::Converter<QEvent* >::toCpp(pyargs[1]);
        // eventFilter(QObject*,QEvent*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QObject::eventFilter(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQObjectFunc_eventFilter_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_eventFilter_TypeError:
        const char* overloads[] = {"PySide.QtCore.QObject, PySide.QtCore.QEvent", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.eventFilter", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_findChild(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "findChild", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyType_Check(pyargs[0]) && Shiboken::Converter<QString >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QString > cpp_arg1_auto_ptr;
        QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QString >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
        // findChild(PyTypeObject*,QString)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        (void)cppSelf; // avoid warnings about unused variables
        // Begin code injection
        
        const bool isEmpty = (*cpp_arg1).isEmpty();
        foreach(QObject* o, cppSelf->children()) {
            if (isEmpty || (*cpp_arg1) == o->objectName()) {
                PyObject* obj = Shiboken::Converter<QObject* >::toPython(o);
                Shiboken::setParent(self, obj);
                if (PyType_IsSubtype(obj->ob_type, (PyTypeObject*)pyargs[1-1]))
                    py_result = obj;
                else
                    Py_DECREF(obj);
            }
        }
        
        // End of code injection


    } else goto SbkQObjectFunc_findChild_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_findChild_TypeError:
        const char* overloads[] = {"PyTypeObject, PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.findChild", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_findChildren(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "findChildren", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyType_Check(pyargs[0]) && Shiboken::Converter<QString >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QString > cpp_arg1_auto_ptr;
        QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QString >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
        // findChildren(PyTypeObject*,QString)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        (void)cppSelf; // avoid warnings about unused variables
        // Begin code injection
        
        py_result = PyList_New(0);
        const bool isEmpty = (*cpp_arg1).isEmpty();
        foreach(QObject* o, cppSelf->children()) {
            if (isEmpty || (*cpp_arg1) == o->objectName()) {
                PyObject* obj = Shiboken::Converter<QObject* >::toPython(o);
                Shiboken::setParent(self, obj);
                if (PyType_IsSubtype(obj->ob_type, (PyTypeObject*)pyargs[1-1]))
                    PyList_Append(py_result, obj);
                Py_DECREF(obj);
            }
        }
        
        // End of code injection


    } else goto SbkQObjectFunc_findChildren_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_findChildren_TypeError:
        const char* overloads[] = {"PyTypeObject, PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.findChildren", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_inherits(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // inherits(const char*)const
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        // Begin code injection
        
        bool retval =  qobjectInheritsInternal(self->ob_type, cpp_arg0) ? true : cppSelf->inherits(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(retval);
        
        // End of code injection


    } else goto SbkQObjectFunc_inherits_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_inherits_TypeError:
        const char* overloads[] = {"str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.inherits", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_installEventFilter(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QObject* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QObject* cpp_arg0 = Shiboken::Converter<QObject* >::toCpp(arg);
        // installEventFilter(QObject*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::installEventFilter(cpp_arg0);
    } else goto SbkQObjectFunc_installEventFilter_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_installEventFilter_TypeError:
        const char* overloads[] = {"PySide.QtCore.QObject", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.installEventFilter", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_isWidgetType(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isWidgetType()const
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QObject::isWidgetType();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_killTimer(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // killTimer(int)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::killTimer(cpp_arg0);
    } else goto SbkQObjectFunc_killTimer_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_killTimer_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.killTimer", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_metaObject(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // metaObject()const
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    // Begin code injection
    
    py_result = Shiboken::Converter<const QMetaObject * >::toPython(cppSelf->metaObject());
    
    // End of code injection


    Shiboken::setParent(self, py_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_moveToThread(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QThread* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QThread* cpp_arg0 = Shiboken::Converter<QThread* >::toCpp(arg);
        // moveToThread(QThread*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::moveToThread(cpp_arg0);
    } else goto SbkQObjectFunc_moveToThread_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_moveToThread_TypeError:
        const char* overloads[] = {"PySide.QtCore.QThread", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.moveToThread", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_objectName(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // objectName()const
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QObject::objectName();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_parent(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // parent()const
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    QObject * cpp_result = cppSelf->QObject::parent();
    py_result = Shiboken::Converter<QObject* >::toPython(cpp_result);
    Shiboken::setParent(py_result, self);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_property(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // property(const char*)const
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        QVariant cpp_result = cppSelf->QObject::property(cpp_arg0);
        py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
    } else goto SbkQObjectFunc_property_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_property_TypeError:
        const char* overloads[] = {"str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.property", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_receivers(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // receivers(const char*)const
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        // Begin code injection
        
        //Avoid return +1 because SignalManager connect to "destroyed()" signal to control object timelife
        int ret = cppSelf->receivers(cpp_arg0);
        if ((strcmp(cpp_arg0, SIGNAL(destroyed())) == 0) && (PySide::SignalManager::instance().hasConnectionWith(cppSelf)))
          ret--;
        py_result = Shiboken::Converter<int >::toPython(ret);
        
        // End of code injection


    } else goto SbkQObjectFunc_receivers_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_receivers_TypeError:
        const char* overloads[] = {"str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.receivers", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_registerUserData(PyObject* self)
{
    PyObject* py_result = 0;

    // registerUserData()
    uint cpp_result = QObject::registerUserData();
    py_result = Shiboken::Converter<unsigned int >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_removeEventFilter(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QObject* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QObject* cpp_arg0 = Shiboken::Converter<QObject* >::toCpp(arg);
        // removeEventFilter(QObject*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::removeEventFilter(cpp_arg0);
    } else goto SbkQObjectFunc_removeEventFilter_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_removeEventFilter_TypeError:
        const char* overloads[] = {"PySide.QtCore.QObject", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.removeEventFilter", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_sender(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // sender()const
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    QObject * cpp_result = cppSelf->QObject::sender();
    py_result = Shiboken::Converter<QObject* >::toPython(cpp_result);

    // Ownership transferences.
    SbkBaseWrapper_setOwnership(py_result, true);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_setObjectName(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QString& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!Shiboken::Converter<QString& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // setObjectName(QString)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::setObjectName(*cpp_arg0);
    } else goto SbkQObjectFunc_setObjectName_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_setObjectName_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.setObjectName", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_setParent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QObject* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QObject* cpp_arg0 = Shiboken::Converter<QObject* >::toCpp(arg);
        // setParent(QObject*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::setParent(cpp_arg0);
        Shiboken::setParent(arg, self);
    } else goto SbkQObjectFunc_setParent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_setParent_TypeError:
        const char* overloads[] = {"PySide.QtCore.QObject", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.setParent", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_setProperty(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setProperty", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<const char * >::isConvertible(pyargs[0]) && Shiboken::Converter<QVariant& >::isConvertible(pyargs[1])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QVariant > cpp_arg1_auto_ptr;
        QVariant* cpp_arg1 = Shiboken::Converter<QVariant* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QVariant& >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QVariant >(cpp_arg1);
        // setProperty(const char*,QVariant)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QObject::setProperty(cpp_arg0, *cpp_arg1);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQObjectFunc_setProperty_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_setProperty_TypeError:
        const char* overloads[] = {"str, PySide.QtCore.QVariant", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.setProperty", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_signalsBlocked(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // signalsBlocked()const
    QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QObject::signalsBlocked();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQObjectFunc_startTimer(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // startTimer(int)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        int cpp_result = cppSelf->QObject::startTimer(cpp_arg0);
        py_result = Shiboken::Converter<int >::toPython(cpp_result);
    } else goto SbkQObjectFunc_startTimer_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_startTimer_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.startTimer", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_timerEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QTimerEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QTimerEvent* cpp_arg0 = Shiboken::Converter<QTimerEvent* >::toCpp(arg);
        // timerEvent(QTimerEvent*)
        QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
        cppSelf->QObject::timerEvent(cpp_arg0);
    } else goto SbkQObjectFunc_timerEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQObjectFunc_timerEvent_TypeError:
        const char* overloads[] = {"PySide.QtCore.QTimerEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QObject.timerEvent", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_tr(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "tr", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // tr(const char*,const char*,int)
            QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
            (void)cppSelf; // avoid warnings about unused variables
            // Begin code injection
            
            QString result;
            if (QCoreApplication::instance()) {
                PyObject *klass = PyObject_GetAttrString(self, "__class__");
                PyObject *cname = PyObject_GetAttrString(klass, "__name__");
                result = QString(QCoreApplication::instance()->translate(PyString_AsString(cname), cpp_arg0, 0, QCoreApplication::CodecForTr, -1));
            
                Py_DECREF(klass);
                Py_DECREF(cname);
            } else {
                result = QString(QString::fromLatin1(cpp_arg0));
            }
            py_result = Shiboken::Converter<QString >::toPython(result);
            
            // End of code injection


        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // tr(const char*,const char*,int)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                (void)cppSelf; // avoid warnings about unused variables
                // Begin code injection
                
                QString result;
                if (QCoreApplication::instance()) {
                    PyObject *klass = PyObject_GetAttrString(self, "__class__");
                    PyObject *cname = PyObject_GetAttrString(klass, "__name__");
                    result = QString(QCoreApplication::instance()->translate(PyString_AsString(cname), cpp_arg0, cpp_arg1, QCoreApplication::CodecForTr, -1));
                
                    Py_DECREF(klass);
                    Py_DECREF(cname);
                } else {
                    result = QString(QString::fromLatin1(cpp_arg0));
                }
                py_result = Shiboken::Converter<QString >::toPython(result);
                
                // End of code injection


            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                // tr(const char*,const char*,int)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                (void)cppSelf; // avoid warnings about unused variables
                // Begin code injection
                
                QString result;
                if (QCoreApplication::instance()) {
                    PyObject *klass = PyObject_GetAttrString(self, "__class__");
                    PyObject *cname = PyObject_GetAttrString(klass, "__name__");
                    result = QString(QCoreApplication::instance()->translate(PyString_AsString(cname), cpp_arg0, cpp_arg1, QCoreApplication::CodecForTr, cpp_arg2));
                
                    Py_DECREF(klass);
                    Py_DECREF(cname);
                } else {
                    result = QString(QString::fromLatin1(cpp_arg0));
                }
                py_result = Shiboken::Converter<QString >::toPython(result);
                
                // End of code injection


            } else goto SbkQObjectFunc_tr_TypeError;
        } else goto SbkQObjectFunc_tr_TypeError;
    } else goto SbkQObjectFunc_tr_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_tr_TypeError:
        const char* overloads[] = {"str, str = None, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.tr", overloads);
        return 0;
}

static PyObject*
SbkQObjectFunc_trUtf8(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "trUtf8", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // trUtf8(const char*,const char*,int)
            QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
            (void)cppSelf; // avoid warnings about unused variables
            // Begin code injection
            
            QString result;
            if (QCoreApplication::instance()) {
                PyObject *klass = PyObject_GetAttrString(self, "__class__");
                PyObject *cname = PyObject_GetAttrString(klass, "__name__");
                result = QString(QCoreApplication::instance()->translate(PyString_AsString(cname), cpp_arg0, 0, QCoreApplication::UnicodeUTF8, -1));
            
                Py_DECREF(klass);
                Py_DECREF(cname);
            } else {
                result = QString(QString::fromLatin1(cpp_arg0));
            }
            py_result = Shiboken::Converter<QString >::toPython(result);
            
            // End of code injection


        } else if (Shiboken::Converter<const char * >::isConvertible(pyargs[1])) {
            const char * cpp_arg1 = Shiboken::Converter<const char * >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // trUtf8(const char*,const char*,int)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                (void)cppSelf; // avoid warnings about unused variables
                // Begin code injection
                
                QString result;
                if (QCoreApplication::instance()) {
                    PyObject *klass = PyObject_GetAttrString(self, "__class__");
                    PyObject *cname = PyObject_GetAttrString(klass, "__name__");
                    result = QString(QCoreApplication::instance()->translate(PyString_AsString(cname), cpp_arg0, cpp_arg1, QCoreApplication::UnicodeUTF8, -1));
                
                    Py_DECREF(klass);
                    Py_DECREF(cname);
                } else {
                    result = QString(QString::fromLatin1(cpp_arg0));
                }
                py_result = Shiboken::Converter<QString >::toPython(result);
                
                // End of code injection


            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                // trUtf8(const char*,const char*,int)
                QObject* cppSelf = Shiboken::Converter<QObject* >::toCpp((PyObject*)self);
                (void)cppSelf; // avoid warnings about unused variables
                // Begin code injection
                
                QString result;
                if (QCoreApplication::instance()) {
                    PyObject *klass = PyObject_GetAttrString(self, "__class__");
                    PyObject *cname = PyObject_GetAttrString(klass, "__name__");
                    result = QString(QCoreApplication::instance()->translate(PyString_AsString(cname), cpp_arg0, cpp_arg1, QCoreApplication::UnicodeUTF8, cpp_arg2));
                
                    Py_DECREF(klass);
                    Py_DECREF(cname);
                } else {
                    result = QString(QString::fromLatin1(cpp_arg0));
                }
                py_result = Shiboken::Converter<QString >::toPython(result);
                
                // End of code injection


            } else goto SbkQObjectFunc_trUtf8_TypeError;
        } else goto SbkQObjectFunc_trUtf8_TypeError;
    } else goto SbkQObjectFunc_trUtf8_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQObjectFunc_trUtf8_TypeError:
        const char* overloads[] = {"str, str = None, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QObject.trUtf8", overloads);
        return 0;
}

static PyMethodDef SbkQObjectMethod_connect = {
    "connect", (PyCFunction)SbkQObjectFunc_connect, METH_VARARGS|METH_STATIC
};

static PyMethodDef SbkQObjectMethod_disconnect = {
    "disconnect", (PyCFunction)SbkQObjectFunc_disconnect, METH_VARARGS|METH_STATIC
};

static PyMethodDef SbkQObject_methods[] = {
    {"blockSignals", (PyCFunction)SbkQObjectFunc_blockSignals, METH_O},
    {"childEvent", (PyCFunction)SbkQObjectFunc_childEvent, METH_O},
    {"children", (PyCFunction)SbkQObjectFunc_children, METH_NOARGS},
    SbkQObjectMethod_connect,
    {"connectNotify", (PyCFunction)SbkQObjectFunc_connectNotify, METH_O},
    {"customEvent", (PyCFunction)SbkQObjectFunc_customEvent, METH_O},
    {"deleteLater", (PyCFunction)SbkQObjectFunc_deleteLater, METH_NOARGS},
    SbkQObjectMethod_disconnect,
    {"disconnectNotify", (PyCFunction)SbkQObjectFunc_disconnectNotify, METH_O},
    {"dumpObjectInfo", (PyCFunction)SbkQObjectFunc_dumpObjectInfo, METH_NOARGS},
    {"dumpObjectTree", (PyCFunction)SbkQObjectFunc_dumpObjectTree, METH_NOARGS},
    {"dynamicPropertyNames", (PyCFunction)SbkQObjectFunc_dynamicPropertyNames, METH_NOARGS},
    {"emit", (PyCFunction)SbkQObjectFunc_emit, METH_VARARGS},
    {"event", (PyCFunction)SbkQObjectFunc_event, METH_O},
    {"eventFilter", (PyCFunction)SbkQObjectFunc_eventFilter, METH_VARARGS},
    {"findChild", (PyCFunction)SbkQObjectFunc_findChild, METH_VARARGS},
    {"findChildren", (PyCFunction)SbkQObjectFunc_findChildren, METH_VARARGS},
    {"inherits", (PyCFunction)SbkQObjectFunc_inherits, METH_O},
    {"installEventFilter", (PyCFunction)SbkQObjectFunc_installEventFilter, METH_O},
    {"isWidgetType", (PyCFunction)SbkQObjectFunc_isWidgetType, METH_NOARGS},
    {"killTimer", (PyCFunction)SbkQObjectFunc_killTimer, METH_O},
    {"metaObject", (PyCFunction)SbkQObjectFunc_metaObject, METH_NOARGS},
    {"moveToThread", (PyCFunction)SbkQObjectFunc_moveToThread, METH_O},
    {"objectName", (PyCFunction)SbkQObjectFunc_objectName, METH_NOARGS},
    {"parent", (PyCFunction)SbkQObjectFunc_parent, METH_NOARGS},
    {"property", (PyCFunction)SbkQObjectFunc_property, METH_O},
    {"receivers", (PyCFunction)SbkQObjectFunc_receivers, METH_O},
    {"registerUserData", (PyCFunction)SbkQObjectFunc_registerUserData, METH_NOARGS|METH_STATIC},
    {"removeEventFilter", (PyCFunction)SbkQObjectFunc_removeEventFilter, METH_O},
    {"sender", (PyCFunction)SbkQObjectFunc_sender, METH_NOARGS},
    {"setObjectName", (PyCFunction)SbkQObjectFunc_setObjectName, METH_O},
    {"setParent", (PyCFunction)SbkQObjectFunc_setParent, METH_O},
    {"setProperty", (PyCFunction)SbkQObjectFunc_setProperty, METH_VARARGS},
    {"signalsBlocked", (PyCFunction)SbkQObjectFunc_signalsBlocked, METH_NOARGS},
    {"startTimer", (PyCFunction)SbkQObjectFunc_startTimer, METH_O},
    {"timerEvent", (PyCFunction)SbkQObjectFunc_timerEvent, METH_O},
    {"tr", (PyCFunction)SbkQObjectFunc_tr, METH_VARARGS},
    {"trUtf8", (PyCFunction)SbkQObjectFunc_trUtf8, METH_VARARGS},
    {0} // Sentinel
};

static PyObject* SbkQObject_getattro(PyObject* self, PyObject* name)
{
    if (self) {
        if (SbkBaseWrapper_instanceDict(self)) {
            PyObject* meth = PyDict_GetItem(SbkBaseWrapper_instanceDict(self), name);
            if (meth) {
                Py_INCREF(meth);
                return meth;
            }
        }
        const char* cname = PyString_AS_STRING(name);
        if (strcmp(cname, "connect") == 0)
            return PyCFunction_NewEx(&SbkQObjectMethod_connect, self, 0);
        if (strcmp(cname, "disconnect") == 0)
            return PyCFunction_NewEx(&SbkQObjectMethod_disconnect, self, 0);
    }
    return PyObject_GenericGetAttr(self, name);
}

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQObject_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtCore.QObject",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::SbkBaseWrapper_Dealloc< QObjectWrapper >,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         SbkQObject_getattro,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQObject_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQObject_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0
};

static SbkBaseWrapperType* SbkQObject_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQObject_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QObject*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}


PyAPI_FUNC(void)
init_QObject(PyObject* module)
{
    SbkPySide_QtCoreTypes[SBK_QOBJECT_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQObject_Type);


    // Fill type discovery information
    SbkQObject_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQObject_Type.type_discovery->addTypeDiscoveryFunction(&SbkQObject_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQObject_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQObject_Type));
    PyModule_AddObject(module, "QObject",
        ((PyObject*)&SbkQObject_Type));


    Shiboken::TypeResolver::createObjectTypeResolver<QObject >("QObject*");
    Shiboken::TypeResolver::createObjectTypeResolver<QObject >(typeid(QObject).name());
}


} // extern "C"

namespace Shiboken
{
// Copy C++ object implementation
// Converter implementations
} // namespace Shiboken


