/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "pyside.hpp"
#include "test_qptr.h"

#include <QtTest/QTest>
#include <QDebug>
#include "boost_headers.hpp"

using namespace boost::python;

using namespace PySide;

void test_qptr::initTestCase()
{
    Py_Initialize();
}

void test_qptr::test_remove_parent()
{
    QObject* raw_parent = new QObject;
    raw_parent->setObjectName("pai");
    qptr<QObject> parent(raw_parent);
    QObject* raw_child = new QObject(raw_parent);
    raw_child->setObjectName("filho");
    qptr<QObject> child(raw_child);
    QCOMPARE(parent.refcount(), 1);
    QCOMPARE(child.refcount(), 1);

    parent.add_child(child);
    QCOMPARE(parent.refcount(), 1);
    QCOMPARE(child.refcount(), 2);
    QVERIFY(!parent.is_null());
    QVERIFY(!child.is_null());
    parent.release(); // remove the parent
    QCOMPARE(parent.refcount(), 0);
    QVERIFY(parent.is_null());
    QVERIFY(child.is_null());
    QVERIFY(!child.has_parent());
    try {
        child->objectName();
        QFAIL("boost::python::error_already_set not throw! AAAHHH!!!");
    } catch (boost::python::error_already_set&) {
    }
    child.release();
}

void test_qptr::test_add_same_child_twice()
{
    qptr<QObject> parent(new QObject);
    qptr<QObject> child(new QObject(parent.get()));
    qptr<QObject> child2(child);
    parent.add_child(child);
    QCOMPARE(parent.refcount(), 1);
    QCOMPARE(child.refcount(), 3);
    parent.add_child(child2);
    QCOMPARE(parent.refcount(), 1);
    QCOMPARE(child.refcount(), 3);
}

void test_qptr::test_remove_child()
{
    QObject* raw_parent = new QObject;
    raw_parent->setObjectName("pai");
    qptr<QObject> parent(raw_parent);
    QObject* raw_child = new QObject(raw_parent);
    raw_child->setObjectName("filho");
    qptr<QObject> child(raw_child);
    QCOMPARE(parent.refcount(), 1);
    QCOMPARE(child.refcount(), 1);

    parent.add_child(child);
    QCOMPARE(parent.refcount(), 1);
    QCOMPARE(child.refcount(), 2);
    QVERIFY(!parent.is_null());
    QVERIFY(!child.is_null());
    parent.remove_child(child); // remove the parent
    QCOMPARE(parent.refcount(), 1);
    QCOMPARE(child.refcount(), 1);
    QVERIFY(!parent.is_null());
    QVERIFY(!child.is_null());
    QVERIFY(!child.has_parent());
}

void test_qptr::test_qptr_operations()
{
    PySide::qptr<QObject> a(new QObject);
    PySide::qptr<QObject> b(new QObject);
    a = b;
    {
        qptr<QObject> c(a);
        QCOMPARE(c.refcount(), 3);
        qptr_base d = c;
        QCOMPARE(d.refcount(), 4);
        QCOMPARE(c.refcount(), 4);
        QCOMPARE(a.refcount(), 4);
    }
    QCOMPARE(a.refcount(), 2);
    QCOMPARE(b.refcount(), 2);
}


QTEST_MAIN( test_qptr )
#include "test_qptr.moc"
