/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <QtCore/QObject>
#include <QtCore/QMetaObject>
#include <QtCore/QMutableHashIterator>
#include <QtCore/QDebug>
#include <QtCore/QEvent>

#include <cstdio>

#include "signal_manager.hpp"
#include "trigger.hpp"
#include "pyqt_signal_slot.hpp"
#include "type_manager.hpp"
#include "parent_policy.hpp"
#include <QThread>

using namespace boost;

namespace PySide
{

struct PYSIDE_LOCAL signal_manager_data
{
    typedef QHash<trigger*, QList<int> > trigger_slot_map;
    typedef QHash<QByteArray, trigger_slot_map> signal_trigger_map;
    typedef QHash<const QObject*, signal_trigger_map> obj_signal_map;

    QHash<const QObject*, trigger*>  m_connection_map;

    // support for python signals (dynamic signals)
    // source => (signal => (trigger => [id, ...]))
    obj_signal_map m_dynamic_signals;

    // support for QObject::sender method
    QMap<QThread*, QObject*> m_senders;
};

static void
fireConnectNotify(QObject* obj, const QByteArray& signal)
{
    python::object py_obj(PySide::ptr(obj));

    if (py_obj.ptr() != Py_None) {
        // Copy the qbytearray to avoid boost::python crash in some weird
        // use cases when converting the const ref to a python object.
        QByteArray signal_cpy(signal);
        python::call_method<void>(py_obj.ptr(), "connectNotify", signal_cpy);
    }
}

static void
fireDisconnectNotify(QObject* obj, const QByteArray& signal)
{
    python::object py_obj(PySide::ptr(obj));

    if (py_obj.ptr() != Py_None) {
        python::call_method<void>(py_obj.ptr(), "disconnectNotify", signal);
    }
}

signal_manager::signal_manager()
{
    m_data = new signal_manager_data();
}

signal_manager::~signal_manager()
{
    delete m_data;
}

signal_manager&
signal_manager::instance()
{
    static signal_manager instance;
    return instance;
}

bool
signal_manager::connect(QObject* src,
                        const pyqt_signal& signal,
                        python::object& callable)
{
    if (PyCallable_Check(callable.ptr())) {
        trigger *t = find_trigger(src);
        bool retval = t->connect(src, signal, callable,
                                 callable_arg_count(callable));
        if (retval)
            fireConnectNotify(src, signal.signature());
        return retval;
    } else {
        PyErr_SetString(PyExc_TypeError, "Callable object expected!");
        throw python::error_already_set();
    }
}

bool
signal_manager::connect(QObject* source,
                        const pyqt_signal& signal,
                        QObject* receiver,
                        const pyqt_slot& slot)
{
    using namespace boost::python;

    const QByteArray& signal_signature(signal.signature());
    const QByteArray& slot_signature(slot.signature());
    if (!QMetaObject::checkConnectArgs(signal_signature, slot_signature))
        return false;
    int signal_index = source->metaObject()->indexOfSignal(signal_signature);
    int slot_index = receiver->metaObject()->indexOfSlot(slot_signature);

    bool retval = false;
    if (signal_index != -1 && slot_index != -1) {
        // C++ -> C++ connection
        retval = QMetaObject::connect(source, signal_index,
                                      receiver, slot_index);
    } else {
        // We have a python slot or signal
        trigger *t = find_trigger(source);
        retval = t->connect(source, signal, receiver, slot);
    }
    if (retval)
        fireConnectNotify(source, signal_signature);
    return retval;
}

bool
signal_manager::disconnect(QObject* source,
                           const pyqt_signal& signal,
                           boost::python::object& callable)
{
    trigger* t = get_trigger(source);
    if (!t) {
        return false;
    }

    int slot_index = -1;
    bool result = t->disconnect(source, signal, callable, & slot_index);
    if (slot_index != -1)
        m_data->m_dynamic_signals[source][signal.signature()][t].removeOne(slot_index);

    if (result)
        fireDisconnectNotify(source, signal.signature());
    return result;
}

bool
signal_manager::disconnect(QObject* source,
                           const pyqt_signal& signal,
                           QObject* receiver,
                           const pyqt_slot& slot)
{
    bool result = false;
    int slot_index = -1;
    const QByteArray& signal_signature = signal.signature();
    bool is_python_signal =
            source->metaObject()->indexOfSignal(signal_signature) == -1;

    trigger *t = get_trigger(source);
    if (t) {
        result = t->disconnect(source, signal, receiver, slot, &slot_index);
    } else {
        const QByteArray& slot_signature = slot.signature();

        if (!is_python_signal &&
            receiver->metaObject()->indexOfSlot(slot_signature) != -1) {
            #if QSLOT_CODE != 1 || QSIGNAL_CODE != 2
                #error QSLOT_CODE and/or QSIGNAL_CODE changed! \
                       change the hardcoded stuff to the correct value!
            #endif
            result = QObject::disconnect(source, signal.toQtStyleSignature(),
                                           receiver, slot.toQtStyleSignature());
        }
    }

    if (t && is_python_signal && slot_index != -1) {
        // remove the entry in m_dynamic_signals collection
        m_data->m_dynamic_signals[source][signal_signature][t].removeOne(slot_index);
    }

    if (result)
        fireDisconnectNotify(source, signal_signature);
    return result;
}

trigger*
signal_manager::find_trigger(QObject* src)
{
    trigger* t = m_data->m_connection_map.value(src);
    if (!t) {
        t = new trigger(src);
        m_data->m_connection_map[src] = t;
        QObject::connect(src, SIGNAL(destroyed()), t, SLOT(deleteLater()));
    }
    return t;
}

trigger*
signal_manager::get_trigger(const QObject *src) const
{
    return m_data->m_connection_map.value(src);
}

void
signal_manager::register_dynamic_signal(QObject *src,
                                        const pyqt_signal& signal,
                                        trigger* trigger,
                                        int slot_id)
{
    m_data->m_dynamic_signals[src][signal.signature()][trigger] << slot_id;
}

void
signal_manager::unregister_qobj(QObject* obj)
{
    trigger* t = get_trigger(obj);
    if (!t)
        return;
    // remove the trigger when its used as a slot in a dynamic signal.
    // this is ugly, stupid and this is slowwwww, O(n^3) where n is the number
    // of dynamic signals declared.
    signal_manager_data::obj_signal_map::iterator obj_signal_it =
            m_data->m_dynamic_signals.begin();
    for (; obj_signal_it != m_data->m_dynamic_signals.end(); ++obj_signal_it) {
        signal_manager_data::signal_trigger_map& signal_trigger =
                *obj_signal_it;
        signal_manager_data::signal_trigger_map::iterator signal_trigger_it =
                signal_trigger.begin();
        for (; signal_trigger_it != signal_trigger.end(); ++signal_trigger_it) {
            QMutableHashIterator<trigger*, QList<int> > it(*signal_trigger_it);
            while (it.hasNext()) {
                it.next();
                if (it.key() == t) {
                    it.remove(); // finally!!!!
                }
            }
        }
    }

    m_data->m_connection_map.remove(obj);
    m_data->m_dynamic_signals.remove(obj);
}

void
signal_manager::emit_(QObject *src,
                      const pyqt_signal& signal,
                      const python::object& args)
{
    QStringList arg_types;
    int args_given;
    int args_required;

    signal.parse_signature(&arg_types);

    args_given = python::len(args);
    args_required = arg_types.size();

    //check if a sufficient number of arguments was received
    if (args_given < args_required) {
        PyErr_SetString(PyExc_TypeError,
                        "Insufficient number of arguments for this signal.");
        throw boost::python::error_already_set();
    }

    //search native signal
    int signal_index = src->metaObject()->indexOfSignal(signal.signature());

    // a python signal!
    if (signal_index == -1) {
        emit_dynamic_signal(src, signal, args);
    } else { // a C++ signal
        QList<QGenericArgument> q_args;
        QStringList args_types;

        parse_objects_to_qarguments(arg_types, args, q_args);
        void* signal_args[args_given+1];

        //return value
        signal_args[0] = 0;

        for (int i=0; i < args_given; i++)
            signal_args[i+1] = q_args[i].data();
        QMetaObject::activate(src, signal_index, signal_args);
    }
}

void
signal_manager::emit_dynamic_signal(QObject* source,
                                    const pyqt_signal& signal,
                                    const boost::python::object& args)
{
    typedef QHash<trigger*, QList<int> > trigger_map;
    trigger_map& hash = m_data->m_dynamic_signals[source][signal.signature()];
    trigger_map::iterator it = hash.begin();

    for (; it != hash.end(); ++it) {
        it.key()->fire(*it, args);
    }
}

void
signal_manager::parse_objects_to_qarguments(const QStringList &args_type,
                                            const python::object &argv,
                                            QList<QGenericArgument> &q_args)
{
    q_args.clear();

    if (args_type.size() > 10) {
        std::fprintf(stderr,
                     "More then 10 arguments not supported in signal/slot\n");
        return;
    }

    for (int i=0; i < 10; i++) {
        q_args << QGenericArgument();
    }

    for (int i=0; i < args_type.size(); i++) {
        q_args[i] = type_manager::instance().to_cpp(args_type[i], argv[i]);
    }
}

int
signal_manager::dynamicsignal_receivers(const QObject *sender,
                                        const pyqt_signal &signal,
                                        int offset) const
{
    int total = offset;
    if (signal.signature() == "destroyed()" && get_trigger(sender)) {
        total--;
    } else if (!total) { // maybe this is a dynamic signal;

        foreach(QList<int> list,
                m_data->m_dynamic_signals.value(sender)
                        .value(signal.signature())) {
            total += list.count();
        }
    }
    return total;
}

QObject*
signal_manager::register_dynamic_slot(QObject* src,
                                      const python::object& callback,
                                      QString& slot_name)
{
    trigger* t = find_trigger(src);
    slot_name = t->register_dynamic_slot(callback).toQtStyleSignature();
    return t;
}

QList< QByteArray >
signal_manager::signals_of(const QObject* obj) {
    signal_manager_data::obj_signal_map::const_iterator it = m_data->m_dynamic_signals.find(obj);
    return it != m_data->m_dynamic_signals.end() ? it->keys() : QList<QByteArray>();
}

const QMetaObject*
signal_manager::get_dynamic_metaobject(QObject* obj) {
    return find_trigger(obj)->metaObject();
}


void
signal_manager::register_sender(QObject* sender)
{
    m_data->m_senders.insert(QThread::currentThread(), sender);
}

void
signal_manager::unregister_sender()
{
    m_data->m_senders.remove(QThread::currentThread());
}

QObject*
signal_manager::sender()
{
    return m_data->m_senders.value(QThread::currentThread());
}

} // namespace PySide

