/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __ABSTRACT_QOBJECT_CONNECTION_HPP__
#define __ABSTRACT_QOBJECT_CONNECTION_HPP__

#include <QStringList>

#include <pyside_global.hpp>
#include <pyqt_signal_slot.hpp>
#include <boost_headers.hpp>

class QObject;

namespace PySide
{

/**
*   Store some information about a signal/slot/callable connection
*/
class PYSIDE_LOCAL abstract_qobject_connection
{
public:
    abstract_qobject_connection(QObject* source);
    virtual ~abstract_qobject_connection();

    virtual bool equals(const boost::python::api::object& callback) const = 0;

    void set_signal_index(int index) { m_signal_index = index; }
    void set_slot_index(int index) { m_slot_index = index; }

    int signal_index() const { return m_signal_index; }
    int slot_index() const { return m_slot_index; }

    void set_signal_args(const QStringList& args) { m_signal_args = args; }
    QObject* source() const { return m_source; }

    virtual void call_slot(const boost::python::list& slot_args) const = 0;
    void call_slot(void** slot_args) const;

private:
    QObject* m_source;
    QStringList m_signal_args;
    int m_signal_index;
    int m_slot_index;

    // disable copy
    abstract_qobject_connection(const abstract_qobject_connection&);
    abstract_qobject_connection& operator=(const abstract_qobject_connection&);
};

class PYSIDE_LOCAL signal_slot_connection : public abstract_qobject_connection
{
public:
    signal_slot_connection(QObject* source,
                           const boost::python::object& callback,
                           const QStringList& signal_args,
                           int num_slot_args);
    void call_slot(const boost::python::list& slot_args) const;
    bool equals(const boost::python::api::object& callback) const;

private:
    PyObject* m_receiver;
    boost::python::object m_function;
    int m_num_slot_args;
};

class PYSIDE_LOCAL signal_signal_connection : public abstract_qobject_connection
{
public:
    signal_signal_connection(QObject* source,
                             QObject* receiver,
                             const pyqt_signal& signal);
    void call_slot(const boost::python::list& slot_args) const;
    bool equals(const boost::python::api::object& callback) const;
private:
    QObject* m_receiver;
    const pyqt_signal m_signal;
};

} // namespace PySide

#endif

