/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <QtCore/QObject>
#include <QtCore/QTimer>
#include <QtCore/QString>
#include <QtCore/QRegExp>
#include <QtCore/QPointer>
#include <QtCore/QList>
#include <QtCore/QMutableListIterator>
#include <QtCore/QtDebug>
#include <QtCore/QVariant>
#include <QtCore/QThread>
#include <QtCore/QThreadPool>
#include <QtCore/QMetaMethod>
#include <QtCore/QMutex>
#include <QtCore/QStack>

#include <algorithm>

#include "boost_headers.hpp"
#include "signal_manager.hpp"
#include "pyqt_signal_slot.hpp"
#include "type_manager.hpp"
#include "trigger.hpp"
#include "abstract_qobject_connection.hpp"
#include "thread_support.hpp"
#include "parent_policy.hpp"

using namespace boost;

namespace PySide
{

struct PYSIDE_LOCAL trigger_method
{
    python::list args;
    const abstract_qobject_connection *ci;
};

struct PYSIDE_LOCAL trigger_data
{

    // This is necessary because QObject not support
    // create child in diferent thread of parent
    QObject *m_parent;

    // slot_index => connection_info, used by qt_metacall
    // to recover the connection_info
    QHash<int, abstract_qobject_connection*> m_connections;

    // dynamic slot list
    QList<QByteArray> m_slot_list;

    // source => signal => [slot_index, ...], used by disconnect methods
    typedef QHash<const QObject*, QHash<QByteArray, QList<int> > > signal_map;
    signal_map m_signal_map;

    // dynamic meta object stuff
    mutable QMetaObject* m_metaobject;
    mutable uint* m_metadata_indexes;
    mutable char* m_metadata;
    QByteArray m_className;

    int m_next_connection_id;

    //used to store calls of methods in thread-safe mode
    QMutex m_mutex;
    int m_current_timer;
    QStack<trigger_method> m_idle_methdos;

    trigger_data(QObject *parent)
        : m_parent(parent),
          m_metaobject(0),
          m_metadata_indexes(0),
          m_metadata(0),
          m_next_connection_id(parent->metaObject()->methodCount()-1),
          m_current_timer(0)
    {}
};


trigger::trigger(QObject *parent)
{
    m_data = new trigger_data(parent);
    qptr<QObject> ptr(parent);

    //Will remove ownership if this pointer does not have a python object
    if (ptr.get_pyobject() == 0)
        ptr.release_ownership();

    if (ptr.is_wrapper()) {
        python::object obj(PySide::ptr(parent));
        m_data->m_className = QByteArray(obj.ptr()->ob_type->tp_name);
    } else {
        m_data->m_className = QByteArray(parent->metaObject()->className());
    }
}

trigger::~trigger()
{
    signal_manager::instance().unregister_qobj(m_data->m_parent);
    invalidate_metaobject();
    qDeleteAll(m_data->m_connections);
    delete m_data;
}

int
trigger::add_connection_info(abstract_qobject_connection* ci,
                             const pyqt_signal& signal)
{
    m_data->m_next_connection_id++;
    m_data->m_connections[m_data->m_next_connection_id] = ci;

    if (!signal.is_null())
        m_data->m_signal_map[ci->source()][signal.signature()]
                << m_data->m_next_connection_id;

    return m_data->m_next_connection_id;
}

pyqt_slot
trigger::register_dynamic_slot(const python::object &callback)
{
    QByteArray slot_name;
    QTextStream slot_name_stream(&slot_name);
    QStringList args;

    QString funcName = python::extract<QString>(python::str(callback));
    funcName.replace(QRegExp("[ \\.<>]"), "");
    slot_name_stream << funcName << '(';
    int num_args = callable_arg_count(callback);
    for(int i=0; i < num_args; i++)
    {
        if (i>0)
            slot_name_stream << ',';
        slot_name_stream << "PyObject*";
        args << "PyObject*";
    }
    slot_name_stream << ')';
    slot_name_stream.flush();

    if (!m_data->m_slot_list.contains(slot_name)) {
        abstract_qobject_connection* ci =
                new signal_slot_connection(this, callback, args, num_args);
        int slot_index = add_connection_info(ci);
        ci->set_slot_index(slot_index);

        m_data->m_slot_list << slot_name;
        invalidate_metaobject();
    }

    return pyqt_slot(slot_name);
}

bool
trigger::connect(QObject* src,
                 const pyqt_signal& signal,
                 const python::object &callback,
                 int num_slot_args,
                 Qt::ConnectionType type)
{
    QStringList signal_args;
    signal.parse_signature(&signal_args);
    signal_args = signal_args.mid(0, num_slot_args);

    abstract_qobject_connection* ci =
            new signal_slot_connection(src, callback,
                                       signal_args, num_slot_args);
    return connect(src, signal, ci, type);
}

bool
trigger::connect(QObject* src,
                 const pyqt_signal& signal,
                 QObject* receiver,
                 const pyqt_slot& slot,
                 Qt::ConnectionType type)
{
    python::object py_receiver(PySide::ptr(receiver));
    QStringList slot_args;
    slot.parse_signature(&slot_args);
    QByteArray function_name = slot.function_name();
    const char* attr_name = function_name.constData();
    if (slot.is_slot()) {
        if (PyObject_HasAttrString(py_receiver.ptr(), attr_name)) {
            return connect(src, signal,
                           py_receiver.attr(attr_name),
                           slot_args.count()+1, type);
        }
    } else {
        // conection to a signal
        abstract_qobject_connection* ci =
                new signal_signal_connection(src, receiver,
                                             static_cast<const pyqt_signal&>(slot));
        return connect(src, signal, ci, type);
    }
    return false;
}

bool
trigger::connect(QObject* src, const pyqt_signal& signal,
                 abstract_qobject_connection* ci, Qt::ConnectionType type)
{
    int slot_index = add_connection_info(ci, signal);
    int signal_index = src->metaObject()->indexOfSignal(signal.signature());
    ci->set_signal_index(signal_index);
    ci->set_slot_index(slot_index);
    ci->set_connection_type(type);

    if (signal_index >= 0) {
        return QMetaObject::connect(src, signal_index, this, slot_index, type);
    } else { // dynamic signals!
        signal_manager::instance().register_dynamic_signal(src, signal, this, slot_index);
    }
    return true;

}

bool
trigger::disconnect(const QObject* src,
                    const pyqt_signal& signal,
                    const python::object &callback,
                    int* slot_index_found)
{
    int old_size = m_data->m_connections.count();
    QList<int>& slot_indexes = m_data->m_signal_map[src][signal.signature()];
    foreach(int slot_index, slot_indexes) {
        abstract_qobject_connection* ci = m_data->m_connections.value(slot_index);
        if (ci && ci->equals(callback)) {
            if (slot_index_found)
                *slot_index_found = slot_index;
            m_data->m_connections.remove(slot_index); // remove connection_infos
            delete ci;
        }
    }
    if (slot_index_found) {
        slot_indexes.removeOne(*slot_index_found);
        if (!slot_indexes.count())
            m_data->m_signal_map[src].remove(signal.signature());
    }
    return old_size > m_data->m_connections.count();
}

bool
trigger::disconnect(const QObject* src,
                    const pyqt_signal& signal,
                    const QObject* receiver,
                    const pyqt_slot& slot,
                    int* slot_index_found)
{
    python::object py_receiver(PySide::ptr(receiver));
    return disconnect(src, signal,
                      py_receiver.attr(slot.function_name().constData()),
                      slot_index_found);
}

int
trigger::qt_metacall(QMetaObject::Call call, int id, void **args)
{
    // deleteLater slot
    if (id == 2) {
        signal_manager::instance().unregister_qobj(m_data->m_parent);
    }

    if (m_data->m_connections.contains(id)) {
        thread_locker lock;
        py_fire(m_data->m_connections[id], args);
        return -1;
    }

    id = QObject::qt_metacall(call, id, args);
    if (id == -1 || call != QMetaObject::InvokeMetaMethod) {
        return id;
    }

    return id;
}

void trigger::call_method(const abstract_qobject_connection *ci, const python::object &args)
try {
    set_sender setter(ci->source());
    ci->call_slot(python::list(args));
} catch (python::error_already_set&) {
    PyErr_Print();
}

void trigger::idle_call_method(const abstract_qobject_connection *ci, const python::object &args)
{
    trigger_method m;
    m.args = python::list(args);
    m.ci = ci;

    m_data->m_idle_methdos.push(m);

    if (!m_data->m_current_timer) {
        m_data->m_mutex.lock();
        m_data->m_current_timer = this->startTimer(50);
        m_data->m_mutex.unlock();
    }
}

void trigger::call_method(const abstract_qobject_connection *ci, void **args)
try {
    set_sender setter(ci->source());
    ci->call_slot(args);
} catch (python::error_already_set&) {
    PyErr_Print();
}

void trigger::timerEvent(QTimerEvent *)
{
    m_data->m_mutex.lock();
    if (!m_data->m_idle_methdos.empty()) {
        trigger_method m = m_data->m_idle_methdos.pop();
        thread_locker lock;
        call_method(m.ci, m.args);
    }

    if (m_data->m_idle_methdos.empty()) {
        this->killTimer(m_data->m_current_timer);
        m_data->m_current_timer = 0;
    }
    m_data->m_mutex.unlock();
}

// Function used by SignalManager to call a slot
// when the signal is a python method
void
trigger::fire(const QList<int> slot_indexes, const python::object &args)
{
    if (m_data->m_parent->signalsBlocked())
        return;

    foreach(int slot_index, slot_indexes) {
        abstract_qobject_connection* ci =
                m_data->m_connections.value(slot_index);

        Qt::ConnectionType c_type = ci->connection_type();
        if (c_type == Qt::AutoConnection) {
            if (m_data->m_parent->thread() != QThread::currentThread())
                c_type = Qt::QueuedConnection;
            else
                c_type = Qt::DirectConnection;
        }
        if (ci) {
            if (c_type == Qt::DirectConnection) {
                call_method(ci, args);
            } else {
                idle_call_method(ci, args);
            }
        }
    }
}

//Function called when slot is python callback
void
trigger::py_fire(const abstract_qobject_connection* ci, void **args)
{
    if (m_data->m_parent->signalsBlocked())
        return;

    call_method(ci, args);
}

int
callable_arg_count(PyObject *callable)
{
    PyObject *func_object = callable;
    long num_args = 0;
    bool is_method = false;

    if (PyMethod_Check(callable)) {
        PyMethodObject *meth_object = (PyMethodObject*) callable;
        func_object = PyMethod_GET_FUNCTION(meth_object);
        is_method = 1;
    }

    if (PyCallable_Check(func_object)) {
        PyCodeObject *obj_code =
                reinterpret_cast<PyCodeObject*>(
                reinterpret_cast<PyFunctionObject*>(func_object)->func_code);
        if (obj_code->co_flags & CO_VARARGS) {
            num_args = -1;
        } else {
            num_args = obj_code->co_argcount;
            if (num_args == 0 && is_method) {
                num_args = 1;
            }
        }
    }

    return num_args;
}


//convert the function signature to metaobject data string:
//[return_type\0][arg0,arg1,...\0]function_signature(arg_type,arg_type)\0
QByteArray
function_data(const QByteArray &ofunc,
              int &signature,
              int &parameters,
              int &type)
{
    QByteArray func(ofunc);
    QByteArray retvalue;
    const char DIV = '\0';

    retvalue = func.split(' ')[0];

    //remove unecessary words
    if (retvalue != func) {
        func.remove(0, retvalue.size());
    } else {
        retvalue.clear();
    }

    func.replace("const", "");
    func.replace('&', "");
    func.replace(' ', "");

    //this not work for templates with ","
    QByteArray ret;

    signature = parameters = type = -1;

    //return value
    if (!retvalue.isEmpty() && (retvalue != "void")) {
        type = 0;
        ret = retvalue + DIV;
    }

    if (!func.contains("()") && !func.contains("(void)")) {
        parameters = ret.size();
        int n_args = func.count(',')+1;
        for (int i=0; i < n_args; i++) {
            if (i>0)
                ret.append(',');

            ret.append(QString("arg__%1").arg(i));
        }
        ret.append(DIV);
    }

    signature = ret.size();
    ret.append(func);
    ret.append(DIV);

    return ret;
}

void
trigger::invalidate_metaobject() {
    delete m_data->m_metaobject;
    m_data->m_metaobject = 0;
    delete[] m_data->m_metadata;
    m_data->m_metadata = 0;
    delete[] m_data->m_metadata_indexes;
    m_data->m_metadata_indexes = 0;
}

static int
strreg(const QByteArray& s, QList<QByteArray>* strings)
{
    int idx = 0;
    for (int i = 0; i < strings->count(); ++i) {
        const QString &str = strings->at(i);
        if (str == s)
            return idx;
        idx += str.length() + 1;
    }
    strings->append(s);
    return idx;
}

void
trigger::update_metaobject() const
{
    Q_ASSERT(!m_data->m_metaobject);

    // these values are from moc source code, generator.cpp:66
    enum MethodFlags {
        AccessPrivate = 0x00,
        AccessProtected = 0x01,
        AccessPublic = 0x02,
        MethodMethod = 0x00,
        MethodSignal = 0x04,
        MethodSlot = 0x08,
        MethodConstructor = 0x0c,
        MethodCompatibility = 0x10,
        MethodCloned = 0x20,
        MethodScriptable = 0x40
    };

    QList<QByteArray> signal_list =
            signal_manager::instance().signals_of(m_data->m_parent);
    uint n_signals = signal_list.count();
    uint n_methods = n_signals + m_data->m_slot_list.count();
    int header[] = {2,            // revision
                    0,            // class name index in m_metadata
                    0, 0,         // classinfo and classinfo index, not used by us
                    n_methods, 0, // method count and method list index
                    0, 0,         // prop count and prop indexes
                    0, 0          // enum count and enum index
                    };

    const int HEADER_LENGHT = sizeof(header)/sizeof(int);
    header[5] = HEADER_LENGHT;
    // header size + 5 indexes per method + an ending zero
    m_data->m_metadata_indexes = new uint[HEADER_LENGHT + n_methods*5 + 1];
    std::memcpy(m_data->m_metadata_indexes, header, sizeof(header));

    QList<QByteArray> strings;
    strreg(m_data->m_className, &strings); // register class string
    const int NULL_INDEX = strreg("", &strings); // register a null string
    int index = HEADER_LENGHT;

    //write signals
    foreach(QByteArray signal, signal_list) {
        m_data->m_metadata_indexes[index++] = strreg(signal, &strings); // func name
        m_data->m_metadata_indexes[index++] = NULL_INDEX; // arguments
        m_data->m_metadata_indexes[index++] = NULL_INDEX; // normalized type
        m_data->m_metadata_indexes[index++] = NULL_INDEX; // tags
        m_data->m_metadata_indexes[index++] = AccessPublic | MethodSignal; // flags
    }
    //write slots
    foreach(QByteArray slot, m_data->m_slot_list) {
        m_data->m_metadata_indexes[index++] = strreg(slot, &strings); // func name
        m_data->m_metadata_indexes[index++] = NULL_INDEX; // arguments
        m_data->m_metadata_indexes[index++] = NULL_INDEX; // normalized type
        m_data->m_metadata_indexes[index++] = NULL_INDEX; // tags
        m_data->m_metadata_indexes[index++] = AccessPublic | MethodSlot; // flags
    }
    m_data->m_metadata_indexes[index++] = 0; // the end

    // create the m_metadata string
    QByteArray str;
    foreach(QByteArray signature, strings) {
        str.append(signature);
        str.append(char(0));
    }
    m_data->m_metadata = new char[str.count()];
    std::copy(str.begin(), str.end(), m_data->m_metadata);

    // create metaobject
    QMetaObject dynamic_metaobject = {
        { m_data->m_parent->metaObject(), m_data->m_metadata,
          m_data->m_metadata_indexes, 0 }
    };

    m_data->m_metaobject = new QMetaObject(dynamic_metaobject);
}

QT_BEGIN_MOC_NAMESPACE
const QMetaObject*
trigger::metaObject() const
{
    if (!m_data->m_metaobject)
        update_metaobject();
    return m_data->m_metaobject;
}
QT_END_MOC_NAMESPACE

} // namespace PySide

