/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <QByteArray>
#include <QDebug>

#include "parent_policy.hpp"
#include "boost_headers.hpp"
#include "abstract_qobject_connection.hpp"
#include "trigger.hpp"
#include "signal_manager.hpp"
#include "type_manager.hpp"

using namespace boost;

namespace PySide
{

abstract_qobject_connection::abstract_qobject_connection(QObject* source)
    : m_source(source)
{
}

abstract_qobject_connection::~abstract_qobject_connection()
{
}

void
abstract_qobject_connection::call_slot(void** slot_args) const
{
    python::list list_args;
    for (int i=0, max = m_signal_args.count(); i < max; i++) {
        python::object pval =
                type_manager::instance().to_python(m_signal_args[i],
                                                   slot_args[i+1]);
        list_args.append(pval);
    }
    call_slot(list_args);
}

signal_slot_connection::signal_slot_connection(QObject* source,
                                               const python::api::object& callback,
                                               const QStringList& signal_args,
                                               int num_slot_args)
    : abstract_qobject_connection(source), m_receiver(0), m_num_slot_args(num_slot_args)
{
    set_signal_args(signal_args);
    PyObject* py_callback = callback.ptr();
    if (PyMethod_Check(py_callback)) {
        m_function = python::object(python::handle<>(python::borrowed<>(PyMethod_GET_FUNCTION(py_callback))));
        if (PyObject* self = PyMethod_GET_SELF(py_callback)) {
            m_receiver = self;
            if (m_num_slot_args > -1)
                m_num_slot_args++;
        }
    } else {
        m_function = callback;
    }
}

void
signal_slot_connection::call_slot(const boost::python::list& slot_args) const
{
    python::list args;

    if (m_receiver) {
        python::object receiver(python::handle<>(python::borrowed<>(m_receiver)));
        args.append(receiver);
        args.extend(slot_args);
    } else {
        args = python::list(slot_args);
    }

    if (m_num_slot_args != -1 && m_num_slot_args < python::len(args)) {
        args = python::list(args.slice(0, m_num_slot_args));
    }
    m_function(*python::tuple(args));
}

bool
signal_slot_connection::equals(const boost::python::api::object& callback) const
{
    PyObject* py_callback = callback.ptr();
    if (PyMethod_Check(py_callback)) {
        return m_receiver == PyMethod_GET_SELF(py_callback) &&
                m_function.ptr() == PyMethod_GET_FUNCTION(py_callback);
    } else {
        return callback == m_function;
    }
    return false;
}

signal_signal_connection::signal_signal_connection(QObject* source,
                                                   QObject* receiver,
                                                   const pyqt_signal& signal)
    : abstract_qobject_connection(source),
      m_receiver(receiver),
      m_signal(signal)
{
    QStringList signal_args;
    signal.parse_signature(&signal_args);
    set_signal_args(signal_args);
}

void
signal_signal_connection::call_slot(const boost::python::list& slot_args) const
{
    signal_manager::instance().emit_(m_receiver, m_signal, slot_args);
}

bool
signal_signal_connection::equals(const boost::python::api::object& callback) const
{
    PyObject* py_callback = callback.ptr();
    PyObject* function = callback.ptr();
    if (PyMethod_Check(py_callback)) {
        python::extract< QObject*> x(PyMethod_GET_SELF(py_callback));
        if (!x.check() || m_receiver != x()) {
            return false;
        }
        function = PyMethod_GET_FUNCTION(py_callback);
    }

    PyObject* function_name = reinterpret_cast<PyFunctionObject*>(function)->func_name;
    char* str = PyString_AsString(function_name);
    return m_signal.function_name() == str;
}

} // namespace PySide

