#!/usr/bin/env python

##############################################################################
##
## This file is part of pymucl.
##
## pymucl is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## pymucl is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with pymucl.  If not, see <http://www.gnu.org/licenses/>.
##
##############################################################################

from PyQt4.QtGui import QWidget, QApplication, QPushButton, QLineEdit, QGridLayout, QTextEdit, QListWidget, QDialog, QLabel
from PyQt4.QtCore import Qt, QTimer, SIGNAL, SLOT

import sys
from datetime import datetime, date
import time

from telechat import ChatGui, GuiFactory, RoomManager, create_chat_client

class QtFactory(GuiFactory):

    def build_gui(self, chat, name):
        widget = ChatWindow(chat)
        widget.setWindowTitle(name)
        widget.show()
        return widget

class ChatWindow(QWidget, ChatGui):

    def __init__(self, chat, parent = None):
        QWidget.__init__(self, parent)

        self.chat = chat

        self.line = line = QLineEdit(self)
        line.connect(line, SIGNAL("editingFinished()"), self.click)

        button = QPushButton("Send", self)
        button.connect(button, SIGNAL("clicked()"), self.click)

        self.text = text = QTextEdit(self)
        text.setReadOnly(True)

        layout = QGridLayout()
        layout.addWidget(text, 0, 0, 1, 2)
        layout.addWidget(line, 1, 0, Qt.AlignBottom)
        layout.addWidget(button, 1, 1)

        self.setLayout(layout)

    def timestamp_format(self, timestamp):
        # TODO: use locals!
        dt = datetime.fromtimestamp(timestamp)
        if dt.date() == date.today():
            fst = "%X"
        else:
            fst = "%c"
        return dt.strftime(fst)
    
    def now_str(self):
        return self.timestamp_format(time.time())

    def click(self):
        if self.line.text():
            self.chat.send_msg(str(self.line.text()))
            self.line.clear()

    def closeEvent(self, event):
        self.chat.close()

    def msg_received(self, user, message, time, mtype):
        timestr = self.timestamp_format(time)
        self.text.append("[%s] %s: %s" % (timestr, user, message))

    def msg_sent(self, message, time):
        timestr = self.timestamp_format(time)
        self.text.append("[%s] %s: %s" % (timestr, self.chat.nickname(), message))

    def status_message(self, msg):
        self.text.append("[%s] %s" % (self.now_str(), msg))

    def chat_error(self, message):
        self.status_message("Error: %s" % message)

    def chat_connected(self):
        # TODO: connection message and jabber history give mixed up timestamps
        self.status_message("Connection established")

    def chat_closed(self):
        self.status_message("Connection closed")

class ChatSelection(QWidget):

    def __init__(self, client, parent = None):
        QWidget.__init__(self, parent)
        self.setWindowTitle("pymucl - Chat selection")

        self.client = client
        self.room_manager = RoomManager()

        self.rooms = rooms = QListWidget(self)

        join_button = QPushButton("Join", self)
        join_button.connect(join_button, SIGNAL('clicked()'), self.join)

        del_button = QPushButton("Remove", self)
        del_button.connect(del_button, SIGNAL('clicked()'), self.remove_room)

        add_button = QPushButton("Add", self)
        add_button.connect(add_button, SIGNAL('clicked()'), self.add_room)

        layout = QGridLayout()
        layout.addWidget(rooms, 0, 0, 1, 2)
        layout.addWidget(add_button, 1, 0)
        layout.addWidget(del_button, 1, 1)
        layout.addWidget(join_button, 2, 0, 1, 2)

        self.setLayout(layout)

        self.update_rooms()

    def update_rooms(self):
        self.rooms.clear()
        for name, acc, room in self.room_manager:
            self.rooms.addItem(name)
        self.rooms.setCurrentRow(0)

    def remove_room(self):
        del self.room_manager[self.rooms.currentRow()]
        self.room_manager.write()
        self.update_rooms()

    def add_room(self):
        cc = ChatCreation(self.client, self.room_manager)
        if cc.exec_() == cc.Accepted:
            self.update_rooms()

    def join(self):
        name, acc, room = self.room_manager[self.rooms.currentRow()]
        self.client.join_chat(room, acc, name)

class ChatCreation(QDialog):

    def __init__(self, client, room_manager, parent = None):
        QDialog.__init__(self, parent)

        self.accs = accs = client.get_accounts()

        self.client = client
        self.room_manager = room_manager

        self.name = name = QLineEdit(self)

        # TODO: account-selection is too small on the N900
        self.acc_widget = acc_widget = QListWidget(self)
        for acc in accs:
            acc_widget.addItem(acc)

        acc_widget.setCurrentRow(0)

        self.channel = channel = QLineEdit(self)

        button = QPushButton("Create", self)
        button.connect(button, SIGNAL('clicked()'), self.select)

        layout = QGridLayout()
        layout.addWidget(QLabel("Name:", self), 0, 0)
        layout.addWidget(name, 0, 1)
        layout.addWidget(QLabel("Account:", self), 1, 0)
        layout.addWidget(acc_widget, 1, 1)
        layout.addWidget(QLabel("Channel:", self), 2, 0)
        layout.addWidget(channel, 2, 1)
        layout.addWidget(button, 3, 1, 1, 2)

        self.setLayout(layout)

    def select(self):
        acc = self.accs[self.acc_widget.currentRow()]
        room = str(self.channel.text())
        name = str(self.name.text())
        self.room_manager.add_room(name, acc, room)
        self.room_manager.write()
        self.accept()

if __name__ == '__main__':
    # setting up the dbus-mainloop
    import dbus.mainloop.qt
    dbus.mainloop.qt.DBusQtMainLoop(set_as_default=True)

    app = QApplication(sys.argv)

    client = create_chat_client(QtFactory())

    cs = ChatSelection(client)
    cs.show()

    #timer = QTimer(app)
    #timer.setSingleShot(True)
    #timer.setInterval(0)
    #timer.connect(timer, SIGNAL("timeout()"), start_app)
    #timer.start()

    app.exec_()

