##############################################################################
##
## This file is part of pymucl.
##
## pymucl is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## pymucl is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with pymucl.  If not, see <http://www.gnu.org/licenses/>.
##
##############################################################################

import dbus

import telepathy
from telepathy.interfaces import CLIENT, CLIENT_HANDLER, CHANNEL, ACCOUNT_MANAGER, CHANNEL_TYPE_TEXT
from telepathy.constants import HANDLE_TYPE_ROOM, CHANNEL_TEXT_MESSAGE_TYPE_DELIVERY_REPORT, CHANNEL_TEXT_MESSAGE_TYPE_NORMAL, HANDLE_TYPE_CONTACT

from UserList import UserList
from ConfigParser import RawConfigParser
import os

DBUS_PROPERTIES = 'org.freedesktop.DBus.Properties'

class ChatGui:

    def msg_received(self, user, message, time, mtype):
        pass

    def msg_sent(self, message, time):
        pass

    def chat_error(self, message):
        pass

    def chat_connected(self):
        pass

    def chat_closed(self):
        pass

class GuiFactory:

    def build_gui(self, chat, name):
        raise NotImplementedError

class RoomManager(UserList):

    ROOM_KEY = 'room'
    ACCOUNT_KEY = 'account'

    CONFIG_FILE = os.path.expanduser('~/.pymucl')

    def __init__(self, l = None):
        UserList.__init__(self, l)

        if l == None:
            parser = RawConfigParser()
            parser.read(self.CONFIG_FILE)

            for section in parser.sections():
                room = parser.get(section, self.ROOM_KEY)
                account = parser.get(section, self.ACCOUNT_KEY)
                self.append((section, account, room))

    def write(self):
        parser = RawConfigParser()

        for name, account, room in self:
            parser.add_section(name)
            parser.set(name, self.ROOM_KEY, room)
            parser.set(name, self.ACCOUNT_KEY, account)

        out = open(self.CONFIG_FILE, 'wb')
        parser.write(out)
    
    def add_room(self, name, account, room):
        self.append((name, account, room))

class ChatChannel:

    # TODO: message type constants

    def __init__(self, bus):
        self.gui = None
        self.channel_o = None
        self.bus = bus

    def resolve_handle(self, htype, handle):
        return self.connection_o.InspectHandles(htype, [handle])[0]

    def beautify_username(self, name):
        # TODO: ugly workaround, sry but i'm tired :P
        index = name.index('/')
        return name[index+1:]

    def connect_channel(self, service_name, path):
        # TODO: use python-telepathy as soon as the ugly bug disappears
        self.service_name = service_name
        self.path = path

        con_path = '/' + service_name.replace('.', '/')
        self.connection_o = self.bus.get_object(service_name, con_path)

        self.channel_o = channel_o = self.bus.get_object(service_name, path)

        channel_o.connect_to_signal("Closed", self.closed_sig)

        channel_o.connect_to_signal("Received", self.received_sig)
        channel_o.connect_to_signal("Sent", self.sent_sig)
        channel_o.connect_to_signal("SendError", self.send_error_sig)

        #self[DBUS_PROPERTIES].Get(CHANNEL, "interfaces",
                #reply_handler = self.interfaces_cb,
                #error_handler = self.fatal_error_cb)

        if self.gui:
            self.gui.chat_connected()

    def close(self):
        if self.channel_o:
            self.channel_o.Close(reply_handler = void, error_handler = self.fatal_error_cb)

    def nickname(self):
        # TODO: find out our nickname in the chat
        return "Me"

    def send_msg(self, message):
        if self.channel_o:
            self.channel_o.Send(dbus.UInt32(CHANNEL_TEXT_MESSAGE_TYPE_NORMAL), message,
                    reply_handler = void,
                    error_handler = self.error_cb)
        else:
            self.gui.chat_error("Not connected!")

    def closed_sig(self):
        print "Connection closed"

        if self.gui:
            self.gui.chat_closed()

    def received_sig(self, mid, time, sender, mtype, flags, text):
        if mtype != CHANNEL_TEXT_MESSAGE_TYPE_DELIVERY_REPORT:
            if self.gui:
                sname = self.resolve_handle(dbus.UInt32(HANDLE_TYPE_CONTACT), sender)
                sname = self.beautify_username(sname)

                self.gui.msg_received(sname, text, time, mtype)

                self.channel_o.AcknowledgePendingMessages([mid],
                        reply_handler = void,
                        error_handler = self.error_cb)

    def send_error_sig(self, error, time, mtype, text):
        print "Error sending '%s'" % text

        if self.gui:
            self.gui.chat_error("Error sending '%s'" % text)

    def sent_sig(self, time, mtype, text):
        if self.gui:
            self.gui.msg_sent(text, time)

    def error_cb(self, error):
        print "Error: %s" % error

        if self.gui:
            self.gui.chat_error(error)

    def fatal_error_cb(self, error):
        self.error_cb(error)
        sys.exit(1)

class ChatClient(telepathy.server.Handler, telepathy.server.DBusProperties):

    def __init__(self, bus, client_name, gui_factory = None):
        self.gui_factory = gui_factory
        self.name = name = '.'.join ([CLIENT, client_name])
        object_path = '/' + name.replace('.', '/')

        bus_name = dbus.service.BusName(name, bus=bus)

        telepathy.server.Handler.__init__(self, bus_name, object_path)
        telepathy.server.DBusProperties.__init__(self)

        self.chats = {}
        self.requests = {}
        self.bus = bus

        self._implement_property_get(CLIENT, {
            'Interfaces': lambda: [ CLIENT_HANDLER ],
            })

        self._implement_property_get(CLIENT_HANDLER, {
            'HandlerChannelFilter': lambda: dbus.Array([
                dbus.Dictionary({
                    'org.freedesktop.Telepathy.Channel.ChannelType':      CHANNEL_TYPE_TEXT,
                    'org.freedesktop.Telepathy.Channel.TargetHandleType': HANDLE_TYPE_ROOM,
                    'org.freedesktop.Telepathy.Channel.Requested':        True,
                }, signature='sv'),
            ], signature='a{sv}')
        })

    def join_chat(self, room, account, name = None):
        # TODO: port to python-telepathy or use Connection/Requests interface
        # TODO: get pending messages

        print room
        print account

        props = {
            "org.freedesktop.Telepathy.Channel.ChannelType":        CHANNEL_TYPE_TEXT,
            "org.freedesktop.Telepathy.Channel.TargetID":           room,
            "org.freedesktop.Telepathy.Channel.TargetHandleType":   HANDLE_TYPE_ROOM,
        }

        # preparing the request
        # TODO: make this async? it's fast because of the request, anyway
        channel_dispatcher = self.bus.get_object("org.freedesktop.Telepathy.ChannelDispatcher", "/org/freedesktop/Telepathy/ChannelDispatcher")
        request_p = channel_dispatcher.EnsureChannel(dbus.ObjectPath(account), dbus.Dictionary(props, signature="sv"), dbus.Int64(0), self.name)
        request_o = self.bus.get_object("org.freedesktop.Telepathy.ChannelDispatcher", request_p)

        # creating the object
        chat = ChatChannel(self.bus)
        self.requests[request_p] = chat

        # building the gui
        if self.gui_factory:
            gui = self.gui_factory.build_gui(chat, name)
            chat.gui = gui

        # finishing up
        request_o.Proceed(reply_handler = void, error_handler = chat.fatal_error_cb)

        return chat

    def get_accounts(self):
        # TODO: port to python-telepathy if possible
        # TODO: using async some way?
        # TODO: get some nice account-names in neat account-objects
        am = self.bus.get_object(ACCOUNT_MANAGER, "/org/freedesktop/Telepathy/AccountManager")
        return am.Get(ACCOUNT_MANAGER, "ValidAccounts")

    def HandleChannels(self, account, connection, channels, requests_satisfied, user_action_time, handler_info):
        service_name = connection[1:].replace('/', '.')

        print "hi"

        # find our request
        request = None
        for r in requests_satisfied:
            if r in self.requests:
                request = r
                break

        print request
        print service_name

        for c in channels:
            print 'client.requests["%s"].connect_channel("%s", "%s")' % (request, service_name, c[0])
            self.requests[request].connect_channel(service_name, c[0])

        # TODO: move channels from requests to chats

        print 'done'

def void(*args):
    pass

def create_chat_client(gui_factory = None):
    bus = dbus.SessionBus()
    client = ChatClient(bus, 'pymucl', gui_factory)
    return client

def test():
    global client

    bus = dbus.SessionBus()

    client = ChatClient(bus, 'pymucl')

    acc = client.get_accounts()[0]
    chat = client.join_chat("testme@conference.chaossource.net", acc)

    return False

if __name__ == '__main__':
    import dbus.glib
    import gobject

    gobject.timeout_add(0, test)
    loop = gobject.MainLoop()
    loop.run()

