#include <stdio.h>
#include <string.h>

#include <X11/Xlib.h>
#include <SDL.h>
#include <SDL_syswm.h>

#include <glib.h>
#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>

#include "private.h"
#include "cinema.h"
#include "debug.h"

#define DEBUG_DOMAIN "CINEMA"
#define TIMER_INTERVAL 1000

static Window window;
static char *media_uri = NULL;
static GstElement *pipeline = NULL;

static SDL_TimerID timer_id = 0;

void CIN_DeInit()
{
	TRACE("DeInit");

	if (pipeline) {
		gst_element_set_state(pipeline, GST_STATE_NULL);
		gst_object_unref(pipeline);
		pipeline = NULL;
	}
	if (timer_id) {
		SDL_RemoveTimer(timer_id);
		timer_id = 0;
	}
	g_free(media_uri);
	media_uri = NULL;
	gst_deinit();
}

int CIN_Init()
{
	SDL_SysWMinfo wminfo;
	SDL_VERSION(&wminfo.version);

	if ( SDL_GetWMInfo(&wminfo) ) {
		window = wminfo.info.x11.window;
	} else {
		return SDL_FALSE;
	}

	gst_init(NULL, NULL);

	TRACE("Init; window xid is 0x%lx", (unsigned long) window);

	return SDL_TRUE;
}

/* Because I'm not sure wheter it is good practice to leave GST pipelines
 * floating around in paused state, I defer creation of it until CIN_Play(). */
int CIN_LoadCIN(const char *url)
{
	g_free(media_uri);
	if (g_ascii_strncasecmp("file:/", url, 6) == 0) {
		media_uri = g_strdup(url);
	} else {
		media_uri = g_filename_to_uri(url, NULL, NULL);
	}
	TRACE("Will play \"%s\"", media_uri);
	return SDL_TRUE;
}

static Uint32 timer_cb(Uint32 interval, void *param)
{
	GstBus *bus;
	GstMessage *msg;
	const GstMessageType msg_mask = GST_MESSAGE_ERROR | GST_MESSAGE_EOS;

	if (!pipeline) {
		timer_id = 0;
		return 0;
	}

	bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));
	if (!bus) {
		timer_id = 0;
		return 0;
	}

	while ((msg = gst_bus_pop_filtered(bus, msg_mask))) {
		switch (GST_MESSAGE_TYPE(msg)) {
			case GST_MESSAGE_ERROR: {
				SDL_Event event;
				GError *err;
				gchar *debug;

				gst_message_parse_error(msg, &err, &debug);
				WARN("GStreamer Pipeline Error: %s", err->message);

				event.type = SDL_USEREVENT;
				event.user.code = SDL_CINEMAEVENT | SDL_CINEMA_ERR;
				event.user.data1 = NULL;
				event.user.data2 = NULL;
				SDL_PushEvent(&event);

				g_error_free(err);
				g_free(debug);

				timer_id = 0;
				return 0;
			}
			break;
			case GST_MESSAGE_EOS: {
				TRACE("GStreamer Pipeline EOS");

				SDL_Event event;
				event.type = SDL_USEREVENT;
				event.user.code = SDL_CINEMAEVENT | SDL_CINEMA_EOS;
				event.user.data1 = NULL;
				event.user.data2 = NULL;
				SDL_PushEvent(&event);

				timer_id = 0;
				return 0;
			}
			break;
			default:
			break;
		}
		gst_message_unref(msg);
	}

	return interval;
}

void CIN_Play()
{
	TRACE("Start playing now");

	if (!pipeline) {
		GstElement *playbin, *videosink;

		pipeline = gst_pipeline_new("cinema");
		playbin = gst_element_factory_make("playbin2", "playbin");
		videosink = gst_element_factory_make("xvimagesink", "videosink");
		gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(videosink), window);
		g_object_set(G_OBJECT(playbin),
			"uri", media_uri,
			"video-sink", videosink,
			NULL);
		gst_bin_add(GST_BIN(pipeline), playbin);

		timer_id = SDL_AddTimer(TIMER_INTERVAL, timer_cb, NULL);
	} else {
		/* Resuming existing pipeline */
	}

	gst_element_set_state(pipeline, GST_STATE_PLAYING);
}

void CIN_Pause()
{
	TRACE("Pause");

	gst_element_set_state(pipeline, GST_STATE_PAUSED);
}

void CIN_Stop()
{
	TRACE("Stop playing now");

	if (pipeline) {
		gst_element_set_state(pipeline, GST_STATE_NULL);
		gst_object_unref(pipeline);
		pipeline = NULL;
	}

	if (timer_id) {
		SDL_RemoveTimer(timer_id);
		timer_id = 0;
	}
}

