// (c) Copyright 2006 notaz, All rights reserved.
// Free for non-commercial use.

// For commercial use, separate licencing terms must be obtained.

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <strings.h>
#include <linux/limits.h>
#include <glib.h>
#include <libosso.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "maemo.h"
#include "../common/emu.h"
#include "emu.h"
#include "940ctl.h"
#include "version.h"
#include "menu/gconf.h"

#define GPLAYER1_KEY kGConfPath "/player1" kGConfPlayerKeyboardPath "/"
#define GPLAYER2_KEY kGConfPath "/player2" kGConfPlayerKeyboardPath "/"

int use_gui = 0;
char **g_argv;
osso_context_t* osso = NULL;
GMainContext* g_context = NULL;
GMainLoop* g_loop = NULL;
int game_running = 0;

void parse_cmd_line(int argc, char *argv[])
{
	int x, unrecognized = 0;

	for(x = 1; x < argc; x++)
	{
		if(argv[x][0] == '-')
		{
			if(strcasecmp(argv[x], "-gui") == 0) {
				use_gui = 1;
			}
			else {
				unrecognized = 1;
				break;
			}
		} else {
			/* External Frontend: ROM Name */
			FILE *f;
			strncpy(romFileName, argv[x], PATH_MAX);
			romFileName[PATH_MAX-1] = 0;
			f = fopen(romFileName, "rb");
			if (f) fclose(f);
			else unrecognized = 1;
			engineState = PGS_ReloadRom;
			break;
		}
	}

	if (unrecognized) {
		printf("\n\n\nPicoDrive v" VERSION " (c) notaz, 2006-2007\n");
		printf("usage: %s [options] [romfile]\n", argv[0]);
		printf( "options:\n"
				"-menu <menu_path> launch a custom program on exit instead of default gp2xmenu\n"
				"-state <param>    pass '-state param' to the menu program\n"
				"-config <file>    use specified config file instead of default 'picoconfig.bin'\n"
				"                  see currentConfig_t structure in emu.h for the file format\n"
				"-selectexit       pressing SELECT will exit the emu and start 'menu_path'\n");
	}
}

static gint dbus_callback( const gchar *interface, const gchar *method, GArray *arguments, gpointer data, osso_rpc_t *retval ) {
	retval->type = DBUS_TYPE_BOOLEAN;

	if( engineState == PGS_Menu ) {
		// Only if we haven't received the startup command yet.
		printf("Osso: Startup method is: %s\n", method);

		if (strcmp(method, "game_run") == 0) {
			emu_set_state( PGS_ReloadRom );
			retval->value.b = TRUE;
		} else if (strcmp(method, "game_continue") == 0) {
			emu_set_state( PGS_ReloadRomState );
			retval->value.b = TRUE;
		} else if (strcmp(method, "game_restart") == 0) {
			emu_set_state( PGS_ReloadRom );
			retval->value.b = TRUE;
		} else if (strcmp(method, "game_close") == 0) {
			emu_set_state( PGS_Quit );
			retval->value.b = TRUE;
		} else {
			emu_set_state( PGS_ReloadRom );
			retval->value.b = FALSE;
		}
	} else {
		if (strcmp(method, "game_close") == 0) {
			printf("Osso: quitting because of D-Bus close message\n");
			emu_set_state( PGS_Quit );
			retval->value.b = TRUE;
		} else {
			retval->value.b = FALSE;
		}
	}

	return OSSO_OK;
}


static gboolean ossoTimeoutCallback(gpointer data)
{
	/*if (startupCommand == STARTUP_COMMAND_UNKNOWN) {
		// Assume that after N seconds we're not going to get a startup reason.
		startupCommand = STARTUP_COMMAND_INVALID;
	}*/

	return FALSE; // This is a timeout, don't call us ever again.
}

static void dbus_hw_callback(osso_hw_state_t *state, gpointer data)
{
	if (state->shutdown_ind) {
		// Shutting down. Try to quit gracefully.
		emu_set_state( PGS_Quit );
	}
	if (state->system_inactivity_ind) {
		// Screen went off, and power saving is active.
		emu_set_state( PGS_Quit );
	}
}

static void osso_init() {
	g_type_init();
	g_set_prgname( "picodrive" );
	g_set_application_name( "PicoDrive" );
	g_context = g_main_context_default();
	g_loop = g_main_loop_new( g_context, FALSE );

	osso = osso_initialize( "com.jcq.picodrive", "1.0", FALSE, NULL );
	if( !osso ) {
		fprintf( stderr, "Error initializing osso\n" );
		exit(1);
	}
	osso_rpc_set_default_cb_f( osso, dbus_callback, NULL );

	osso_hw_state_t hwStateFlags = { FALSE };
	hwStateFlags.shutdown_ind = TRUE;
	hwStateFlags.system_inactivity_ind = TRUE;
	osso_hw_set_event_cb(osso, &hwStateFlags, dbus_hw_callback, 0);

	printf( "libosso initialized\n" );

}

static void osso_deinit() {
	if( !osso ) return;

	// Send a goodbye message to the launcher
	osso_return_t ret;
	osso_rpc_t retval = { 0 };
	ret = osso_rpc_run(osso, "com.jcq.picodrive.startup",
				"/com/jcq/picodrive/startup", "com.jcq.picodrive.startup",
				game_running ? "game_pause" : "game_close", &retval, DBUS_TYPE_INVALID);

	if (ret != OSSO_OK) {
		printf("Osso: failed to notify launcher\n");
	}
	osso_rpc_free_val(&retval);

	osso_deinitialize(osso);
	g_main_loop_unref(g_loop);
	g_main_context_unref(g_context);

	osso = NULL;
}

static void osso_config() {
	GConfClient* gcc = gconf_client_get_default();
	char* rom = gconf_client_get_string( gcc, kGConfRomFile, NULL );
	if( rom ) {
		strcpy( romFileName, rom );
		printf( "Loading ROM %s...", romFileName );
	}

	if( gconf_client_get_bool( gcc, kGConfSound, NULL ) )
		currentConfig.EmuOpt |= 0x04;
	else
		currentConfig.EmuOpt &= ~0x04;

	if( gconf_client_get_bool( gcc, kGConfDisplayFramerate, NULL ) )
		currentConfig.EmuOpt |= 0x02;
	else
		currentConfig.EmuOpt &= ~0x02;

	currentConfig.Frameskip = gconf_client_get_int( gcc, kGConfFrameskip, NULL ) - 1;

	memset( key_map, 0, sizeof(key_map) );
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "a", NULL) & 0xFF ] = MDB_A;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "b", NULL) & 0xFF ] = MDB_B;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "c", NULL) & 0xFF ] = MDB_C;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "x", NULL) & 0xFF ] = MDB_X;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "y", NULL) & 0xFF ] = MDB_Y;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "z", NULL) & 0xFF ] = MDB_Z;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dn", NULL) & 0xFF ] = MDB_UP;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "ds", NULL) & 0xFF ] = MDB_DOWN;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "de", NULL) & 0xFF ] = MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dw", NULL) & 0xFF ] = MDB_LEFT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "start", NULL) & 0xFF ] = MDB_START;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dne", NULL) & 0xFF ] = MDB_UP | MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dnw", NULL) & 0xFF ] = MDB_UP | MDB_LEFT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dse", NULL) & 0xFF ] = MDB_DOWN | MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dsw", NULL) & 0xFF ] = MDB_DOWN | MDB_LEFT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "quit", NULL) & 0xFF ] = CMD_QUIT;
}

int main(int argc, char *argv[])
{
	romFileName[0] = 0;
	putenv("SDL_VIDEO_X11_WMCLASS=picodrive");
	g_argv = argv;

	chdir( "/opt/picodrive" );
	engineState = PGS_ReloadRom;
	emu_setDefaultConfig();

	if (argc > 1) parse_cmd_line( argc, argv );
	if( use_gui ){
		osso_init();
		engineState = PGS_Menu;
		osso_config();
	}

	if( !strlen(romFileName) ) {
		fprintf( stderr, "No ROM loaded, exiting" );
		osso_deinit();
		exit(1);
	}

	maemo_init();
	sharedmem_init();
	emu_Init();


	for (;;)
	{
		switch (engineState)
		{
			case PGS_Menu:
				usleep( 50000 );
				break;

			case PGS_ReloadRom:
				if (emu_ReloadRom())
					engineState = PGS_Running;
				else {
					printf("PGS_ReloadRom == 0\n");
					engineState = PGS_Menu;
				}
				break;
			case PGS_ReloadRomState:
				if (emu_ReloadRom())
					engineState = PGS_Running;
				else {
					printf("PGS_ReloadRom == 0\n");
					engineState = PGS_Menu;
				}
				emu_load_snapshot();
				break;
			case PGS_RestartRun:
				engineState = PGS_Running;

			case PGS_Running:
				game_running = 1;
				emu_Loop();
				break;

			case PGS_Quit:
				if( use_gui ) emu_save_snapshot();
				goto endloop;

			default:
				printf("engine got into unknown state (%i), exitting\n", engineState);
				goto endloop;
		}
	}

	endloop:

	osso_deinit();

	emu_Deinit();
	sharedmem_deinit();
	maemo_deinit();

	return 0;
}
