
/*
	MAEMO specific emulator code

	Uses SDL for video, audio and event processing

*/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <fcntl.h>
#include <errno.h>
#include <SDL/SDL.h>
#include <glib.h>

#include "maemo.h"
#include "usbjoy.h"
#include "../common/arm_utils.h"
#include "emu.h"

#define min(a,b) (a<b?a:b)

extern GMainContext* g_context;

int running = 0;
pthread_t sdl_thread;

void *md_screen;
#define SND_BUFFER_SIZE 0x10000
unsigned char sound_buffer[SND_BUFFER_SIZE];
volatile int soundptrr, soundptrw;

SDL_Surface* screen;
unsigned long keystates;

void sdl_key( SDLKey key, int press ) {
	unsigned short cmd = key_map[key];
	if( cmd < 0x8000 ) {
		if( press ) keystates |= cmd; else keystates &= ~cmd;
	} else switch( cmd ) {
		case CMD_QUIT:
			running = 0;
			emu_set_state(PGS_Quit);
		break;
	}
}

void* SDL_Loop( void* obj ) {
	while( running ) {
		SDL_Event event;
		if( SDL_PollEvent( &event ) ) {
			switch( event.type ) {
			case SDL_MOUSEBUTTONDOWN:
				break;
			case SDL_KEYDOWN:
				sdl_key( event.key.keysym.scancode, 1 );
				break;
			case SDL_KEYUP:
				sdl_key( event.key.keysym.scancode, 0 );
				break;
			case SDL_QUIT:
				running = 0;
				engineState = PGS_Quit;
				break;
			}
		} else {
			if( g_context ) g_main_context_iteration( g_context, FALSE );
		}
		SDL_Delay( 50 );
	}
	return NULL;
}

void audio_loop( void* userdata, Uint8* stream, int len ) {
	register int ptrr = soundptrr, ptrw = soundptrw;
	int total = ptrw - ptrr;
	if( total < 0 ) {
		total = min( SND_BUFFER_SIZE - ptrr, len );
		memcpy( stream, sound_buffer + ptrr, total );
		stream += total;
		len -= total;
		ptrr = ( ptrr + total ) % SND_BUFFER_SIZE;
		total = ptrw;
	}
	total = min( total, len );
	memcpy( stream, sound_buffer + ptrr, total );
	soundptrr = ( ptrr + total ) % SND_BUFFER_SIZE;
}

/* video stuff */
void maemo_video_flip(void)
{
	register int i, j;
	register unsigned short *buffer = md_screen;
	unsigned short *frame = screen->pixels;
	register unsigned short *scan1 = frame;
	register unsigned short *scan2 = frame;
	for( j = 0; j < 240; j++ ) {
		scan1 = frame + j*800*2 + 80;
		scan2 = frame + j*800*2 + 800 + 80;
		for( i = 0; i < 320; i++ ) {
			*scan1++ = *buffer;
			*scan1++ = *buffer;
			*scan2++ = *buffer;
			*scan2++ = *buffer++;
		}
	}
	SDL_Flip(screen);
}

void maemo_memset_buffer(int offset, int byte, int len)
{
	memset( md_screen + offset, byte, len );
}


void maemo_cls(void)
{
	memset( md_screen, 0, 320*240*2 );
}


unsigned long maemo_joystick_read(int allow_usb_joy)
{
	return keystates;
}

static int s_oldrate = 0, s_oldbits = 0, s_oldstereo = 0;

void maemo_start_sound(int rate, int bits, int stereo)
{
	SDL_AudioSpec as;
	memset( &as, 0, sizeof(as) );
	as.callback = audio_loop;
	as.channels = stereo ? 2 : 1;
	as.format = AUDIO_S16;
	as.freq = rate;
	as.samples = 0x400;

	memset( sound_buffer, 0, sizeof(sound_buffer) );
	soundptrw = soundptrr = 0;

	SDL_OpenAudio( &as, NULL );
	SDL_PauseAudio(0);
}


void maemo_sound_write(void *buff, int len)
{
	register int ptrr = soundptrr, ptrw = soundptrw;
	register int total = ptrr - ptrw - 1;
	if( total < 0 ) {
		total = min( SND_BUFFER_SIZE - ptrw, len );
		memcpy( sound_buffer + ptrw, buff, total );
		buff += total;
		len -= total;
		ptrw = ( ptrw + total ) % SND_BUFFER_SIZE;
	}
	total = min( len, total );
	memcpy( sound_buffer + ptrw, buff, total );
	soundptrw = ( ptrw + total ) % SND_BUFFER_SIZE;
}

void maemo_sound_sync(void)
{
	//ioctl(sounddev, SOUND_PCM_SYNC, 0);
}

void maemo_sound_volume(int l, int r)
{
 	/*l=l<0?0:l; l=l>255?255:l; r=r<0?0:r; r=r>255?255:r;
 	l<<=8; l|=r;
 	ioctl(mixerdev, SOUND_MIXER_WRITE_PCM, &l);*/ /*SOUND_MIXER_WRITE_VOLUME*/
}

/* common */
void maemo_init(void)
{
	printf("entering init()\n"); fflush(stdout);

	SDL_Init( SDL_INIT_VIDEO | SDL_INIT_AUDIO | SDL_INIT_EVENTTHREAD );
	SDL_WM_SetCaption( "PicoDrive for Maemo 1.0", NULL );
	SDL_ShowCursor( SDL_DISABLE );
	SDL_WM_GrabInput( SDL_GRAB_OFF );

	screen = SDL_SetVideoMode( 800, 480, 16, SDL_HWSURFACE | SDL_DOUBLEBUF );
	SDL_WM_ToggleFullScreen( screen );
	SDL_SetCursor( SDL_DISABLE );

	md_screen = malloc( 320*240*2 );

	keystates = 0;
	/* init usb joys -GnoStiC */
	maemo_usbjoy_init();

	running = 1;
	pthread_create( &sdl_thread, NULL, SDL_Loop, NULL );

	printf("exitting init()\n"); fflush(stdout);
}

char *ext_menu = 0, *ext_state = 0;

void maemo_deinit(void)
{
	running = 0;
	pthread_join( sdl_thread );

	maemo_usbjoy_deinit();

	SDL_PauseAudio(1);
	SDL_CloseAudio();

	free( md_screen );

	SDL_FreeSurface( screen );
	SDL_Quit();

	printf("all done, quitting ");
}

/* lprintf */
void lprintf(const char *fmt, ...)
{
	va_list vl;

	va_start(vl, fmt);
	vprintf(fmt, vl);
	va_end(vl);
}

