#include "TrailTrackModel.h"

#include <QDebug>
#include <QTextStream>

TrailTrackModel::TrailTrackModel(QObject* parent): QAbstractTableModel(parent), m_recording(false)
{

}

QVariant TrailTrackModel::data(const QModelIndex& index, int role) const
{
    QtMobility::QGeoPositionInfo pos = m_track[index.row()];
    //qDebug() << index << pos;

    if (role == Qt::DisplayRole) {
        if (index.column() == 0) {
            return pos.timestamp().toString(Qt::ISODate);
        } else if (index.column() == 1) {
            return pos.coordinate().longitude();
        } else if (index.column() == 2) {
            return pos.coordinate().latitude();
        }
    }

    return QVariant();
}
QVariant TrailTrackModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole){
        if (section == 0) {
            return "Time";
        } else if (section == 1) {
            return "Longitude";
        } else if (section == 2) {
            return "Latitude";
        }
    }
return QVariant();
}

int TrailTrackModel::columnCount(const QModelIndex& parent) const
{
    return 3;
}

int TrailTrackModel::rowCount(const QModelIndex& parent) const
{
    return m_track.count();
}

TrailTrackModel::~TrailTrackModel()
{

}

void TrailTrackModel::positionUpdated(const QGeoPositionInfo& pos)
{
    //qDebug() << pos;
    m_currentPosition = pos;

    if (m_recording && pos.attribute(QGeoPositionInfo::HorizontalAccuracy) <= 100) {
        beginInsertRows(QModelIndex(), rowCount(), rowCount() + 1);
        m_track.append(pos);
        endInsertRows();
    }
}

QtMobility::QGeoPositionInfo& TrailTrackModel::currentPosition()
{
    return m_currentPosition;
}

void TrailTrackModel::setRecord(bool r)
{
    qDebug() << "Record:" << r;
    m_recording = r;
}

void TrailTrackModel::clear()
{
    beginRemoveRows(QModelIndex(), 0, rowCount());
    m_track.clear();
    endRemoveRows();
}

Qt::ItemFlags TrailTrackModel::flags(const QModelIndex &index) const
{
    return Qt::ItemIsEnabled;
}

QString TrailTrackModel::toText()
{
    QString str;
    QTextStream out(&str);
    out << "<?xml version=\"1.0\" standalone=\"yes\"?>" << endl;
    out << "<?xml-stylesheet type=\"text/xsl\" href=\"details.xsl\"?>" << endl;
    out << "<gpx" << endl;
    out << "version=\"1.0\"" << endl;
    out << "creator=\"PGZ Every Trail for Qt\"" << endl;
    out << "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"" << endl;
    out << "xmlns=\"http://www.topografix.com/GPX/1/0\"" << endl;
    out << "xmlns:topografix=\"http://www.topografix.com/GPX/Private/TopoGrafix/0/1\"" << endl;
    out << "xsi:schemaLocation=\"http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd http://www.topografix.com/GPX/Private/TopoGrafix/0/1 http://www.topografix.com/GPX/Private/TopoGrafix/0/1/topografix.xsd\">" << endl;

    out << "<trk>" << endl;
    out << "<trkseg>" << endl;

    foreach(QGeoPositionInfo pos, m_track) {
        out << "<trkpt lat=\""<< pos.coordinate().latitude() << "\" lon=\"" << pos.coordinate().longitude() << "\">" << endl;
        out << "<ele>" << pos.coordinate().altitude() << "</ele>" << endl;
        out << "<time>" << pos.timestamp().toString(Qt::ISODate) << "</time>" << endl;
        out << "<sym>Waypoint</sym>" << endl;
        out << "</trkpt>" << endl;
    }
    out << "</trkseg>" << endl;
    out << "</trk>" << endl;
    out << "</gpx>" << endl;

    return str;
}
