#ifndef TRAILINTERFACE_H
#define TRAILINTERFACE_H

#include <QString>
#include <QAbstractListModel>
#include <QProgressDialog>
#include <QPushButton>
#include <QMap>
#include  <QtLocation/QGeoPositionInfoSource>

using namespace QtMobility;

#include "trailtrip.h"

class QNetworkAccessManager;
class QNetworkReply;
class QAuthenticator;

class TrailInterface : public QAbstractListModel
{
    Q_OBJECT
public:
    TrailInterface();

    //!Website functions
    //!login to the service
    void login(const QString&, const QString&);

    //!get list of trips for a user
    void trips(const QString &user, const QString& pass, int uid);

    //!Search for trips
    void tripSearch(bool searchTerm, bool searchProximity, const QString &term, const QGeoPositionInfo&, const QString proximity);

    //!Create a basic trip
    void createBasicTrip(const QString &title, const QString &description, const QString &activity, int visibility);

    //!Add location to trip
    void tripSetGPX(long trip_id, const QByteArray& gpx);

    //!Complete a trip
    void tripComplete(long trip_id);

    //!Load an image syncronously
    QPixmap getImage(QUrl);


    void getTripXML(long trip_id);

    //!General functions
    //!Return the credentials for the current user
    QString currentUser(){return m_user;}
    QString currentPassword(){return m_pass;}
    long currentUserId(){return m_uid;}

    const QList<TrailTrip*> tripList(){return m_trips;}
    TrailTrip* trip(int t){return m_trips[t];}

    //!Get list of activities
    QStringList activities();

    //!Model functions
    virtual QVariant data(const QModelIndex& index, int role = Qt::DisplayRole) const;
    virtual int rowCount(const QModelIndex& parent = QModelIndex()) const;

    QGeoPositionInfoSource* location() const {return m_location;}

signals:

    void busy(bool);

    //!When logged in, emit the user id or -1
    void loginComplete(long);

    //!Emitted when some trips are loaded
    void tripsLoaded(bool);

    //!Emitted when the activites have been loaded on startup
    void gotActivities(const QStringList&);

    //!Trip creation
    void createBasicTripDone(long);
    void tripSetGPXDone(long);
    void tripCompleteDone(long);

    //!Track data loaded
    void trackLoaded(const QDomDocument&);

private:
    //!Login details for the everytrail api
    static const QString key;
    static const QString secret;

    //!Log in details for the current user
    QString m_user;
    QString m_pass;
    long m_uid;

    //!Possible activities
    QMap<int, QString> m_activities;

    //!The users trips
    QList<TrailTrip*> m_trips;

    void loadActivities();

    QNetworkAccessManager *m_networkManager;

    QGeoPositionInfoSource *m_location;

    //!Parse replies
    long parseLogin(const QString&);
    bool parseTrips(const QString&);
    bool parseActivities(const QString&);
    long parseCreateBasicTrip(const QString&);
    long parseTripSetGPX(const QString&);
    long parseTripComplete(const QString&);
    bool parseTracks(const QString&);

private slots:
    void slotAuthenticationRequired ( QNetworkReply * reply, QAuthenticator * authenticator );
    void slotReplyFinished ( QNetworkReply* );
};

static void forceButtonColor(QPushButton* btn, QColor c)
{
    QPalette p = btn->palette();
    p.setColor(QPalette::ButtonText, c);
    btn->setPalette(p);
}

#endif // TRAILINTERFACE_H
