#include "trailinterface.h"
#include <QtNetwork/QNetworkAccessManager>
#include <QUrl>
#include <QtNetwork/QAuthenticator>
#include <QDebug>
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QtXml/QDomDocument>
#include <QMessageBox>
#include <QApplication>

#include "trailtrip.h"
#include "formpost.h"

const QString TrailInterface::key = QString("da0a1054b2dc9e8fd52d1af480ff1d9e");
const QString TrailInterface::secret = QString("cf8282a798c96cc4");

TrailInterface::TrailInterface()
{
    m_networkManager = new QNetworkAccessManager();
    connect(m_networkManager, SIGNAL(authenticationRequired(QNetworkReply*,QAuthenticator*)), this, SLOT(slotAuthenticationRequired(QNetworkReply*,QAuthenticator*)));
    connect(m_networkManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(slotReplyFinished(QNetworkReply*)));

    loadActivities();

    //Create the location source
    m_location = QtMobility::QGeoPositionInfoSource::createDefaultSource(this);

    if (m_location) {
	m_location->setPreferredPositioningMethods(QGeoPositionInfoSource::AllPositioningMethods);
        m_location->setUpdateInterval(5000);
        m_location->startUpdates();
    }

}

void TrailInterface::slotAuthenticationRequired ( QNetworkReply * reply, QAuthenticator * authenticator )
{
    authenticator->setUser(key);
    authenticator->setPassword(secret);
}

void TrailInterface::login(const QString& user, const QString& pass)
{
    emit busy(true);

    QUrl url("http://www.everytrail.com/api/user/login");

    url.addQueryItem("version", "3");
    url.addQueryItem("username", user);
    url.addQueryItem("password", pass);

    //Store the login details here for later use where required
    m_user = user;
    m_pass = pass;

    m_networkManager->get(QNetworkRequest(url));
}

void TrailInterface::trips(const QString &user, const QString& pass, int uid)
{
    emit busy(true);

    QUrl url("http://www.everytrail.com/api/user/trips");

    url.addQueryItem("version", "3");
    url.addQueryItem("username", user);
    url.addQueryItem("password", pass);
    url.addQueryItem("user_id", QString::number(uid));

    m_networkManager->get(QNetworkRequest(url));
}

void TrailInterface::loadActivities()
{
    emit busy(true);

    QUrl url("http://www.everytrail.com/api/activities");

    m_networkManager->get(QNetworkRequest(url));
}

void TrailInterface::slotReplyFinished(QNetworkReply* reply)
{
    qDebug() << "Reply Finished";

    emit busy(false);

    if (reply->url().path() == "/api/user/login")
    {
        qDebug() << "Reply from login";
        emit(loginComplete(parseLogin(reply->readAll())));
    }

    if (reply->url().path() == "/api/user/trips")
    {
        qDebug() << "Reply from trips";
        emit(tripsLoaded(parseTrips(reply->readAll())));
    }

    if (reply->url().path() == "/api/activities")
    {
        qDebug() << "Reply from activities";
        parseActivities(reply->readAll());
    }

    if (reply->url().path() == "/api/trip/create")
    {
        qDebug() << "Reply from create trip";
        emit(createBasicTripDone(parseCreateBasicTrip(reply->readAll())));
    }

    if (reply->url().path() == "/api/trip/addgpx")
    {
        qDebug() << "Reply from setgpx";
        emit(tripSetGPXDone(parseTripSetGPX(reply->readAll())));
    }

    if (reply->url().path() == "/api/trip/complete")
    {
        qDebug() << "Reply from trip complete";
        emit(tripCompleteDone(parseTripComplete(reply->readAll())));
    }

    if (reply->url().path() == "/api/trip/search")
    {
        qDebug() << "Reply from trip search";
        emit(tripsLoaded(parseTrips(reply->readAll())));
    }

    if (reply->url().path() == "/api/trip/tracks")
    {
        qDebug() << "Reply from trip tracks";
        parseTracks(reply->readAll());
    }
}

long TrailInterface::parseLogin(const QString &xml)
{
    QDomDocument doc;
    QDomElement response;

    if (doc.setContent(xml))
    {
        response = doc.firstChildElement();

        qDebug() << response.nodeName();

        if (response.nodeName() == "etUserLoginResponse")
        {
            if (response.attribute("status", "error") == "success")
            {
                m_uid = response.firstChild().firstChild().nodeValue().toLong();

            }
            else
            {
                m_uid = response.firstChild().nodeValue().toLong() * -1;
            }
            return m_uid;
        }
    }
    else
    {
        qDebug() << "Error parsing xml";
    }

    return -1;
}

bool TrailInterface::parseActivities(const QString &xml)
{
    QDomDocument doc;
    QDomElement response;


    if (doc.setContent(xml))
    {
        response = doc.firstChildElement();

        qDebug() << response.nodeName();

        if (response.nodeName() == "etApiActivitiesResponse")
        {
            if (response.attribute("status", "error") == "success")
            {
                m_activities.clear();
                for (int i = 0; i < response.firstChild().childNodes().count(); ++i)
                {
                    QDomElement a = response.firstChild().childNodes().at(i).toElement();
                    m_activities[a.attribute("id").toInt()] = a.text();
                }
                qDebug() << m_activities;
                QStringList activities = m_activities.values();
                activities.sort();
                emit gotActivities(activities);
                return true;
            }
        }
    }
    else
    {
        qDebug() << "Error parsing xml";
    }
    return false;
}

bool TrailInterface::parseTrips(const QString &xml)
{
    QDomDocument doc;
    QDomElement response;

    qDebug() << xml;

    if (doc.setContent(xml))
    {
        response = doc.firstChildElement();

        qDebug() << response.nodeName();

        if (response.nodeName() == "etUserTripsResponse" || response.nodeName() == "etTripSearchResponse")
        {
            if (response.attribute("status", "error") == "success")
            {
                QAbstractItemModel::beginResetModel();
                qDeleteAll(m_trips);
                m_trips.clear();
                for (int i = 0; i < response.firstChild().childNodes().count(); ++i)
                {
                    qDebug() << "Got Trip:" << i;
                    qDebug() << response.firstChild().childNodes().at(i).nodeName();
                    m_trips << new TrailTrip(response.firstChild().childNodes().at(i).toElement());
                }
                QAbstractItemModel::endResetModel();
                return true;
            }
        }
    }
    else
    {
        qDebug() << "Error parsing xml";
    }
    return false;
}

QPixmap TrailInterface::getImage(QUrl url)
{
    qDebug() << "Getting image:" << url;
    emit busy(true);
    QNetworkRequest request;
    request.setUrl(url);
    QPixmap pix;

    QNetworkReply *reply = m_networkManager->get(request);

    while(reply->isRunning())
    {
        QApplication::processEvents();
    }

    pix.loadFromData(reply->readAll());

    emit busy(false);
    qDebug() << "Done";
    return pix;
}

void TrailInterface::getTripXML(long trip_id)
{
    qDebug() << "Getting Trip XML:" << trip_id;

    emit busy(true);

    QUrl url("http://www.everytrail.com/api/trip/tracks");

    url.addQueryItem("version", "3");
    url.addQueryItem("username", m_user);
    url.addQueryItem("password", m_pass);
    url.addQueryItem("trip_id", QString::number(trip_id));

    m_networkManager->get(QNetworkRequest(url));
}

bool TrailInterface::parseTracks(const QString& xml)
{
    qDebug() << "Parsing tracks...";

    QDomDocument doc;
    QDomElement response;

    if (doc.setContent(xml))
    {
        response = doc.firstChildElement();

        if (response.nodeName() == "etTripTracksResponse")
        {
            if (response.attribute("status", "error") == "success")
            {
                emit trackLoaded(doc);
                return true;
            } else {
                return false;
            }
        }
    }
    else
    {
        qDebug() << "Error parsing xml";
    }
    return false;

}

QStringList TrailInterface::activities()
{
    return m_activities.values();
}

void TrailInterface::createBasicTrip(const QString &title, const QString &description, const QString &activity, int visibility)
{
    emit busy(true);

    QUrl url("http://www.everytrail.com/api/trip/create");

    url.addQueryItem("version", "3");
    url.addQueryItem("username", m_user);
    url.addQueryItem("password", m_pass);
    url.addQueryItem("title", title);
    url.addQueryItem("description", description);
    url.addQueryItem("activity", activity);
    url.addQueryItem("visibility_id", QString::number(visibility));
    url.addQueryItem("time", QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss"));

    m_networkManager->get(QNetworkRequest(url));
}

long TrailInterface::parseCreateBasicTrip(const QString &xml)
{
    qDebug() << "TrailInterface::parseCreateBasicTrip" << xml;

    QDomDocument doc;
    QDomElement response;

    long trip_id;

    if (doc.setContent(xml))
    {
        response = doc.firstChildElement();

        qDebug() << response.nodeName();

        if (response.nodeName() == "etTripCreateResponse")
        {
            if (response.attribute("status", "error") == "success")
            {
                qDebug() << response.firstChild().toElement().text().toLong();
                trip_id = response.firstChild().toElement().text().toLong();
            }
            else
            {
                trip_id = -1;
            }
            qDebug() << trip_id;
            return trip_id;
        }
    }
    else
    {
        qDebug() << "Error parsing xml";
    }

    return -1;
}

void TrailInterface::tripSetGPX(long trip_id, const QByteArray &gpx)
{
    emit busy(true);

    FormPost *formpost = new FormPost(m_networkManager);
    formpost->addField("version", "3");
    formpost->addField("username", m_user);
    formpost->addField("password", m_pass);
    formpost->addField("trip_id", QString::number(trip_id));
    formpost->addFile("gpx", gpx, "file.gpx", "application/octet-stream");
    formpost->postData("http://www.everytrail.com/api/trip/addgpx");
}

long TrailInterface::parseTripSetGPX(const QString &xml)
{
    qDebug() << "TrailInterface::parseTripSetGPX" << xml;

    QDomDocument doc;
    QDomElement response;

    long trip_id;

    if (doc.setContent(xml))
    {
        response = doc.firstChildElement();

        qDebug() << response.nodeName();

        if (response.nodeName() == "etTripAddgpxResponse")
        {
            if (response.attribute("status", "error") == "success")
            {
                trip_id = response.firstChild().toElement().text().toLong();
            }
            else
            {
                trip_id = -1;
            }
            return trip_id;
        }
    }
    else
    {
        qDebug() << "Error parsing xml";
    }

    return -1;
}

void TrailInterface::tripComplete(long trip_id)
{
    emit busy(true);

    QUrl url("http://www.everytrail.com/api/trip/complete");

    url.addQueryItem("version", "3");
    url.addQueryItem("username", m_user);
    url.addQueryItem("password", m_pass);
    url.addQueryItem("trip_id", QString::number(trip_id));

    m_networkManager->get(QNetworkRequest(url));
}

long TrailInterface::parseTripComplete(const QString &xml)
{
    qDebug() << "TrailInterface::parseTripComplete" << xml;

    QDomDocument doc;
    QDomElement response;

    long trip_id;

    if (doc.setContent(xml))
    {
        response = doc.firstChildElement();

        qDebug() << response.nodeName();

        if (response.nodeName() == "etTripCompleteResponse")
        {
            if (response.attribute("status", "error") == "success")
            {
                trip_id = response.firstChild().toElement().text().toLong();
            }
            else
            {
                trip_id = -1;
            }
            return trip_id;
        }
    }
    else
    {
        qDebug() << "Error parsing xml";
    }

    return -1;
}

void TrailInterface::tripSearch(bool searchTerm, bool searchProximity, const QString &term, const QGeoPositionInfo& pos, const QString proximity)
{
    if (!searchTerm && !searchProximity) {
        //nothing to search for so just return
        return;
    }
    emit busy(true);

    QUrl url("http://www.everytrail.com/api/trip/search");

    url.addQueryItem("version", "3");
    if (searchTerm){
        url.addQueryItem("q", term);
    }

    if (searchProximity) {
        url.addQueryItem("lat", QString::number(pos.coordinate().latitude()));
        url.addQueryItem("lon", QString::number(pos.coordinate().longitude()));
        url.addQueryItem("proximitiy", proximity);
    }

    m_networkManager->get(QNetworkRequest(url));
}

int TrailInterface::rowCount(const QModelIndex &parent) const
{
    return m_trips.count();
}

QVariant TrailInterface::data(const QModelIndex &index, int role) const
{
    if (index.row() < m_trips.count()) {
        if (role == Qt::DisplayRole) {
            return m_trips[index.row()]->tripName();
        } else if (role == Qt::UserRole + 1) {
            return m_trips[index.row()]->activity();
        } else if (role == Qt::UserRole + 2) {
            return m_trips[index.row()]->durationString();
        }
    }

    return QVariant();
}
