#include "TrailCreateTrip.h"
#include <QPushButton>
#include <QComboBox>
#include <QVBoxLayout>
#include <QLabel>
#include <QFileDialog>
#include <QFileInfo>
#include <QLineEdit>
#include <QCheckBox>
#include <QPlainTextEdit>
#include <QMessageBox>
#include <QDebug>

#include "trailinterface.h"

TrailCreateTrip::TrailCreateTrip(TrailInterface *interface, QWidget *parent) :
    QWidget(parent)
{
    m_interface = interface;
    m_layout = new QVBoxLayout(this);

    m_fileButton = new QPushButton(tr("Select GPX file..."), this);
    connect(m_fileButton, SIGNAL(clicked()), this, SLOT(selectFile()));
    m_layout->addWidget(m_fileButton);

    m_title = new QLineEdit(this);
    m_layout->addWidget(new QLabel(tr("Title")));
    m_layout->addWidget(m_title);

    m_description = new QPlainTextEdit(this);
    m_layout->addWidget(new QLabel(tr("Description")));
    m_layout->addWidget(m_description);

    m_activityCombo = new QComboBox(this);
    m_layout->addWidget(new QLabel(tr("Activity")));
    m_layout->addWidget(m_activityCombo);

    m_public = new QCheckBox(tr("Public"), this);
    m_public->setCheckState(Qt::Checked);
    m_layout->addWidget(new QLabel(tr("Visibility")));
    m_layout->addWidget(m_public);

    m_createButton = new QPushButton(tr("Create Trip"), this);
    connect(m_createButton, SIGNAL(clicked()), this, SLOT(createTrip()));
    m_layout->addWidget(m_createButton);

    m_layout->addStretch();

    connect(m_interface, SIGNAL(gotActivities(QStringList)), this, SLOT(setActivities(QStringList)));
    connect(m_interface, SIGNAL(createBasicTripDone(long)), this, SLOT(createdBasicTrip(long)));
    connect(m_interface, SIGNAL(tripSetGPXDone(long)), this, SLOT(tripSetGPXDone(long)));
    connect(m_interface, SIGNAL(tripCompleteDone(long)), this, SLOT(tripCompleteDone(long)));
}

TrailCreateTrip::~TrailCreateTrip()
{


}

void TrailCreateTrip::selectFile()
{
    m_fileName = QFileDialog::getOpenFileName(this, "Select file", QString(), "GPX (*.gpx)");
    if (!m_fileName.isEmpty()) {
        QFileInfo f(m_fileName);
        m_fileButton->setText(f.fileName());
    }
}

void TrailCreateTrip::setActivities(const QStringList &activities)
{
    m_activityCombo->clear();
    m_activityCombo->addItems(activities);
}

void TrailCreateTrip::createTrip()
{
    if (m_fileName.isEmpty()) {
        QMessageBox::information(this, tr("No GPX file selected"), tr("You must select a GPX file to create a trip"));
        return;
    }
    if (m_title->text().isEmpty()) {
        QMessageBox::information(this, tr("No title entered"), tr("You must enter a title to create a trip"));
        return;
    }

/*
    QFile gpx(m_fileName);

    if (gpx.open(QIODevice::ReadOnly))  {
        QByteArray ba = gpx.readAll();

        qDebug() << ba.toBase64();

        m_interface->tripSetGPX();
        gpx.close();
    }

    return;
*/
    m_interface->createBasicTrip(m_title->text(), m_description->toPlainText(), m_activityCombo->currentText(), (m_public->checkState() == Qt::Checked  ? 1 : 0));

}

void TrailCreateTrip::createdBasicTrip(long trip_id)
{
    qDebug() << "..." << trip_id;
    if (trip_id <= 0) {
        QMessageBox::information(this, tr("Error"), tr("There was an error creating the trip"));
        return;
    }

    QFile gpx(m_fileName);

    if (gpx.open(QIODevice::ReadOnly))  {
        QByteArray ba = gpx.readAll();

        m_interface->tripSetGPX(trip_id, ba);

        //gpx.close();
    }
}

void TrailCreateTrip::tripSetGPXDone(long trip_id)
{
    if (trip_id <= 0) {
        QMessageBox::information(this, tr("Error"), tr("There was an error setting the trip GPX track"));
        return;
    }

    m_interface->tripComplete(trip_id);
}

void TrailCreateTrip::tripCompleteDone(long trip_id)
{
    if (trip_id <= 0) {
        QMessageBox::information(this, tr("Error"), tr("There was an error completing the trip"));
        return;
    }

    QMessageBox::information(this, tr("Success"), tr("The trip was created successfully"));
}
