#ifndef CAMERATHREAD_H
#define CAMERATHREAD_H

#include "cameraoverlay.h"
#include "assert.h"
#include "math.h"
#include <FCam/N900.h>
#include <FCam/Event.h>

#include <QThread>
#include <QMutex>
#include <QImage>

//! CameraThread class.
/*!
    This class creates the thread used by FCam to control the camera.
    It uses FCam auto-expose, auto-white-balance and auto-focus routines.
    It calculates sharness level of each frame and if given requirements are met,
    it saves the frame for OCR processing.
*/

class CameraThread : public QThread
{
    Q_OBJECT;

public:
    //! CameraThread constructor
    CameraThread(CameraOverlay *o, QObject *parent = NULL) : QThread(parent)
    {
        overlay = o;
        keepGoing = true;
        //Use ocr
        // ocr = new OCR();
        //connect(this->ocr, SIGNAL(onOCRDone(QString)), this, SLOT(onOCRDone(QString)));
        //connect(this->ocr, SIGNAL(onOCRError(QString)), this, SLOT(onOCRError(QString)));
    }

    //! Static utility function that tests whether the lens cover is currently shut
    static bool lensCovered();

public slots:
    //! Pause camera thread
    void pause()
    {
        //! Order the sensor to stop streaming
        sensor.stopStreaming();
    }

    //! Shut down camera thread
    void stop()
    {
        keepGoing = false;
    }

    //! Allow the camera thread to return. This _must_ be called after handling a panic() signal
    void failGracefully()
    {
        exitLock.unlock();
    }

signals:
    //! Emit cover state
    void lensCoverClosed();
    void lensCoverOpened();

    //! An irrecoverable error has occured
    void panic(FCam::Event event);
    
protected:
    //! Start camera thread
    void run();

private:
    //! An FCam sensor and lens
    FCam::N900::Sensor sensor;
    FCam::N900::Lens lens;

    FCam::N900::Frame frame;
    FCam::Image image;

    //! A shot to represent the viewfinder
    FCam::Shot viewfinder;

    //! A corresponding shot
    FCam::Shot shot;

    //! The camera thread checks this flag once per iteration. If it's false, it terminates.
    bool keepGoing;
    //! Flag for OCR
    bool ocrActive;

    //! A pointer to the overlay, so we can use its framebuffer as a memory destination for viewfinding.
    CameraOverlay *overlay;
    
    //! A lock that prevents the camera thread from exiting before showing an error message
    QMutex exitLock;
    //OCR instance
   // OCR *ocr;
private slots:
    //! Handle OCR processing done
    void onOCRDone(QString st);
    //! Handle OCR processing error
    void onOCRError(QString st);
};

#endif // CAMERATHREAD_H
