#define CTF_LOG_FORMAT_USE_TIME
#define CTF_THROW_EXIT

#include "../CTF/src/ctf.h"
#include <kernel/general.h>
#include <msa_module.h>
#include <gconf/gconf-client.h>
#include <glib.h>
#include "kernel_test_func.h"
#include "kernel/kernel.h"
#include "kernel/handler.h"

GLOBAL_DECL();

CTF_TEST_DATA_FILE_NAME_DECL;
CTF_STAT_FILE_NAME_DECL;


CTF_TEST_CASE(kernel_put_data_test) {

    CTF_TEST_NAME("kernel_put_data_test");

    xmlDocPtr reqXml;
    int repository_length = 0;

    kernel_initialization();
    reqXml = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><TargetID>xxxvkontakte</TargetID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");

    kernel_put_data(reqXml);
    repository_length = repository_get_length();
    CTF_CHECK(repository_length == 1);

    reqXml = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>db</SourceID><TargetID>xxx</TargetID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");

    kernel_put_data(reqXml);

    repository_length = repository_get_length();
    CTF_CHECK(repository_length == 2);


    CTF_TEST_NAME("repository_get_oldest_test");

    transit_data* tr_data;
    tr_data = repository_get_oldest_transit_data(0);

    CTF_CHECK(!strcmp(tr_data->source_id, "ui"));


    CTF_TEST_NAME("repository_get_newest_test");
    transit_data* tr_data2;
    tr_data2 = repository_get_newest_transit_data(0);
    
    CTF_CHECK(!strcmp(tr_data2->source_id, "db"));


    CTF_TEST_NAME("repository_free_test");

    repository_free();
    repository_length = repository_get_length();
    CTF_CHECK(repository_length == 0);
}


/* Status for transit data structure, sets for each target */
typedef enum process_status {
    NOT_SEND = 0,       // Message not send yet.
    IN_PROCESS = 2,     // Message is processed by "target". 
    PROCESSED = 4,      // Message processed by "target".
    FORCE_REMOVE = 8,   // Remove "target" from send message process. 
    IGNORE = 16         // Ignore send message for "target'.
} process_status;


 typedef struct targets {
    xmlChar* name;          // Target name.
    gint order_number;      // Order number in query.
    process_status status;  // Target status.
    guint attempts_count;   // Count of send message. 
} target_data;


CTF_TEST_CASE(convert_to_kernel_format_test) {

    CTF_TEST_NAME("convert_to_kernel_format_test");

    xmlDocPtr test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><TargetID>Target1</TargetID><TargetID>Target2</TargetID><TargetID>Target3</TargetID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");

/*
  xmlChar* id;
    xmlChar* source_id;
    GList* targets; 
    xmlDocPtr content;
    
     xmlChar* name;          // Target name.
    gint order_number;      // Order number in query.
    process_status status;  // Target status.
    guint attempts_count;
*/
    transit_data* tr_data = convert_to_kernel_format_call(test_data);

    CTF_CHECK(!strcmp(tr_data->source_id, "ui"));
    
    CTF_TEST_NAME("check_targets_after_convert");
    target_data* target = (target_data*)g_list_nth_data(tr_data->targets, 0);
    CTF_CHECK(!strcmp(target->name, "Target1"));
    CTF_CHECK((target->order_number == -1));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0));    
   
    target = (target_data*)g_list_nth_data(tr_data->targets, 1);
    CTF_CHECK(!strcmp(target->name, "Target2"));
    CTF_CHECK((target->order_number == -1));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0));    
   
    target = (target_data*)g_list_nth_data(tr_data->targets, 2);
    CTF_CHECK(!strcmp(target->name, "Target3"));
    CTF_CHECK((target->order_number == -1));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0));    
   
    CTF_TEST_NAME("check_targets_count_after_convert");
    CTF_CHECK(g_list_length(tr_data->targets) == 3);
   
   
    CTF_TEST_NAME("convert_to_kernel_format_test(no targets)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");

    tr_data = convert_to_kernel_format_call(test_data);

    CTF_CHECK(tr_data == NULL);
    
     CTF_TEST_NAME("convert_to_kernel_format_test(no source)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><TargetID>Target1</TargetID><TargetID>Target2</TargetID><TargetID>Target3</TargetID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");

    tr_data = convert_to_kernel_format_call(test_data);

    CTF_CHECK(tr_data == NULL);
    
    
     CTF_TEST_NAME("convert_to_kernel_format_test(no targets, no source)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");

    tr_data = convert_to_kernel_format_call(test_data);

    CTF_CHECK(tr_data == NULL);
    
    
    CTF_TEST_NAME("convert_to_kernel_format_test(incorrect doc)");
    
    test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransiData><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransiData>");

    tr_data = convert_to_kernel_format_call(test_data);

    CTF_CHECK(tr_data == NULL);
   
   
    CTF_TEST_NAME("convert_to_kernel_format_test(doc == NULL)");
    
    tr_data = convert_to_kernel_format_call(NULL);

    CTF_CHECK(tr_data == NULL);
   
}



CTF_TEST_CASE(add_target_test) {

    CTF_TEST_NAME("add_target_test");

    xmlDocPtr test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");
    
    add_target(test_data, "Target1", 1);
    add_target(test_data, "Target2", 2);

    GList* targets = get_targets_call(test_data);
    
    target_data* target = (target_data*)g_list_nth_data(targets, 0);
    CTF_CHECK(!strcmp(target->name, "Target1"));
    CTF_CHECK((target->order_number == 1));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0)); 

    target = (target_data*)g_list_nth_data(targets, 1);
    CTF_CHECK(!strcmp(target->name, "Target2"));
    CTF_CHECK((target->order_number == 2));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0)); 

    CTF_TEST_NAME("check_targets_count");
    CTF_CHECK(g_list_length(targets) == 2);




   CTF_TEST_NAME("add_target_test(incorrect order number)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");
    
    add_target(test_data, "Target1", -1);

    targets = get_targets_call(test_data);
    
    target = (target_data*)g_list_nth_data(targets, 0);
    CTF_CHECK(!strcmp(target->name, "Target1"));
    CTF_CHECK((target->order_number == -1));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0)); 

    CTF_TEST_NAME("check_targets_count");
    CTF_CHECK(g_list_length(targets) == 1);


    CTF_TEST_NAME("add_target_test(incorrect target)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");
    
    add_target(test_data, "", -1);
    add_target(test_data, NULL, -1);

    targets = get_targets_call(test_data);
    
    CTF_CHECK(g_list_length(targets) == 0);


    CTF_TEST_NAME("add_target_test(incorrect doc)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?>");
    
    add_target(NULL, "Target2", 1);
    add_target(test_data, "Target1", 0);

    targets = get_targets_call(test_data);
    
    CTF_CHECK(g_list_length(targets) == 0);
   
}





CTF_TEST_CASE(get_targets_test) {

    CTF_TEST_NAME("get_targets_test");

    xmlDocPtr test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><TargetID>Target1</TargetID><TargetID>Target2</TargetID><TargetID>Target3</TargetID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");    
    
    GList* targets = get_targets_call(test_data);
    
    target_data* target = (target_data*)g_list_nth_data(targets, 0);
    CTF_CHECK(!strcmp(target->name, "Target1"));
    CTF_CHECK((target->order_number == -1));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0)); 

    target = (target_data*)g_list_nth_data(targets, 1);
    CTF_CHECK(!strcmp(target->name, "Target2"));
    CTF_CHECK((target->order_number == -1));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0)); 

    target = (target_data*)g_list_nth_data(targets, 2);
    CTF_CHECK(!strcmp(target->name, "Target3"));
    CTF_CHECK((target->order_number == -1));    
    CTF_CHECK((target->status == NOT_SEND));    
    CTF_CHECK((target->attempts_count == 0)); 

    CTF_TEST_NAME("check_targets_count");
    CTF_CHECK(g_list_length(targets) == 3);


    CTF_TEST_NAME("add_target_test(incorrect target)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");
    
    targets = get_targets_call(test_data);

    CTF_CHECK(targets == NULL);
    CTF_CHECK(g_list_length(targets) == 0);

    CTF_TEST_NAME("add_target_test(incorrect doc)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?>");

    targets = get_targets_call(test_data);

    CTF_CHECK(targets == NULL);
    CTF_CHECK(g_list_length(targets) == 0);
    
    targets = get_targets_call(NULL);

    CTF_CHECK(targets == NULL);
    CTF_CHECK(g_list_length(targets) == 0);

   
}


CTF_TEST_CASE(get_source_test) {

    CTF_TEST_NAME("get_source_test");

    xmlDocPtr test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><SourceID>ui</SourceID><TargetID>Target1</TargetID><TargetID>Target2</TargetID><TargetID>Target3</TargetID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");    
    
    xmlChar* source = get_source_call(test_data);

    CTF_CHECK(!strcmp(source, "ui"));    
    
    
    CTF_TEST_NAME("get_source_test(no source)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?><TransitData><TargetID>Target1</TargetID><TargetID>Target2</TargetID><TargetID>Target3</TargetID><Content><Request class=\"profile\" function=\"getProfile\"><Params></Params></Request></Content></TransitData>");    
    
    source = get_source_call(test_data);

    CTF_CHECK(source == NULL);
    
    
    CTF_TEST_NAME("get_source_test(incorrect doc)");

    test_data = xmlParseDoc("<?xml version=\"1.0\"?><root />");    
    
    source = get_source_call(test_data);

    CTF_CHECK(source == NULL);
    
    CTF_TEST_NAME("get_source_test(doc == NULL)");
    source = get_source_call(NULL);
    CTF_CHECK(source == NULL);
   
}   


