
/*
 * jabber.c.
 * This file is part of MSA project.
 *
 * Copyright (C) 2009 - Alexander A. Lomov.
 *
 * MSA project is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA project is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with <program name>; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <glib.h>
//#include <glib/gprintf.h>
#include <stdlib.h>
#include <string.h>
//#include <loudmouth-1.0/loudmouth/lm-connection.h>
#include <loudmouth/loudmouth.h>
#include <glib-2.0/glib/gmacros.h>
#include <glib-2.0/glib/gmain.h>
#include <loudmouth-1.0/loudmouth/lm-message-node.h>
#include <libxml2/libxml/xmlstring.h>
//#include <loudmouth-1.0/loudmouth/lm-proxy.h>
#include "jabber.h"
#include "debug.h"
#include "jabber_requests.h"
#include "jabber_general.h"

#define JABBER_TYPE_ATTR "type"
#define JABBER_ERROR_CODE_ATTR "code"
#define JABBER_ERROR_TYPE_ATTR "type"
#define JABBER_GET_VALUE "get"
#define JABBER_TYPE_ERROR_VALUE "error"
#define JABBER_QUERY_NODE_NAME "query"
#define JABBER_ERROR_NODE_NAME "error"

#define JABBER_IQ_REGISTER_NS "jabber:iq:register"

#define AT "@"
#define JABBER_RESOURCE "msa"

#define DRV_DEBUG_PREFIX "JB_CLIENT:"


static GMainLoop* main_loop = NULL;
static GMainContext* main_context = NULL;
typedef struct _jabber_context {
    LmConnection* connection;
    LmProxy* proxy;
    gboolean is_proxy_changed;
    gboolean use_proxy;
    gchar* jid;
    gchar* resource;
    gchar* username;
    gchar* host;
    gchar* password;
} jabber_drv_context;



////////////////////////////////////////////////////////////////////////////////
/*Some functions for console client, for fast debug. They must be killed later*/


////////////////////////////////////////////////////////////////////////////////


typedef struct jabber_error {
    gint code;
    gchar* type;
    gchar* message;
} jabber_error;



static jabber_drv_context jb_context;
static jabber_error last_error;

static jabber_error* new_jabber_error(gint code, gchar* type, gchar* message)
{
    jabber_error* error = g_try_new(jabber_error, 1);

    if (error == NULL) {
        return NULL;
    }

    error->code = code;
    error->type = type;
    error->message = message;

    return error;
}

static void free_jabber_error(jabber_error* error) {
    if (error == NULL) {
        return;
    }

    g_free(error->message);
    g_free(error->type);

    error->message = NULL;
    error->type = NULL;
    error->code = 0;

    g_free(error);
}


static gchar* get_error_text(gint code) {

    gchar* error_text = NULL;
    switch (code) {
        case JABBER_CANT_AUTH_CONNECT: {
            error_text = g_strdup(JABBER_CANT_AUTH_CONNECT_TEXT);
            break;
        }
        case JABBER_CANT_CONNETC: {
            error_text = g_strdup(JABBER_CANT_CONNETC_TEXT);
            break;
        }
    }
    return error_text;
}

static void set_error(gint code) {

    last_error.code = 0;
    g_free(last_error.message);
    g_free(last_error.type);

    last_error.message = NULL;
    last_error.type = NULL;

    if (code == 0) {
        return;
    }

    last_error.code = code;
    last_error.message = get_error_text(code);
}

jabber_error get_last_error()
{
    return last_error;
}

gchar* jabber_get_jid_with_resource()
{
    gchar* jid_r = g_strconcat(jb_context.jid, "/", jb_context.resource, NULL);
    return jid_r;
}

void jabber_init()
{
    DRIVER_DEBUG(3)("%s jabber_init START", DRV_DEBUG_PREFIX);
    jb_context.is_proxy_changed = FALSE;
    jb_context.connection = NULL;
    jb_context.host = NULL;
    jb_context.jid = NULL;
    jb_context.password = NULL;
    jb_context.proxy = NULL;
    jb_context.use_proxy = TRUE;
    main_loop = g_main_loop_new(NULL, FALSE);
    main_context = g_main_context_new();

    last_error.code = 0;
    last_error.message = NULL;
    last_error.type = NULL;

    DRIVER_DEBUG(3)("%s jabber_init END", DRV_DEBUG_PREFIX);
}



gint jabber_shtd()
{
    DRIVER_DEBUG(3)("%s jabber_shtd START", DRV_DEBUG_PREFIX);
    if (jb_context.connection != NULL) {
        lm_connection_close(jb_context.connection, NULL);
        lm_connection_unref(jb_context.connection);
        DRIVER_MESSAGE(7)("%s Connection closed", DRV_DEBUG_PREFIX);
    }
    g_free(jb_context.host);
    g_free(jb_context.jid);
    g_free(jb_context.resource);
    g_free(jb_context.password);

    if (jb_context.proxy != NULL) {
        lm_proxy_unref(jb_context.proxy);
    }
    g_main_loop_quit(main_loop);
    g_main_loop_unref(main_loop);
    DRIVER_DEBUG(3)("%s jabber_shtd END", DRV_DEBUG_PREFIX);
    return 0;
}

gint jabber_set_user_context(const gchar* jid_resource, const gchar* password)
{
    gchar** user_data = g_strsplit_set(jid_resource, "/", 2);
    gint len = g_strv_length(user_data);
    
    if (len == 0 || password == NULL) {
        g_strfreev(user_data);
        return -1;
    }

    jb_context.jid = g_strdup(user_data[0]);

    if (user_data[1] == NULL) {
        jb_context.resource = g_strdup(JABBER_RESOURCE);
    } else {
        jb_context.resource = g_strdup(user_data[1]);
    }
    g_strfreev(user_data);

    user_data = g_strsplit_set(jb_context.jid, AT, 2);
    
    if (user_data[0] != NULL) {
        jb_context.username = user_data[0];
        jb_context.host = user_data[1];
        user_data[0] = NULL;
        user_data[1] = NULL;
    }
    jb_context.password = g_strdup(password);
    g_strfreev(user_data);

    DRIVER_MESSAGE(5)("%s Set user content: jid = %s; Host = %s; Resource = %s", \
    DRV_DEBUG_PREFIX, jb_context.jid, jb_context.host, jb_context.resource);
    return 0;
}

void jabber_use_proxy(gboolean use)
{
    jb_context.use_proxy = use;
}

static gboolean is_proxy_changed(const gchar* host, const guint port)
{

    if (host != NULL) {
        if (strlen(host) < 3)
            return FALSE;
    }
    
    if (jb_context.proxy == NULL && host != NULL) {
        return TRUE;
    }

    if (jb_context.proxy != NULL && (host == NULL || port <= 0)){
        return TRUE;
    }

    const gchar* proxy_host = lm_proxy_get_server(jb_context.proxy);
    const guint proxy_port = lm_proxy_get_port(jb_context.proxy);

    if (port != proxy_port) {
        return TRUE;
    }

    if (strcmp(host, proxy_host) != 0) {
        return TRUE;
    }
    return FALSE;
}

gint jabber_set_proxy(const gchar* host, const gint port)
{
    DRIVER_DEBUG(5)("%s jabber_set_proxy START (host = %s, port = %i)", \
            DRV_DEBUG_PREFIX, host, port);

    if (is_proxy_changed(host, port) == FALSE) {
        jb_context.is_proxy_changed = FALSE;
        DRIVER_MESSAGE(5)("%s Proxy not changed END", DRV_DEBUG_PREFIX);
        return 0;
    }

    if (jb_context.proxy != NULL) {
        lm_proxy_unref(jb_context.proxy);
    }

    if (host == NULL || port == 0) {
        jb_context.proxy = NULL;
    } else {
        jb_context.proxy = lm_proxy_new_with_server(LM_PROXY_TYPE_HTTP, host, (guint)port);
    }
    jb_context.is_proxy_changed = TRUE;

    DRIVER_DEBUG(5)("%s jabber_set_proxy END", DRV_DEBUG_PREFIX);
    return 0;
}

const gchar* jabber_get_proxy_host()
{
    if (jb_context.proxy == NULL) {
        return NULL;
    }

    return lm_proxy_get_server(jb_context.proxy);
}

guint jabber_get_proxy_port()
{
    if (jb_context.proxy == NULL) {
        return 0;
    }

    return lm_proxy_get_port(jb_context.proxy);
}




static gboolean connect()
{
    if (jb_context.connection == NULL) {
        jb_context.connection = lm_connection_new_with_context(jb_context.host, main_context);
    }

    if (jb_context.use_proxy == TRUE && jb_context.proxy != NULL) {
        DRIVER_MESSAGE(7)("%s Use proxy: host = %s, port = %i", DRV_DEBUG_PREFIX,\
                lm_proxy_get_server(jb_context.proxy), lm_proxy_get_port(jb_context.proxy));
        lm_connection_set_proxy(jb_context.connection, jb_context.proxy);
    } else {
        lm_connection_set_proxy(jb_context.connection, NULL);
    }
    jb_context.is_proxy_changed = FALSE;

    GError* error = NULL;
    if (lm_connection_is_open(jb_context.connection) == FALSE) {
        if (lm_connection_open_and_block(jb_context.connection, &error) == FALSE) {
            //TODO: Fucking connect!
            // Fuck, open source... Sometimes function open connection, but return error.
            // This just for check: error is real or not....
            if (lm_connection_is_open(jb_context.connection) == FALSE) {
                jb_context.connection = NULL;
                return FALSE;    
            }
            // lm_connection_unref(jb_context.connection);
        }
    }

    return TRUE;
}

static gint auth_jabber_connect()
{
    DRIVER_DEBUG(5)("%s auth_jabber_connect START", DRV_DEBUG_PREFIX);
    if (jb_context.connection == NULL) {
        DRIVER_MESSAGE(7)("%s Connect not create", DRV_DEBUG_PREFIX);
        return JABBER_CONNECT_NOT_OPEN;
    }

    if (lm_connection_is_open(jb_context.connection) == FALSE) {
        DRIVER_MESSAGE(7)("%s Connect not opened END", DRV_DEBUG_PREFIX);
        return JABBER_CONNECT_NOT_ESTABLISHED;
    }
    
    if (lm_connection_authenticate_and_block(jb_context.connection, jb_context.username,
            jb_context.password, jb_context.resource, NULL) == FALSE) {
        DRIVER_MESSAGE(7)("%s Can't auth connect END", DRV_DEBUG_PREFIX);
        return JABBER_CANT_AUTH_CONNECT;
    }

    lm_connection_send_raw(jb_context.connection, "<presence/>", NULL);
    DRIVER_DEBUG(5)("%s auth_jabber_connect END", DRV_DEBUG_PREFIX);
    return 0;
}


gint jabber_connect()
{
    DRIVER_DEBUG(5)("%s jabber_connect START", DRV_DEBUG_PREFIX);
    if (jb_context.connection != NULL) {
        
        if (lm_connection_is_open(jb_context.connection) == TRUE) {
            DRIVER_MESSAGE(7)("%s Connection allready opened END", DRV_DEBUG_PREFIX);
            return auth_jabber_connect();//JABBER_ALLREADY_CONNECT;
        }
    }

    if (connect() != TRUE) {
        DRIVER_MESSAGE(7)("%s Can't connect to server: %s END", DRV_DEBUG_PREFIX, jb_context.host);
        return JABBER_CANT_CONNETC;
    }

    DRIVER_DEBUG(5)("%s jabber_connect END", DRV_DEBUG_PREFIX);
    return auth_jabber_connect();
}


gint jabber_reconnect()
{
    DRIVER_DEBUG(5)("%s jabber_reconnect START", DRV_DEBUG_PREFIX);
    if (jb_context.connection != NULL) {
        if (lm_connection_is_open(jb_context.connection) == TRUE) {
               lm_connection_close(jb_context.connection, NULL);
               DRIVER_MESSAGE(5)("%s Close old connection", DRV_DEBUG_PREFIX);
        }
      //  lm_connection_unref(jb_context.connection);
      //  jb_context.connection == NULL;
    }

    //jb_context.is_proxy_changed == TRUE;
    DRIVER_DEBUG(5)("%s jabber_reconnect END", DRV_DEBUG_PREFIX);
    return jabber_connect();
}




gboolean is_jabber_connect()
{
    if (jb_context.connection == NULL) {
        return FALSE;
    }

    return lm_connection_is_open(jb_context.connection);
}



/**
 * @brief Check the message is an error or not.
 *
 * Try find "type" attribute and check equals it an "error".
 *
 * @param message message for check.
 *
 * @retrn TRUE if message is error, or FALSE otherwise.
 */
static gboolean is_message_error(LmMessage* message) 
{
    const gchar* value = lm_message_node_get_attribute(message->node, 
            JABBER_TYPE_ATTR);
    
    if (value == NULL) {
        return FALSE;
    } else if (strncmp(value, JABBER_TYPE_ERROR_VALUE, 
            strlen(JABBER_TYPE_ERROR_VALUE)) == 0) {
        return TRUE;
    }
    
    return TRUE;
}



static jabber_error* get_error_from_message(LmMessage* message) {
    if(message == NULL) {
        return NULL;
    }
    
    LmMessageNode* error_node = lm_message_node_get_child(message->node, 
            JABBER_ERROR_NODE_NAME);    
    
    if (error_node == NULL) {
        return NULL;
    }

    const gchar* text_val = lm_message_node_get_attribute(error_node, 
            JABBER_ERROR_CODE_ATTR); 
    gint code = atoi(text_val);
    
    text_val = lm_message_node_get_attribute(error_node, JABBER_ERROR_TYPE_ATTR);
    gchar* type = g_strdup(text_val);
    
    text_val = lm_message_node_get_value(error_node);
    gchar* error_message = g_strdup(text_val);
    
    return new_jabber_error(code, type, error_message);
}



/**
 * @brief Receives data for registartion.
 *
 * Receives registartion data from server using given connection. 
 *
 * @param connection open connection to server.
 *
 * @return result message on success or NULL otherwise.
 */
static LmMessage* get_register_data(LmConnection* connection) {
    if (lm_connection_is_open(connection) == FALSE) {    
        return NULL;
    }

    LmMessage* message = lm_message_new(NULL, LM_MESSAGE_TYPE_IQ) ;
    
    LmMessageNode* node = lm_message_node_add_child(message->node,
            JABBER_QUERY_NODE_NAME, NULL);
    
    lm_message_node_set_attribute(node, XMLNS_ATTR, JABBER_IQ_REGISTER_NS);
    
    return lm_connection_send_with_reply_and_block(connection, message, NULL);
}


LmHandlerResult handler_get_message(LmMessageHandler* handler,
        LmConnection* connection, LmMessage* message, gpointer user_data) {
    
    g_debug("IN Handler");
    g_debug("M2: %s\n",  lm_message_node_to_string(lm_message_get_node(message)));
    
    return LM_HANDLER_RESULT_ALLOW_MORE_HANDLERS;
}


int
main1() {
    
    LmProxy* proxy = lm_proxy_new_with_server(LM_PROXY_TYPE_HTTP, "127.0.0.1", 8080);
    
    GError       *error = NULL;
    LmMessage    *m;
    LmConnection* connection = lm_connection_new("jabber.org");
     //    lm_connection_set_jid (connection, "lomalan@infos.infos.ru");
    //    g_debug("PORT: %i", lm_connection_get_port(connection));
    
    lm_connection_set_proxy(connection, proxy);
    //lm_connection_set_port(connection, 80);
    
    
    if (!lm_connection_open_and_block(connection, &error)) {
        g_error("!Failed to open: %s\n", error->message);
    }
//    LmMessage * m21 = lm_message_new("jabber.infos.ru", LM_MESSAGE_TYPE_IQ);
    get_register_data(connection);
    g_debug("out2");
    
    
    
    
    if (!lm_connection_authenticate_and_block(connection,
            "lomalan", "B1`7nh367s_=-a",
            "test",
            &error)) {
        g_error("Failed to authenticate: %s\n", error->message);
    }
    get_register_data(connection);
    g_debug("out1");

    LmMessageHandler*  handler = lm_message_handler_new(handler_get_message,
            NULL,
            NULL);
    
    
    
    m = lm_message_new("lomalan@gmail.com", LM_MESSAGE_TYPE_MESSAGE);
    lm_message_node_add_child(m->node, "body", "Test message!!!");
    if (!lm_connection_send(connection, m, &error)) {
        g_error("Send failed: %s\n", error->message);
    }
    g_debug("out %i, %s, \n%s", lm_connection_is_authenticated(connection),
            lm_connection_get_jid(connection), lm_message_node_to_string  ( lm_message_get_node(m) ) );
    
//	  lm_connection_send_raw(connection, "<message to=\"lomalan@gmail.com\" id=\"42911440286\"> <body>Test message2!!!</body></message>", NULL);
    lm_connection_send_raw(connection, "<presence/>", NULL);
    // lm_connection_send_raw(connection,
    //         "<iq  type='set'> <vCard xmlns='vcard-temp'><NICKNAME>KAIN!!!</NICKNAME></vCard></iq>", NULL);
    
    LmMessage * m2 = lm_message_new("jabber.infos.ru", LM_MESSAGE_TYPE_IQ);
//	 	 lm_message_node_set_attribute(m2->node, "from", "lomalan@jabber.infos.ru/test");
    lm_message_node_set_attribute(m2->node, "type", "get");
    
    LmMessageNode *  node = lm_message_node_add_child(m2->node, "query", NULL);
    lm_message_node_set_attribute(node, "xmlns", "jabber:iq:register");
    g_debug("M2: %s\n",  lm_message_node_to_string  ( lm_message_get_node(m2) ) );
    
    LmMessage * r_m2 = lm_connection_send_with_reply_and_block(connection, m2, NULL);
    
    if(m2!=NULL){
        g_debug("M2_rep: %s\n",  lm_message_node_to_string  ( lm_message_get_node(r_m2) ) );
    }
    
    lm_message_unref(m2);
    lm_message_unref(m);
    lm_connection_close(connection, NULL);
    lm_connection_unref(connection);
    lm_proxy_unref(proxy);
    return 0;
}





static LmConnection* get_connection()
{
    gint result = 0;
    LmConnection* conn = jb_context.connection;
    if (conn == NULL) {
        jb_context.is_proxy_changed = TRUE;
        DRIVER_MESSAGE(7)("%s Get new connection.", DRV_DEBUG_PREFIX);
        result = jabber_connect();
    } else if (lm_connection_is_open(conn) == FALSE
            || lm_connection_is_authenticated(conn) == FALSE
            || jb_context.is_proxy_changed == TRUE) {
        DRIVER_MESSAGE(7)("%s Try reconnect. (is open = %i, is auth = %i, proxy = %i)", \
                DRV_DEBUG_PREFIX, lm_connection_is_open(conn), lm_connection_is_authenticated(conn), jb_context.is_proxy_changed == TRUE);
        result = jabber_reconnect();
    }

    if (result == JABBER_SUCCESS) {
            return jb_context.connection;
    }
    DRIVER_DEBUG(5)("%s Get connection faild with code: %i", DRV_DEBUG_PREFIX, result);
    return NULL;
}

xmlChar* jabber_get_own_profile()
{
    DRIVER_DEBUG(5)("%s jabber_get_own_profile START", DRV_DEBUG_PREFIX);
    LmConnection* connection = get_connection();
    if (connection == NULL) {
        DRIVER_DEBUG(5)("%s Can't get profile, connection is not setup", DRV_DEBUG_PREFIX);
        return NULL;
    }
    DRIVER_MESSAGE(7)("%s Get vcard from server.", DRV_DEBUG_PREFIX);
    LmMessage* message = build_get_vcard_request(NULL, NULL);
    LmMessage* reply_message = 
            lm_connection_send_with_reply_and_block(connection, message, NULL);

    LmMessageNode* node = lm_message_get_node(reply_message);

    gchar* reply_text = lm_message_node_to_string(node);
    
    DRIVER_MESSAGE(10)("%s Get profile: %s", DRV_DEBUG_PREFIX, reply_text);
    lm_message_node_unref(node);
    lm_message_unref(message);
    lm_message_unref(reply_message);
        
    DRIVER_DEBUG(5)("%s jabber_get_own_profile END", DRV_DEBUG_PREFIX);
    return BAD_CAST reply_text;
}

xmlChar* jabber_set_own_profile(xmlDocPtr profile)
{
    DRIVER_DEBUG(5)("%s jabber_set_own_profile START", DRV_DEBUG_PREFIX);
    LmConnection* connection = get_connection();
    if (connection == NULL) {
        DRIVER_DEBUG(5)("%s Can't set profile, connection is not setup", DRV_DEBUG_PREFIX);
        return NULL;
    }

    LmMessage* message = build_set_vcard_request(NULL, NULL, profile);

        LmMessage* reply_message =
            lm_connection_send_with_reply_and_block(connection, message, NULL);

    LmMessageNode* node = lm_message_get_node(reply_message);

    gchar* reply_text = lm_message_node_to_string(node);

    DRIVER_MESSAGE(10)("%s Set profile, server reply: %s", DRV_DEBUG_PREFIX, reply_text);
    lm_message_node_unref(node);
    lm_message_unref(message);
    lm_message_unref(reply_message);

    return BAD_CAST reply_text;
}


xmlChar* jabber_send_message(xmlDocPtr message_doc)
{
    DRIVER_DEBUG(5)("%s jabber_send_message START", DRV_DEBUG_PREFIX);
    LmConnection* connection = get_connection();
    if (connection == NULL) {
        DRIVER_DEBUG(5)("%s Can't send message, connection is not setup", DRV_DEBUG_PREFIX);
        return NULL;
    }

    LmMessage* message = build_send_message_request(message_doc);

    LmMessage* reply_message =
        lm_connection_send_with_reply_and_block(connection, message, NULL);

    LmMessageNode* node = lm_message_get_node(reply_message);

    gchar* reply_text = lm_message_node_to_string(node);

    DRIVER_MESSAGE(10)("%s Send message, server reply: %s", DRV_DEBUG_PREFIX, reply_text);
    lm_message_node_unref(node);
    lm_message_unref(message);
    lm_message_unref(reply_message);

    return BAD_CAST reply_text;
}


xmlChar* jabber_get_friends()
{
    DRIVER_DEBUG(5)("%s jabber_get_friends START", DRV_DEBUG_PREFIX);
    LmConnection* connection = get_connection();
    if (connection == NULL) {
        DRIVER_DEBUG(5)("%s Can't get friends, connection is not setup", DRV_DEBUG_PREFIX);
        return NULL;
    }
    DRIVER_MESSAGE(7)("%s Start get friends from server.", DRV_DEBUG_PREFIX);
    LmMessage* message = build_get_friends_request(NULL, NULL);
    LmMessage* reply_message =
            lm_connection_send_with_reply_and_block(connection, message, NULL);

    LmMessageNode* node = lm_message_get_node(reply_message);

    gchar* reply_text = lm_message_node_to_string(node);

    DRIVER_MESSAGE(10)("%s End get friends: %s", DRV_DEBUG_PREFIX, reply_text);
    lm_message_node_unref(node);
    lm_message_unref(message);
    lm_message_unref(reply_message);

    DRIVER_DEBUG(5)("%s jabber_get_own_profile END", DRV_DEBUG_PREFIX);
    return BAD_CAST reply_text;
}