/*
 * driver.c - jabber driver.
 * This file is part of MSA program.
 *
 * Copyright (C) 2009 - Alexander A. Lomov.
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with <program name>; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include <libxml2/libxml/tree.h>

#include "../../msa.h"
#include "../../msa_module.h"
#include "driver.h"
#include "jabber_general.h"
#include "debug.h"
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml2/libxml/parser.h>

#define PARAMS_TAG_NAME "Params"
#define CONTENT_TAG_NAME "Content"
#define SOURCE_ID_TAG_NAME "SourceID"
#define TARGET_ID_TAG_NAME "TargetID"
#define TRANSIT_DATA_TAG_NAME "TransitData"
#define CONTENT_NODE_XPATH "//TransitData/Content"
#define REQUEST_PARAMS_NODE_XPATH "/Request/Params"


#define RESPONSE_TAG_NAME "Response"

#define CLASS_ATTR_NAME "class"
#define CLASS_NAME_SETTINGS "settings"
#define CLASS_NAME_FRIENDS "friends"
#define CLASS_NAME_PROFILE "profile"
#define CLASS_NAME_MESSAGE "message"

#define CLASS_NAME_SYSTEM_MESSAGE "systemMessages"

#define FUNC_NAME_SEND_MESSAGE "sendMessage"

#define FUNC_ATTR_NAME "function"
#define FUNC_NAME_GET_SETTINGS "getSettings"
#define FUNC_NAME_SET_SETTINGS "setSettings"

#define FUNC_NAME_UPDATE_PROFILE "updateProfile"
#define FUNC_NAME_GET_PROFILE "getProfile"
#define FUNC_NAME_SET_PROFILE "setProfile"

#define FUNC_NAME_GET_FRIENDS "getListFriends"
#define FUNC_NAME_UPDATE_FRIENDS "updateListFriends"
//#define FUNC_NAME_SET_PROFILE "setProfile"

#define FUNC_NAME_ERROR_MESSAGE "errorMessage"
#define FUNC_NAME_INFO_MESSAGE "infoMessage"


#define DATA_TYPE_TAG_STRING "string"
#define DATA_TYPE_TAG_IMG "img"
#define DATA_TYPE_TAG_BOOLEAN "boolean"
#define DATA_TYPE_TAG_NUMBER "number"
#define DATA_TYPE_NAME_ATTR "name"

#define DATA_TYPE_ATTR_LOGIN_VALUE "login"
#define DATA_TYPE_ATTR_PASSWORD_VALUE "password"
#define DATA_TYPE_ATTR_AUTOUPDATE_VALUE "autoupdate"
#define DATA_TYPE_ATTR_STATUS_VALUE "active"

#define DATA_TYPE_ATTR_MESSAGE_MODULE_NAME_VALUE "moduleName"
#define DATA_TYPE_ATTR_MESSAGE_CODE_VALUE "code"
#define DATA_TYPE_ATTR_MESSAGE_TEXT_VALUE "text"

#define MAX_TEXT_SIZE 128

#define MESSAGE_CODE_ERROR 404
#define MESSAGE_CODE_SUCCESS 410


#define DRV_DEBUG_PREFIX "JB_DRV:"



/* Function for work with xml data */
static xmlChar* get_source_id(const xmlDocPtr doc);
//static GList* get_targets(xmlDocPtr doc);
//static target_data* create_target(const xmlNodePtr target_node);
static xmlXPathObjectPtr get_nodeset(const xmlDocPtr doc, const xmlChar* xpath);
static xmlNodePtr get_content_node(const xmlDocPtr doc);

static void set_proxy(msa_driver_info* info);
static xmlDocPtr build_response_doc(const gchar* class, const gchar* func_name);


static xmlDocPtr process_request(xmlDocPtr request);
static xmlDocPtr build_get_settings_response();
static xmlChar* get_req_class(xmlDocPtr request);
xmlChar* get_req_function(xmlDocPtr request);

static xmlNodePtr get_node_by_xpath(const xmlDocPtr doc, const xmlChar* xpath);

static int insert_settings_to_params(xmlDocPtr response);

static xmlDocPtr build_response_tr_doc(const gchar* sourceId, const gchar* targetId,
        const xmlDocPtr content_doc);

static xmlDocPtr build_system_message_response(const gchar* function_name,
        const gint code, const gchar* message_text);

static gint seve_settings(xmlDocPtr request);

static void set_jabber_parameters();




/**************************************************/
/*************** External functions ***************/
/**
 * Module initialization. Prepare driver for work.
 *
 * @param module_setup informaton for initialization.
 *
 * @return 0 on success or not 0 otherwise.
 */
int msa_module_init(msa_module* module_setup)
{
    DRIVER_DEBUG(5)("%s msa_module_init START", DRV_DEBUG_PREFIX);

    proxy_host = NULL;
    proxy_port = 0;
    
    module_setup->name = g_strdup(DRIVER_NAME);
    module_setup->send = driver_send;
    module_setup->shutdown = jabber_shtd;
    module_setup->pic = NULL;

    gchar* pic_content = NULL;
    gsize content_len = 0;

    g_file_get_contents(DRIVER_PIC_PATH, &pic_content, &content_len, NULL);

    if (pic_content != NULL) {
        module_setup->pic = g_base64_encode((guchar*)pic_content, content_len);
    }

    GConfClient* client = gconf_client_get_default ();

    /* Get the initial value from GConf. */
    gchar* path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_STATUS, NULL);
    gint value = gconf_client_get_int(client, path, NULL);
    g_free(path);
    
    path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_LOGIN, NULL);
    gchar* login = gconf_client_get_string(client, path, NULL);
    g_free(path);
    
    path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_PASSWORD, NULL);
    gchar* password = gconf_client_get_string(client, path, NULL);
    g_free(path);
    path = NULL;
    
    module_setup->status = value;

    if (value == 0) {
        module_setup->status = OFF;
        module_setup->state = NOT_CONF;
    } else {
        if (login == NULL || password == NULL) {
            module_setup->state = NOT_CONF;
            module_setup->status = OFF;
        } else
            module_setup->state = CONF;
    }

    jabber_init();
    jabber_set_user_context(login, password);
    //set_proxy(module_setup->);

    g_free(login);
    g_free(password);
    g_object_unref((gpointer)client);

    //jabber_set_proxy(DEBUG_PROXY_HOST, DEBUG_PROXY_PORT);
    //jabber_connect();
    DRIVER_DEBUG(5)("%s msa_module_init END", DRV_DEBUG_PREFIX);
    return 0;
}



gboolean is_proxy_change(msa_driver_info* info)
{
    if (info == NULL) {
        return FALSE;
    }
    
    const gchar* proxy_host = jabber_get_proxy_host();
    guint proxy_port = jabber_get_proxy_port();

#ifdef NORMAL_PROXY
    if (proxy_port != info->port) {
        return TRUE;
    }

    if (proxy_port != NULL && info->proxy != NULL) {
        if(g_strcmp0(info->proxy, proxy_port) != 0) {
            return TRUE;
        }
    }
#else
    if (proxy_port != DEBUG_PROXY_PORT) {
        return TRUE;
    }
   if (proxy_host != NULL && info->proxy != NULL) {
        if(strcmp(DEBUG_PROXY_HOST, proxy_host) != 0) {
            return TRUE;
        }
    }
#endif
    
    return FALSE;
}

gint init_jabber_connect(msa_driver_info* info)
{
    if (is_proxy_change(info) == TRUE) {
        g_debug("CON INIT %s %i", info->proxy, info->port);
        set_proxy(info);
#ifdef NORMAL_PROXY
        jabber_set_proxy(info->proxy, info->port);
#else
       DRIVER_MESSAGE(7)("%s Using psevdo-proxy: host = %s, port = %i", \
       DRV_DEBUG_PREFIX, DEBUG_PROXY_HOST, DEBUG_PROXY_PORT);
       jabber_set_proxy(DEBUG_PROXY_HOST, (gint)DEBUG_PROXY_PORT);
#endif
        return jabber_reconnect();
    }

    gint connect_result = jabber_connect();
    if (connect_result != JABBER_ALLREADY_CONNECT
            || connect_result != JABBER_SUCCESS)  {
        return JABBER_CANT_CONNETC;
    }
    return 0;
}


/**
 * @brief Function for send external data to driver.
 *
 * This function used by external system, address sets in the initialization.     
 *
 * @see msa_module_init()
 *
 * @param request_tr transit data request of external system.
 * @param response transit data respons for external system.
 * @param info options for driver (proxy or something else). 
 *
 * @return 0 on success or not 0 otherwise.
 */
gint driver_send(xmlDocPtr request_tr, xmlDocPtr* response_tr, 
        msa_driver_info* info)
{
    DRIVER_DEBUG(7)("%s driver_send START", DRV_DEBUG_PREFIX);

    if (request_tr == NULL) {
        DRIVER_MESSAGE(5)("%s driver_send: request == NULL", DRV_DEBUG_PREFIX);
		return 1;
	}
    
    xmlDocDump(stdout, request_tr);
xmlChar* source_id = get_source_id(request_tr);

    set_proxy(info);
#ifdef NORMAL_PROXY
    jabber_set_proxy(info->proxy, info->port);
#else
    jabber_set_proxy(DEBUG_PROXY_HOST, DEBUG_PROXY_PORT);
 #endif
/*
    if (init_jabber_connect(info) != 0) {
        g_debug("NOT CON");
        xmlDocPtr message = build_system_message_response(FUNC_NAME_ERROR_MESSAGE, 403, DRV_NETW_ERROR);
        xmlDocPtr tmp = build_response_tr_doc(DRIVER_ID, source_id, message);
       *response_tr = xmlCopyDoc(tmp, 1);
       xmlFreeDoc(tmp);
       xmlDocDump(stdout, *response_tr);
       g_free(source_id);
       return NULL;
    }
*/

    xmlNodePtr content_node = get_content_node(request_tr);
    xmlNodePtr request_node = content_node->children;
    xmlUnlinkNode(request_node);
  
    xmlDocPtr request = xmlNewDoc(BAD_CAST XML_VERSION);
    request->encoding = BAD_CAST g_strdup(XML_ENCODING);
    xmlDocSetRootElement(request, request_node);
  
   xmlDocDump(stdout, request);
    
   xmlDocPtr response = process_request(request);   
   if (response == NULL) {
       response_tr = NULL;
   } else { // tmp need for debug. We copy doc, and it has no links with this module.
       xmlDocPtr tmp = build_response_tr_doc(DRIVER_ID, (const gchar*)source_id,
               (const xmlDocPtr)response);
       *response_tr = xmlCopyDoc(tmp, 1);
       xmlFreeDoc(tmp);
       xmlDocDump(stdout, *response_tr);
   }

   DRIVER_DEBUG(7)("%s driver_send END", DRV_DEBUG_PREFIX);
   
   return 0;

    xmlDocPtr reqXml = NULL;
    xmlDocPtr doc = xmlNewDoc(BAD_CAST XML_VERSION);
    doc->encoding = BAD_CAST g_strdup(XML_ENCODING);
    xmlNodePtr node = xmlDocGetRootElement(request_tr)->children;
    xmlNodePtr node2 =NULL;

   vk_req(request, doc);

    


    node = xmlDocGetRootElement(doc);
    node2 = xmlCopyNode(node, 1);

    //xmlFreeDoc(doc);
    reqXml = xmlNewDoc(BAD_CAST XML_VERSION);
    reqXml->encoding = BAD_CAST g_strdup(XML_ENCODING);

    node = xmlNewDocNode(reqXml, NULL, BAD_CAST TRANSIT_DATA_TAG_NAME, NULL);
    xmlDocSetRootElement(reqXml, node);

    xmlNodePtr child;
    child = xmlNewDocNode(reqXml, NULL, BAD_CAST SOURCE_ID_TAG_NAME, NULL);
    xmlNodeSetContent(child, BAD_CAST DRIVER_ID);
    xmlAddChild(node, child);

    child = xmlNewDocNode(reqXml, NULL, BAD_CAST TARGET_ID_TAG_NAME, NULL);
//    xmlNodeSetContent(child, source_name);
    xmlAddChild(node, child);

    child = xmlNewDocNode(reqXml, NULL, BAD_CAST CONTENT_TAG_NAME, NULL);
    xmlAddChild(node, child);

    node = node->children;

    while (node != NULL) {
        if (xmlStrncmp(node->name, BAD_CAST CONTENT_TAG_NAME, 
                xmlStrlen(BAD_CAST CONTENT_TAG_NAME)) ==  0) {
            xmlAddChild(node, node2);
            break;
        }
        node = node->next;  
    }
    //xmlDocDump(stdout, reqXml);
    *response_tr = xmlCopyDoc(reqXml, 1);
    xmlFreeDoc(request);
    xmlFreeDoc(reqXml);
    DRIVER_DEBUG(7)("%s driver_send END", DRV_DEBUG_PREFIX);
    return 0;
}


/**
 * @brief Build Transit data xml structure.
 *
 * @param sourceId identifier of sender.
 * @param targetId identifier of recipient/target, who receives this data. 
 * @param content_doc document containing content for transit data structure.
 *
 * @return transit data xml structure.
*/
static xmlDocPtr build_response_tr_doc(const gchar* sourceId, const gchar* targetId,
        const xmlDocPtr content_doc)
{
    xmlDocPtr response_tr = xmlNewDoc(BAD_CAST XML_VERSION);
    response_tr->encoding = BAD_CAST g_strdup(XML_ENCODING);
    
    xmlNodePtr root_node = xmlNewDocNode(response_tr, NULL, 
            BAD_CAST TRANSIT_DATA_TAG_NAME, NULL);
    xmlDocSetRootElement(response_tr, root_node);
    
    xmlNodePtr node = xmlNewDocNode(response_tr, NULL, 
            BAD_CAST SOURCE_ID_TAG_NAME, BAD_CAST sourceId);
    xmlAddChild(root_node, node);
    
    node = xmlNewDocNode(response_tr, NULL, 
            BAD_CAST TARGET_ID_TAG_NAME, BAD_CAST targetId);
    xmlAddChild(root_node, node);
    
    node = xmlNewDocNode(response_tr, NULL, 
            BAD_CAST CONTENT_TAG_NAME, NULL);
    xmlAddChild(root_node, node); // Content node of transit data
    
    xmlNodePtr content_node = xmlDocGetRootElement(content_doc);
    
    // Add content to content node of transit data
    if (content_node != NULL) {
        content_node = xmlCopyNode(content_node, 1);
        xmlAddChild(node, content_node);
    }

    return response_tr;
}


/***************************************************/
/***************** Static functions ****************/
/**
 * @brief Set proxy setting using external info data.
 *
 * @param info options for driver (proxy or something else). 
 */
static void set_proxy(msa_driver_info* info)
{    
    if (info != NULL) {
        proxy_host = info->proxy;
        proxy_port = info->port;
    } else {
        proxy_host = NULL;
        proxy_port = 0;
    }
}





/**
 * @brief Gets a source id from transin data.
 *
 * @param doc xml document describing transit data.
 *
 * @return content of source id node, NULL otherwise.
 */
static xmlChar* get_source_id(const xmlDocPtr doc)
{
    xmlXPathObjectPtr result = get_nodeset(doc, BAD_CAST SOURCE_TAG_XPATH);
   
    if (result == NULL) {
        return NULL;
    }
     
    xmlNodeSetPtr nodeset = result->nodesetval;
   
    xmlChar* source_id = NULL;
    if (nodeset->nodeNr > 0) {
        source_id = xmlNodeGetContent(nodeset->nodeTab[0]);
    }
   
    xmlXPathFreeObject(result);
    return source_id;
}


/**
 * @brief Find node by given xpath expression.
 *
 * @param doc xml document for searrch.
 * @param xpath xpatp expression.
 *
 * @return found node on success or NULL otherwise.
 */
static xmlNodePtr get_node_by_xpath(const xmlDocPtr doc, const xmlChar* xpath)
{
    xmlXPathObjectPtr result = get_nodeset(doc, xpath);
    
    if (result == NULL) {
        return NULL;
    }
     
    xmlNodeSetPtr nodeset = result->nodesetval;
   
    xmlNodePtr node = NULL;
    if (nodeset->nodeNr > 0) {
        node = nodeset->nodeTab[0];
    }
   
    xmlXPathFreeObject(result);
    return node;
}


/**
 * @brief Gets a content of transin data.
 *
 * @param doc xml document describing transit data.
 *
 * @return content of transit data on success or NULL otherwise.
 */
static xmlNodePtr get_content_node(const xmlDocPtr doc)
{
    xmlXPathObjectPtr result = get_nodeset(doc, BAD_CAST CONTENT_NODE_XPATH);
   
    if (result == NULL) {
        return NULL;
    }
     
    xmlNodeSetPtr nodeset = result->nodesetval;
   
    xmlNodePtr content_node = NULL;
    if (nodeset->nodeNr > 0) {
        content_node = nodeset->nodeTab[0];
    }
   
    xmlXPathFreeObject(result);
    return content_node;
}


/**
 * @brief Gets nodes from xml document using XPath expression.
 *
 * @param doc xml document.
 * @param xpath XPath expression.
 *
 * @return find nodes or NULL otherwise.
 */
static xmlXPathObjectPtr get_nodeset(const xmlDocPtr doc, const xmlChar* xpath)
{
    if (xmlDocGetRootElement(doc) == NULL || xpath == NULL) {
        return NULL;
    }
	xmlXPathContextPtr context = xmlXPathNewContext(doc);
   
	if (context == NULL) {
        DRIVER_MESSAGE(5)("%s Error in xmlXPathNewContext", DRV_DEBUG_PREFIX);
		return NULL;
	}
	
	xmlXPathObjectPtr result = xmlXPathEvalExpression(xpath, context);
	xmlXPathFreeContext(context);
	
	if (result == NULL) {
        DRIVER_MESSAGE(5)("%s Error in xmlXPathEvalExpression", DRV_DEBUG_PREFIX);
		return NULL;
	}
	
	if (xmlXPathNodeSetIsEmpty(result->nodesetval)) {
        DRIVER_MESSAGE(5)("%s get_nodeset: EMPTY XPATH", DRV_DEBUG_PREFIX);        
		xmlXPathFreeObject(result);
		return NULL;
	}
	
	return result;
}



static xmlDocPtr process_settings_request(const xmlDocPtr request)
{
    xmlChar* function = get_req_function(request);
    xmlDocPtr response = NULL;
    
    if (xmlStrncmp(function, BAD_CAST FUNC_NAME_GET_SETTINGS, 
            xmlStrlen(BAD_CAST FUNC_NAME_GET_SETTINGS)) == 0) {
        response = build_get_settings_response();
    } else if (xmlStrncmp(function, BAD_CAST FUNC_NAME_SET_SETTINGS, 
            xmlStrlen(BAD_CAST FUNC_NAME_SET_SETTINGS)) == 0) {
        if (seve_settings(request) == 0) {
           response = build_system_message_response(FUNC_NAME_INFO_MESSAGE,
                   MESSAGE_CODE_SUCCESS, DRV_SETT_SAVED);
        } else {
            response = build_system_message_response(FUNC_NAME_ERROR_MESSAGE, 
                   MESSAGE_CODE_ERROR, DRV_SETT_ERROR);
        }
}




    xmlFree(function);
    return response;
}


static xmlDocPtr build_get_profile_response()
{
    DRIVER_DEBUG(5)("%s build_get_profile_response START", DRV_DEBUG_PREFIX);
    xmlChar* text_profile = BAD_CAST jabber_get_own_profile();

    if (text_profile == NULL) {
        return NULL;
    }

    xmlDocPtr jb_profile = xmlParseDoc(text_profile);
    xmlFree(text_profile);
    xmlDocPtr profile = convert_jabber_profile_to_inner_format(jb_profile,"d","x", FUNC_NAME_UPDATE_PROFILE);
    xmlFreeDoc(jb_profile);
    DRIVER_DEBUG(5)("%s build_get_profile_response END", DRV_DEBUG_PREFIX);
    return profile;
}


gint save_profile(const xmlDocPtr request)
{
    xmlDocPtr profile = convert_inner_profile_to_jabber_format(request);

    jabber_set_own_profile(profile);
    return 0;
}

gint send_message(const xmlDocPtr request)
{

    gchar* jid_r = jabber_get_jid_with_resource();
    xmlDocPtr message = convert_inner_message_to_jabber_format(request,
            (const gchar*)jid_r, DRIVER_ID);

    g_free(jid_r);
    jabber_send_message(message);
 
    return 0;
}



static xmlDocPtr process_profile_request(const xmlDocPtr request)
{
    xmlChar* function = get_req_function(request);
    xmlDocPtr response = NULL;

    DRIVER_DEBUG(7)("%s process_profile_request: func name = %s START", \
            DRV_DEBUG_PREFIX, function);
    if (xmlStrncmp(function, BAD_CAST FUNC_NAME_UPDATE_PROFILE,
            xmlStrlen(BAD_CAST FUNC_NAME_UPDATE_PROFILE)) == 0) {
        response = build_get_profile_response();
    } else if (xmlStrncmp(function, BAD_CAST FUNC_NAME_GET_PROFILE,
            xmlStrlen(BAD_CAST FUNC_NAME_GET_PROFILE)) == 0) {
        response = build_get_profile_response();
/*
        if (seve_settings(request) == 0 ) {
           response = build_system_message_response(FUNC_NAME_INFO_MESSAGE,
                   MESSAGE_CODE_SUCCESS, DRV_SETT_SAVED);
        } else {
            response = build_system_message_response(FUNC_NAME_ERROR_MESSAGE,
                   MESSAGE_CODE_ERROR, DRV_SETT_ERROR);
        }
*/
    } else if (xmlStrncmp(function, BAD_CAST FUNC_NAME_SET_PROFILE,
            xmlStrlen(BAD_CAST FUNC_NAME_SET_PROFILE)) == 0) {
       save_profile(request);

    }

    xmlFree(function);
    DRIVER_DEBUG(7)("%s process_profile_request END", DRV_DEBUG_PREFIX);
    return response;
}


static xmlDocPtr build_get_friends_response()
{
    DRIVER_DEBUG(5)("%s build_get_friends_response START", DRV_DEBUG_PREFIX);
    xmlChar* text_roster = BAD_CAST jabber_get_friends();

    if (text_roster == NULL) {
        return NULL;
    }

    xmlDocPtr jb_roster = xmlParseDoc(text_roster);
    xmlFree(text_roster);
    xmlDocPtr friends = convert_jabber_roster_to_inner_format(jb_roster, NULL,
            DRIVER_ID, FUNC_NAME_UPDATE_FRIENDS);
    xmlFreeDoc(jb_roster);
    DRIVER_DEBUG(5)("%s build_get_friends_response END", DRV_DEBUG_PREFIX);
    return friends;
}


static xmlDocPtr process_friends_request(const xmlDocPtr request)
{
    xmlChar* function = get_req_function(request);
    xmlDocPtr response = NULL;

    DRIVER_DEBUG(7)("%s process_friends_request: func name = %s START", \
            DRV_DEBUG_PREFIX, function);

    if (xmlStrncmp(function, BAD_CAST FUNC_NAME_UPDATE_FRIENDS, 
            xmlStrlen(BAD_CAST FUNC_NAME_UPDATE_FRIENDS)) == 0) {
        response = build_get_friends_response();
    }

    xmlFree(function);
    DRIVER_DEBUG(7)("%s process_profile_request END", DRV_DEBUG_PREFIX);
    return response;
}


static xmlDocPtr process_messages_request(const xmlDocPtr request)
{
    xmlChar* function = get_req_function(request);
    xmlDocPtr response = NULL;

    DRIVER_DEBUG(7)("%s process_messages_request: func name = %s START", \
            DRV_DEBUG_PREFIX, function);

    if (xmlStrncmp(function, BAD_CAST FUNC_NAME_SEND_MESSAGE,
            xmlStrlen(BAD_CAST FUNC_NAME_SEND_MESSAGE)) == 0) {
        send_message(request);
    }

    xmlFree(function);
    DRIVER_DEBUG(7)("%s process_messages_request END", DRV_DEBUG_PREFIX);
    return response;
}



static xmlDocPtr process_request(const xmlDocPtr request)
{
    xmlChar* class = get_req_class(request);
    xmlDocPtr response = NULL;
    
    if (xmlStrncmp(class, BAD_CAST CLASS_NAME_SETTINGS, 
            xmlStrlen(BAD_CAST CLASS_NAME_SETTINGS)) == 0) {
        
        response = process_settings_request(request);
    } else if (xmlStrncmp(class, BAD_CAST CLASS_NAME_PROFILE,
            xmlStrlen(BAD_CAST CLASS_NAME_PROFILE)) == 0) {
        response = process_profile_request(request);
    } else if (xmlStrncmp(class, BAD_CAST CLASS_NAME_FRIENDS,
            xmlStrlen(BAD_CAST CLASS_NAME_FRIENDS)) == 0) {
        response = process_friends_request(request);
    } else if (xmlStrncmp(class, BAD_CAST CLASS_NAME_MESSAGE,
            xmlStrlen(BAD_CAST CLASS_NAME_MESSAGE)) == 0) {
        response = process_messages_request(request);
    }

    xmlFree(class);
    return response;
}



/**
 * @brief Gets settings from request and try save it into GConf system.
 *
 * @param request request document with driver parameters.
 *
 * @return 0 on success or not 0 otherwise.
 */
static gint seve_settings(xmlDocPtr request)
{   
    xmlNodePtr params_node = get_node_by_xpath(request, BAD_CAST REQUEST_PARAMS_NODE_XPATH);

    if (params_node == NULL) {
        return -1;
    }
    
    // Get node describes setting, then gous on sibling nodes. 
    xmlNodePtr setting_node = params_node->children;
    
    GConfClient* gc_client = gconf_client_get_default();
    gboolean is_settings_save = TRUE;
    
    while(setting_node != NULL) {
        gboolean is_save_succeeded = FALSE;
        
        gchar* setting_name = (gchar*)xmlGetProp(setting_node, BAD_CAST DATA_TYPE_NAME_ATTR);
        gchar* setting_val = (gchar*)xmlNodeGetContent(setting_node);
        
        if (strncmp(setting_name, DRIVER_LOGIN, strlen(DRIVER_LOGIN)) == 0) {
            gchar* gc_path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_LOGIN, NULL);
            is_save_succeeded = gconf_client_set_string(gc_client, gc_path, 
                    setting_val, NULL);  
            g_free(gc_path);
        } else if (strncmp(setting_name, DRIVER_PASSWORD, strlen(DRIVER_PASSWORD)) == 0) {
            gchar* gc_path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_PASSWORD, NULL);
            is_save_succeeded = gconf_client_set_string(gc_client, gc_path, 
                    setting_val, NULL);
            g_free(gc_path);
        } else if (strncmp(setting_name, DRIVER_AUTOUPDATE, strlen(DRIVER_AUTOUPDATE)) == 0) {
            gchar* gc_path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_AUTOUPDATE, NULL);
            is_save_succeeded = gconf_client_set_int(gc_client, gc_path, 
                    atoi(setting_val), NULL);
            g_free(gc_path);
        } else if (strncmp(setting_name, DRIVER_STATUS, strlen(DRIVER_STATUS)) == 0) {
            gchar* gc_path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_STATUS, NULL);
            is_save_succeeded = gconf_client_set_int(gc_client, gc_path, 
                    atoi(setting_val), NULL);
            g_free(gc_path);
           
        }
        setting_node = setting_node->next;
        g_free(setting_name);
        g_free(setting_val);
        
        if (is_save_succeeded == FALSE) {
            is_settings_save = FALSE;
        }
    }
    g_object_unref((gpointer)gc_client);

    if (is_settings_save == FALSE) {
        return -1;
    }

    set_jabber_parameters();
    return 0;
}








/**
 * Gets class of request.
 *
 * @param request request from the external system/moduels.
 *
 * @return class of request.
 */
static xmlChar* get_req_class(xmlDocPtr request)
{
    xmlNodePtr rootNode = xmlDocGetRootElement(request);
    return xmlGetProp(rootNode, BAD_CAST CLASS_ATTR_NAME);
}

/**
 * Gets function name of request.
 *
 * @param request request from the external system/moduels.
 *
 * @return function name of request.
 */
xmlChar* get_req_function(xmlDocPtr request)
{
    xmlNodePtr rootNode = xmlDocGetRootElement(request);
    return xmlGetProp(rootNode, BAD_CAST FUNC_ATTR_NAME);
}


/**
 * Create empty response xml doc with given class and function name.
 *
 * @param class class of response.
 * @param func_name function name of class.
 *
 * @return response xml document on success or NULL otherwise.
 */
static xmlDocPtr build_response_doc(const gchar* class, const gchar* func_name)
{
    xmlDocPtr doc = xmlNewDoc(BAD_CAST XML_VERSION);
    doc->encoding = BAD_CAST g_strdup(XML_ENCODING);

    xmlNodePtr root_node = xmlNewDocNode(doc, NULL, BAD_CAST RESPONSE_TAG_NAME, NULL);
    xmlSetProp(root_node, BAD_CAST CLASS_ATTR_NAME, BAD_CAST class);
    xmlSetProp(root_node, BAD_CAST FUNC_ATTR_NAME, BAD_CAST func_name);
    xmlDocSetRootElement(doc, root_node);
    
    xmlNodePtr params_node = xmlNewDocNode(doc, NULL, BAD_CAST PARAMS_TAG_NAME, NULL);
    xmlAddChild(root_node, params_node);
    
    return doc;
}


static void add_node_to_params(xmlDocPtr doc, const gchar* name, 
        const gchar* value, const gchar* type_attr, const gchar* type_attr_val)
{
    if (doc == NULL || name == NULL || type_attr == NULL 
            || type_attr_val == NULL) {
        return;
    }

    xmlNodePtr params_node = xmlDocGetRootElement(doc)->children;
    
    if (params_node == NULL) {
        return;
    }
    
    xmlNodePtr node = xmlNewDocNode(doc, NULL, BAD_CAST name, BAD_CAST value);
    xmlSetProp(node, BAD_CAST type_attr, BAD_CAST type_attr_val);
    xmlAddChild(params_node, node);
}


/**
 * @brief Build respons with driver parameters.
 *
 * @return Response xml document.
 */
static xmlDocPtr build_get_settings_response()
{
    DRIVER_DEBUG(5)("%s build_get_settings_response START", DRV_DEBUG_PREFIX);        
    xmlDocPtr response = build_response_doc(CLASS_NAME_SETTINGS, 
            FUNC_NAME_GET_SETTINGS);
    
    insert_settings_to_params(response);
    
    DRIVER_DEBUG(5)("%s build_get_settings_response END", DRV_DEBUG_PREFIX);        
    
    return response;
}


/**
 * Get driver settings from GConf.
 *
 * @param doc - response
 *
 * @return 0 on success or 1 otherwise.
 */
static int insert_settings_to_params(xmlDocPtr response)
{
    DRIVER_DEBUG(7)("%s insert_settings_to_params START", DRV_DEBUG_PREFIX);        
    
    GConfClient* client = gconf_client_get_default();
    
    gchar* path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_STATUS, NULL);
    gint status = gconf_client_get_int(client, path, NULL);
    gchar* status_str = g_strdup_printf("%d",status);
    g_free(path);
    
    path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_LOGIN, NULL);
    gchar* login = gconf_client_get_string (client, path, NULL);
    g_free(path);
    
    path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_PASSWORD, NULL);
    gchar* password = gconf_client_get_string (client, path, NULL);
    g_free(path);
    
    path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_AUTOUPDATE, NULL);
    gint autoupdate = gconf_client_get_int (client, path, NULL);
    gchar* autoupdate_str = g_strdup_printf("%d", autoupdate);
    g_free(path);
    
    add_node_to_params(response, DATA_TYPE_TAG_STRING, login, 
            DATA_TYPE_NAME_ATTR, DATA_TYPE_ATTR_LOGIN_VALUE);
    
    add_node_to_params(response, DATA_TYPE_TAG_STRING, password, 
            DATA_TYPE_NAME_ATTR, DATA_TYPE_ATTR_PASSWORD_VALUE);

    add_node_to_params(response, DATA_TYPE_TAG_NUMBER, autoupdate_str, 
            DATA_TYPE_NAME_ATTR, DATA_TYPE_ATTR_AUTOUPDATE_VALUE);

    add_node_to_params(response, DATA_TYPE_TAG_BOOLEAN, status_str, 
            DATA_TYPE_NAME_ATTR, DATA_TYPE_ATTR_STATUS_VALUE);
    
    g_free(login);
    g_free(password);
    g_free(autoupdate_str);
    g_free(status_str);

    DRIVER_DEBUG(7)("%s insert_settings_to_params END", DRV_DEBUG_PREFIX);            
    return 0;
}

static xmlDocPtr build_system_message_response(const gchar* function_name,
        const gint code, const gchar* message_text) {
    xmlDocPtr message = build_response_doc(CLASS_NAME_SYSTEM_MESSAGE, function_name);
    
    add_node_to_params(message, DATA_TYPE_TAG_STRING, DRIVER_NAME,
            DATA_TYPE_NAME_ATTR, DATA_TYPE_ATTR_MESSAGE_MODULE_NAME_VALUE);

    gchar* code_str = g_strdup_printf("%d", code);
    add_node_to_params(message, DATA_TYPE_TAG_STRING, code_str, 
            DATA_TYPE_NAME_ATTR, DATA_TYPE_ATTR_MESSAGE_CODE_VALUE);
    g_free(code_str);

    add_node_to_params(message, DATA_TYPE_TAG_STRING, message_text, 
            DATA_TYPE_NAME_ATTR, DATA_TYPE_ATTR_MESSAGE_TEXT_VALUE);
    
    return message;
}




static void set_jabber_parameters()
{
    GConfClient* client = gconf_client_get_default ();

    /* Get the initial value from GConf. */
    gchar* path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_STATUS, NULL);
    gint value = gconf_client_get_int(client, path, NULL);
    g_free(path);

    path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_LOGIN, NULL);
    gchar* login = gconf_client_get_string(client, path, NULL);
    g_free(path);

    path = g_strconcat(DRIVER_GCONF_PATH, DRIVER_PASSWORD, NULL);
    gchar* password = gconf_client_get_string(client, path, NULL);
    g_free(path);
    path = NULL;

    jabber_shtd();
    jabber_init();
    jabber_set_user_context(login, password);
    g_object_unref((gpointer)client);
}
