
/*
 * converter.c - convert in/from jabber request/response format from/to external format.
 * This file is part of MSA program.
 *
 * Copyright (C) 2009 - Alexander A. Lomov. All rights reserved.
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with <program name>; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include "converter.h"
#include "debug.h"

#define DRV_DEBUG_PREFIX "JB_CONV:"
#define XSLT_VCARD_PATH "vcard.xslt"
#define XSLT_REV_VCARD_PATH "rev_vcard.xslt"
#define XSLT_ROSTER_PATH "roster.xslt"
#define XSLT_REV_MESSAGE "rev_message.xslt"


#define XSLT_PREFIX_ID_PARAM_NAME "PREFIX_ID"
#define XSLT_PREFIX_PARAM_NAME "PREFIX"
#define XSLT_FUNC_NAME_PARAM_NAME "FUNC_NAME"
#define XSLT_OWN_JID_PARAM_NAME "OWN_JID"
#define XSLT_ID_PARAM_NAME "OWN_JID"

/**
 * @brief Load stylescheet associated with given schema.
 *
 * @param schema_name name of schema.
 * @param is_revers indicates that need load the reverse transformation.
 *
 * @return stylesheet document on success or NULL otherwise.
 */
xsltStylesheetPtr xmlu_get_stylesheet(const gchar* schema_name)
{
    if (schema_name == NULL) {
        return NULL;
    }

    gchar* xslt_path = NULL;

/*
    if (is_revers == TRUE) {
        xslt_path = cm_get_revers_xslt_path_key(schema_name);
    } else {
        xslt_path = cm_get_xslt_path_key(schema_name);
    }
*/
    g_debug("XSLT: %s", xslt_path);
    if (xslt_path == NULL) {
        return NULL;
    }

    return xsltParseStylesheetFile(BAD_CAST xslt_path);
}



/**
 * @brief Load stylescheet associated with given schema and applyed it to document.
 *
 * @param schema_name name of schema.
 * @param doc xml document.
 * @param params NULL-terminated list of strings.
 *        format: name value name value ... NULL.
 *
 * @return xml document on success or NULL otherwise.
 */
xmlDocPtr xmlu_apply_stylesheet(const gchar* xslt_path, const xmlDocPtr doc, const gchar** params)
{
    if (doc == NULL || xslt_path == NULL) {
        return NULL;
    }

    xsltStylesheetPtr style = xsltParseStylesheetFile(BAD_CAST xslt_path);

    if (style == NULL) {
        return NULL;
    }

    xmlDocPtr result = xsltApplyStylesheet(style, doc, params);

	xsltFreeStylesheet(style);

	return result;
}




xmlDocPtr convert_inner_profile_to_jabber_format(const xmlDocPtr profile)
{
    gchar* path = g_strconcat(DRIVER_DATADIR, XSLT_REV_VCARD_PATH, NULL);
    xmlDocPtr jb_profile = xmlu_apply_stylesheet(path, profile, NULL);

    g_free(path);

    return jb_profile;
}




xmlDocPtr convert_jabber_profile_to_inner_format(const xmlDocPtr jb_profile,
        const gchar* own_id, const gchar* driver_prefix, const gchar* func_name)
{
    DRIVER_DEBUG(5)("%s convert_jabber_profile_to_inner_format START", DRV_DEBUG_PREFIX);
    gchar* path = g_strconcat(DRIVER_DATADIR, XSLT_VCARD_PATH, NULL);
    gchar* id = g_strconcat(own_id, driver_prefix, NULL);

    DRIVER_MESSAGE(10)("%s Get data (jabber profile - vcard): ", DRV_DEBUG_PREFIX); \
    xmlDocDump(stdout, jb_profile);

    xmlNodePtr root_node = xmlDocGetRootElement(jb_profile);
    xmlNsPtr* ns_list = xmlGetNsList(jb_profile, root_node);

    //FIXME: namesapce check, and xslt convert.
    // Now remove ns href link from node for more simple xslt convert.
    if (ns_list != NULL) {
        xmlFree((xmlChar*)ns_list[0]->href); // Not correct
        ns_list[0]->href = NULL;
    }
    ns_list = NULL;

    gchar** params = g_new(gchar*, 5);
    params[0] = g_strdup(XSLT_PREFIX_ID_PARAM_NAME);
    params[1] = NULL; // g_strconcat("\"", id, "\"", NULL);
    params[2] = g_strdup(XSLT_FUNC_NAME_PARAM_NAME);
    params[3] = g_strconcat("\"", func_name, "\"", NULL);
    params[4] = NULL;

    xmlDocPtr profile = xmlu_apply_stylesheet(path, jb_profile, 
            (const gchar**)params);

    g_free(id);
    g_free(path);
    g_strfreev(params);
    DRIVER_MESSAGE(10)("Inner profile format: "); \
    xmlDocDump(stdout, profile);
    DRIVER_DEBUG(5)("%s convert_jabber_profile_to_inner_format END", DRV_DEBUG_PREFIX);
    return profile;
}


xmlDocPtr convert_jabber_roster_to_inner_format(const xmlDocPtr jb_roster,
        const gchar* own_id, const gchar* driver_prefix, const gchar* func_name)
{
    DRIVER_DEBUG(5)("%s convert_jabber_roster_to_inner_format START", DRV_DEBUG_PREFIX);
    gchar* path = g_strconcat(DRIVER_DATADIR, XSLT_ROSTER_PATH, NULL);
    gchar* id = g_strconcat(own_id, driver_prefix, NULL);

    DRIVER_MESSAGE(10)("%s Get data (jabbers roster): ", DRV_DEBUG_PREFIX); \
        xmlDocDump(stdout, jb_roster);

    xmlNodePtr root_node = xmlDocGetRootElement(jb_roster);
    xmlNsPtr* ns_list = xmlGetNsList(jb_roster, root_node);

    //FIXME: namesapce check, and xslt convert.
    // Now remove ns href link from node for simple xslt convert.
    if (ns_list != NULL) {
        xmlFree((xmlChar*)ns_list[0]->href); // Not correct
        ns_list[0]->href = NULL;
    }
    ns_list = NULL;

    gchar** params = g_new(gchar*, 7);
    params[0] = g_strdup(XSLT_PREFIX_ID_PARAM_NAME);
    params[1] = NULL; // g_strconcat("\"", id, "\"", NULL);
    params[2] = g_strdup(XSLT_PREFIX_PARAM_NAME);
    params[3] = g_strconcat("\"", driver_prefix, "\"", NULL);
    params[4] = g_strdup(XSLT_FUNC_NAME_PARAM_NAME);
    params[5] = g_strconcat("\"", func_name, "\"", NULL);
    params[6] = NULL;

    xmlDocPtr profile = xmlu_apply_stylesheet(path, jb_roster,
            (const gchar**)params);

    g_free(id);
    g_free(path);
    g_strfreev(params);
    DRIVER_MESSAGE(10)("Inner profile format: "); \
    xmlDocDump(stdout, profile);
    DRIVER_DEBUG(5)("%s convert_jabber_profile_to_inner_format END", DRV_DEBUG_PREFIX);
    return profile;
}



 xmlDocPtr convert_inner_message_to_jabber_format(const xmlDocPtr message,
        const gchar* own_id, const gchar* driver_prefix)
 {
    DRIVER_DEBUG(5)("%s convert_inner_message_to_jabber_format START", DRV_DEBUG_PREFIX);
    gchar* xslt_file_path = g_strconcat(DRIVER_DATADIR, XSLT_REV_MESSAGE, NULL);

    DRIVER_MESSAGE(10)("%s Get data (inner message): ", DRV_DEBUG_PREFIX); \
        xmlDocDump(stdout, message);

    gchar** params = g_new(gchar*, 5);
    params[0] = g_strdup(XSLT_PREFIX_PARAM_NAME);
    params[1] = g_strconcat("\"", driver_prefix, "\"", NULL); 
    params[2] = g_strdup(XSLT_OWN_JID_PARAM_NAME);
    params[3] = g_strconcat("\"", own_id, "\"", NULL);
    params[4] = NULL;

    xmlDocPtr jb_message = xmlu_apply_stylesheet(xslt_file_path, message,
            (const gchar**)params);

    g_free(xslt_file_path);
    g_strfreev(params);
    DRIVER_MESSAGE(10)("Jabber message format: "); \
        xmlDocDump(stdout, jb_message);
    DRIVER_DEBUG(5)("%s convert_inner_message_to_jabber_format END", DRV_DEBUG_PREFIX);
    return jb_message;
}

