/*******************************************************************************
**
** vktransport.h - transport module for vkgallery
** Based on livkontakte API:  http://oss.fruct.org/wiki/Libvkontakte/xml-data
**
** Copyright (C) 2010 Sergey Zakharov
**
** vkgallery is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** vkgallery is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with QMF-Vkontakte plug-in; if not, write to the Free Software
** Foundation, Inc., 51 Franklin St, Fifth Floor,
** Boston, MA  02110-1301  USA
**
*******************************************************************************/

#ifndef VKTRANSPORT_H
#define VKTRANSPORT_H

#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <glib.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <mysocials_driver.h>

#include <QObject>
#include <QString>
#include <QImage>
#include <QTextCodec>
#include <QLibrary>
#include <QList>
#include <QIcon>

#include "photo.h"
#include "friend.h"
#include "utils.h"
#include "driverinfo.h"

// if VK_LOGING define, then will writes libvkontakte api transfer data
#undef VK_LOGING

#ifdef VK_LOGING
// Name of file for writing log. Will located in temp directory
#define VK_LOGING_FILE "vkLog.log"
#include <QDir>
#endif

// id of client in requests to library
#define CLIENT_NAME "vk_client"

#define VK_AT '@'
#define VK_FAKE_ADDRESS "@vk.com"
#define VK_FAKE_ADDRESS_PATTERN "\\d+@vk\\.com"

#define VK_DEFAULT_SUBJECT "..."
#define VK_SUBJECT_LENGTH 20
#define VK_MESSAGES_AT_ONCE 10
#define VK_MIN_MESSAGES_AT_ONCE 2

#define VK_UID_SEPARATOR '_'
class VkResponse;
class VkSystemMessage;

class VkTransport : public QObject
{
    Q_OBJECT

public:
    VkTransport(QString libName, QString accName);
    ~VkTransport();

    enum Action
    {
        updateProfileAction,
        getProfilesAction,
        updateListFriendsAction,
        getListFriendsAction,
        getSettingsAction,
        setSettingsAction,
        getNewInboxMessagesAction,
        getOutboxMessagesAction,
        getInboxMessagesAction,
        deleteMessageAction,
        readMessageAction,
        sendMessageAction,
        errorMessageAction,
        infoMessageAction,
        captchaMessageAction,
        getBinaryDataAction,
        updateInboxMessagesAction,
        updateOutboxMessagesAction,
        getPhotoAction,
        getPhotosAction,
        getAlbumsAction,
        unknownAction
    };

    // Account id. uses to identify account of service (e.g. vkontakte0)
    QString accountId;
    
    // the name of library file (e.g. libvkontakte.so)
    QString libraryName;

    int activeRequests;
    bool needShutdown;

    void vkGetProfile();
    void vkGetFriends();
    void vkGetAlbums(QString);
    void vkGetPhotos(QString, QString);
    bool vkDownloadPhoto(QString url, QString local_file_name);
    void vkDownloadPhotoList(PhotoList list, Photo curPhoto, int nearest);
    void vkInit(QString, uint);
    void vkGetSettings();
    void vkSetSettings(QString);
    void vkSendMessage(QByteArray strRecipientId, QByteArray strText);
    void vkReadMessage(QString messageId);
    void vkDeleteMessage(QString messageId);
    void vkGetNewInbox();
    void vkClose();

    DriverInfo *getDriverInfo();
    static DriverInfo *getDriverInfo(QString libraryPath);

public slots:
    void vkGetSend();
    void vkGetSend(int reqFrom, int reqTo);
    void vkGetInbox();
    void vkGetInbox(int reqFrom, int reqTo);
    xmlDocPtr vkSendRequest(QByteArray xmlRequest, Action acc);

signals:
    void readyRead();
    void albumsReceived(QString accountid, QString friendId, AlbumList, bool isLastUpdate);
    void photosReceived(QString accountid, QString friendId, QString albumId, PhotoList, bool isLastUpdate);
    void friendsReceived(QString accountid, FriendList, bool isLastUpdate);
    void settingsReceived(QString accountid, QString);
    void profileReceived(QString accountId, Friend);

    void responseReceived(VkResponse*);

    void errorOccurred(QString accountId, int errNum, QString errMsg, VkTransport::Action acc);
    void updateStatus(QString);
    void progressChanged(uint, uint);

private:

    void vkGenerateSignal(xmlDocPtr xmlResponse, Action acc);
    // A set of functions that are used to parse XML response.
    void parseOneNodeSet(xmlXPathObjectPtr &object, xmlChar *xpathRequest,
                         xmlXPathContextPtr &content);
    char* parseString(xmlChar *xpathRequest, xmlXPathContextPtr &content,
                      Action acc, bool important);
    char* parseString(QByteArray charRequest, xmlXPathContextPtr &content,
                      Action acc, bool important);
    xmlNodePtr xpath_get_node(char* path, xmlDocPtr doc);
    xmlXPathObject* xpath(gchar* req, xmlDocPtr doc);

    QString vkGenerateFileName(QString path, QString url);
    VkSystemMessage vkCheckMessage(xmlXPathContextPtr, Action acc);
    void updateFriendsIcons(FriendList list);
    void updateAlbumsIcons(AlbumList list, QString album_owner);
    void updatePhotoIcons(PhotoList list, QString friendId, QString albumId);
    // Pointer to social network driver library
    QLibrary *msa;

    // profile settings in library
    struct msa_module *driverModule;

    // Encoder from one-bit to two-bit encoding (utf-8)
    QTextCodec *decoder;
};

class VkResponse {
public:
    VkResponse();
    ~VkResponse();
    VkTransport::Action action; // Status of Response
    QByteArray captchaImg;      // Image of captcha, for getting user
    QByteArray midFrom;         // First Message id from Response
    QByteArray midTo;           // Last Message id from Response
    QString text;              // Info or error text from libkontakte
    QByteArray code;            // Info or error code from libkontakte
};

class VkSystemMessage {
public:
    VkSystemMessage();
    ~VkSystemMessage();
    bool isError;
    QString text;              // Info or error text from libkontakte
    QString code;            // Info or error code from libkontakte
};

#endif // VKTRANSPORT_H
