#ifndef SERVICEMGR_H
#define SERVICEMGR_H

#include <QObject>
#include <QList>
#include <QFile>
#include <QDebug>
#include <QDomDocument>
#include <QtConcurrentRun>
#include <QMetaType>
#include <QHash>
#include <QDomNode>
#include <QtNetwork/QNetworkProxy>
#include <QUrl>
#include <QSettings>

#include "vktransport.h"
#include "account.h"
#include "utils.h"

#define FILE_ACCOUNTS_LIST "/accounts.xml"
#define FILE_SETTINGS "/gallery.xml"

#define NO_PROXY "no-proxy"
#define SYSTEM_PROXY "system-proxy"
#define USER_PROXY "user-proxy"

#define NODE_ACCOUNTS_ROOT "profiles"
#define NODE_ACCOUNT "profile"

#define SETTINGS_PROXY "proxy"
#define SETTINGS_PROXYPORT "proxy-port"
#define SETTINGS_PROXYHOST "proxy-host"
#define SETTINGS_AUTOROTATE "auto-rotate"

class ServiceMgr : public QObject
{
    Q_OBJECT

public:
    ServiceMgr(QObject *parent = 0);
    ServiceMgr(const ServiceMgr&);
    ~ServiceMgr();

    // get list of friends
    FriendList getFriends(bool isNeedUpdate, bool useSignal);

    // get list of albums
    AlbumList getAlbums(Friend curFriend, bool isNeedUpdate, bool useSignal);

    /**
     * @brief Gets list of my albums.
     *
     * @param isNeedUpdate true if need to get new album list, otherwise - false.
     * @return QList<Album *> stored list of albums. Newest list will be sent using gotAlbums().
    */
    AlbumList getMyAlbums(bool isNeedUpdate, bool useSignal);

    // get content of album
    PhotoList getPhotos(Album curAlbum, bool isNeedUpdate, bool useSignal);

    AccountList getAccounts();

    void downloadPhotos(Photo startPhoto, int nearest);

    /**
     * @brief describes type of used proxy.
     *
     */
    enum ProxyType {
        NoProxy, SystemProxy, UserProxy
    };

    QSettings *settings;

    void updateDriverSettings();

    QList<DriverInfo*> getDrivers();

    void addAccount(QString accountName, DriverInfo *driver);
    void deleteAccount(Account *oldAcc);

    PhotoList getPhotosForAlbum(Photo curPhoto);
    PhotoList getPhotosForAlbum(Account *curAcc, Photo curPhoto);

private:
    // list of user accounts
    AccountList accounts;

    // list of threads that currently running
    QHash<QString, QFuture<void> > threads;

    // load accounts from localdatadir/FILE_ACCOUNTS_LIST
    void loadAccounts();

    // load settings from localdatadir/FILE_SETTINGS
    void loadSettings();

    void storeSettings();

    // remove unused threads
    void cleanThreads();

    // number of friends update processings
    int friendsUpdate;

    int albumsUpdate;

    int photosUpdate;

signals:
    void updateFriends(FriendList list, bool isLastUpdate);
    void updateAlbumList(QString userId, AlbumList, bool isLastUpdate);
    void updatePhotoList(QString albumId, PhotoList, bool isLastUpdate);
    void updateAccounts(AccountList);
    void updatePhoto(QString accountId, QString ownerId, QString albumId, PhotoList list, bool isLastUpdate);
    void errorOccured(QString errMsg, VkTransport::Action);

private slots:
    void gotFriends(QString accountid, FriendList, bool isLastUpdate);
    void gotAlbumList(QString accountid, QString friendId, AlbumList, bool isLastUpdate);
    void gotPhotoList(QString accountid, QString friendId, QString albumId, PhotoList, bool isLastUpdate);
    void gotProfile(QString accountId, Friend profile);
    void storeAccounts(AccountList);
    void gotErrorMsg(QString accountId, int errNum, QString errMsg, VkTransport::Action acc);
};

Q_DECLARE_METATYPE(ServiceMgr::ProxyType)

#endif // SERVICEMGR_H
