#include "account.h"

Account::Account(QString libraryName)
{
    this->transport = new VkTransport(libraryName, NULL);

    QObject::connect(this->transport, SIGNAL(settingsReceived(QString, QString)), this, SLOT(gotSettings(QString, QString)));
    QObject::connect(this->transport, SIGNAL(profileReceived(QString,Friend)), this, SLOT(gotProfile(QString,Friend)));
    this->isFirstRequest = true;
    this->isNetworkEnabled = false;
}

Account::Account(QString libraryName, QString accountId)
{
    this->transport = new VkTransport(libraryName, accountId);
    QObject::connect(this->transport, SIGNAL(settingsReceived(QString, QString)), this, SLOT(gotSettings(QString, QString)));
    QObject::connect(this->transport, SIGNAL(profileReceived(QString,Friend)), this, SLOT(gotProfile(QString,Friend)));
    this->isFirstRequest = true;
    this->isNetworkEnabled = false;
}

Account::~Account()
{
    if (this->transport != NULL) {
        this->transport->vkClose();
        delete this->transport;
    }
}

void Account::setProxy(QString proxyHost, int proxyPort) {
    this->transport->vkInit(proxyHost, proxyPort);
}

// store settings to account
void Account::gotSettings(QString accountId, QString settings) {
    qDebug() << "I got settings for " << accountId;
    if (this->transport->accountId == accountId)
    {
        this->settings = settings;
        this->saveAccount();
    }
}

void Account::gotProfile(QString accountId, Friend data) {
    qDebug() << "I got profile "<< accountId << " \"" << data.name << "\" in " << data.objectName();
    if (this->transport->accountId == accountId)
    {
        this->isNetworkEnabled = true;
        this->profile = data;
        qDebug() << "SetUp profile for " << this->profile.name << " = " << data.name;

        // store settings of driver
        this->transport->vkGetSettings();
        emit this->updateProfile(accountId, this->profile);
    }
}

void Account::saveAccount()
{
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_ACCOUNT_ROOT);
    out.appendChild(profile);

    QDomElement node = out.createElement(NODE_ACCOUNT_SETTINGS);
    if (this->settings.isEmpty())
    {
        this->settings = this->oldSettings;
        qDebug() << "Uses old settings to store";
    }
    QDomText t = out.createTextNode(this->settings);
    profile.appendChild(node);
    node.appendChild(t);

    node = out.createElement(NODE_ACCOUNT_LIBRARY);
    t = out.createTextNode(this->transport->libraryName);
    profile.appendChild(node);
    node.appendChild(t);

    profile.appendChild(Friend::toQDomElement(out, this->profile));

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(this->transport->accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + QString(FILE_ACCOUNT_DATA));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    //file.write(output.toUtf8());
    file.write(out.toByteArray(1));
    file.close();
}

Account* Account::loadAccount(QString accountId)
{
    QFile file(Utils::getAccountDir(accountId) + QString(FILE_ACCOUNT_DATA));
    if (file.open(QIODevice::ReadOnly) == false)
        return NULL;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    QString libName = doc.documentElement().elementsByTagName(NODE_ACCOUNT_LIBRARY).at(0).firstChild().toText().data();

    Account *ret = new Account(libName, accountId);

    ret->oldSettings =  doc.documentElement().elementsByTagName(NODE_ACCOUNT_SETTINGS).at(0).firstChild().toText().data();

    if (!ret->oldSettings.isNull())
        ret->transport->vkSetSettings(ret->oldSettings);

    ret->profile = Friend::fromQDomElement(doc.documentElement().elementsByTagName(NODE_FRIEND_ROOT).at(0).toElement(), accountId);

    return ret;
}

FriendList Account::getFriendList() {
    if (!this->friends.isEmpty())
        return this->friends;

    // try to load friend list
    QFile file(Utils::getAccountDir(this->transport->accountId) + QString(FILE_FRIENDS_DATA));
    if (file.open(QIODevice::ReadOnly) == false)
        return this->friends;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    this->friends.clear();
    QDomNodeList lst = doc.elementsByTagName(NODE_FRIEND_ROOT);

    qDebug() << "Start load of " << lst.length() << " friends";
    for (uint i = 0; i < lst.length(); i++) {
        this->friends.append(Friend::fromQDomElement(lst.at(i).toElement(), this->transport->accountId));
    }

    return this->friends;
}

void Account::setFriendList(FriendList newFriends) {
    this->friends = newFriends;

    // store friends into cache
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_FRIENDS_ROOT);
    out.appendChild(profile);

    for (int i = 0; i < newFriends.length(); i++) {
        profile.appendChild(Friend::toQDomElement(out, newFriends[i]));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(this->transport->accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + QString(FILE_FRIENDS_DATA));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    //file.write(output.toUtf8());
    file.write(out.toByteArray(1));
    file.close();
}

AlbumList Account::getAlbumList() {
    if (!this->albums.isEmpty())
        return this->albums;

    this->albums = this->profile.getAlbumList();
    return this->albums;
}

void Account::setAlbumList(AlbumList list) {
    this->albums = list;
    this->profile.setAlbumList(list, this->transport->accountId, this->profile.id);
}

Friend Account::getProfile(bool isNeedUpdate) {
    // if first check was failed then return old profile
    if (!this->isFirstRequest && !this->isNetworkEnabled)
        return this->profile;

    if (this->profile.id.isEmpty() || isNeedUpdate || this->isFirstRequest) {
        qDebug() << "Request for download profile for " << this->transport->accountId;
        this->isFirstRequest = false;
        QtConcurrent::run(this->transport, &VkTransport::vkGetProfile);
    }

    return this->profile;
}
