#include "qeventfeed.h"

#include <QFile>

#include "utils/utils.h"

class QEventFeedData : public QSharedData
{
public:
    QEventFeedData();
    QEventFeedData(const QEventFeedData& other);
    ~QEventFeedData();

    /**
      \brief The ID of account.
      */
    QString accountId;

    /**
      \brief The ID of event's author.
      */
    QString ownerId;

    /**
      \brief The name of event's author.
      */
    QString ownerName;

    /**
      \brief The contents of the event or comment.
      */
    QString text;

    /**
      \brief The event time creation.
      */
    QDateTime created;

    /**
      \brief Attached object.
      */
    QAttachment attach;
};

QEventFeedData::QEventFeedData()
    : QSharedData()
{
}

QEventFeedData::QEventFeedData(const QEventFeedData& other)
    : QSharedData(other)
{
    accountId = other.accountId;
    attach = other.attach;
    created = other.created;
    ownerId = other.ownerId;
    ownerName = other.ownerName;
    text = other.text;
}

QEventFeedData::~QEventFeedData()
{
}


QEventFeed::QEventFeed()
{
    d = new QEventFeedData;
}

QEventFeed::QEventFeed(const QEventFeed &src)
{
    d = src.d;
}

QEventFeed::~QEventFeed()
{

}

QEventFeed& QEventFeed::operator =(QEventFeed const& src)
{
    d = src.d;
    return *this;
}

QString QEventFeed::typeToString(const FeedType type)
{
    QString name;

    switch(type)
    {
    case photoFeed:
        name = "image";
        break;
    case messageFeed:
        name = "message";
        break;
    case commentFeed:
        name = "comment";
        break;
    case videoFeed:
        name = "video";
        break;
    case audioFeed:
        name = "audio";
        break;
    case likeFeed:
        name = "like";
        break;
    case tagFeed:
        name = "tag";
        break;
    case friendFeed:
        name = "friend";
    }

    return name;
}

void QEventFeed::storeFeed(const QEventFeedList &list, const FeedType type, QString accountId)
{
    QString name = QEventFeed::typeToString(type);

    // store feed into cache
    QDomDocument out("MyDoc");
    QDomElement root = out.createElement(NODE_FEED_ROOT);
    root.setAttribute(ATTR_REFRESH, list.timeCreate().toString(DATE_TIME_FORMAT));
    out.appendChild(root);

    for (int i = 0; i < list.length(); i++) {
        root.appendChild(list[i].toQDomElement(out));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + "/" + QString(FILE_FEED_DATA).arg(name));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    //file.write(output.toUtf8());
    file.write(out.toByteArray(1));
    file.close();

}

QEventFeedList QEventFeed::loadFeed(QString accountId, FeedType type)
{
    QEventFeedList ret;

    QString name = QEventFeed::typeToString(type);

    QFile file(Utils::getAccountDir(accountId) + QString(FILE_FEED_DATA).arg(name));
    if (file.open(QIODevice::ReadOnly) == false)
        return ret;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    // get list updated time
    QDomElement root = doc.elementsByTagName(NODE_FEED_ROOT).at(0).toElement();
    QDateTime time_create = QDateTime::fromString(root.attribute(ATTR_REFRESH, ""), DATE_TIME_FORMAT);
    ret.setTimeCreate(time_create);

    // parse list of events
    QDomNodeList lst = doc.elementsByTagName(NODE_EVENT_ROOT);

    qDebug() << "Start load of " << lst.length() << " events from " << accountId;
    for (uint i = 0; i < lst.length(); i++) {
        ret.append(QEventFeed::fromQDomElement(lst.at(i).toElement(), accountId, type));
    }

    return ret;
}

QString QEventFeed::getKey(QString accountId, FeedType type)
{
    QString name = QEventFeed::typeToString(type);
    return QString("feed_%1_%2").arg(accountId).arg(name);
}

QDomElement QEventFeed::toQDomElement(QDomDocument &doc) const
{
    QDomElement ret = doc.createElement(NODE_EVENT_ROOT);

    // owner ID
    QDomElement node = doc.createElement(NODE_EVENT_OWNERID);
    QDomText t = doc.createTextNode(this->ownerId());
    ret.appendChild(node);
    node.appendChild(t);

    // owner name
    node = doc.createElement(NODE_EVENT_OWNER_NAME);
    t = doc.createTextNode(this->ownerName());
    ret.appendChild(node);
    node.appendChild(t);

    // text
    if (!this->text().isEmpty())
    {
        node = doc.createElement(NODE_EVENT_TEXT);
        t = doc.createTextNode(this->text());
        ret.appendChild(node);
        node.appendChild(t);
    }

    // created
    node = doc.createElement(NODE_EVENT_CREATED);
    t = doc.createTextNode(this->created().toString(DATE_TIME_FORMAT));
    ret.appendChild(node);
    node.appendChild(t);

    // attachment
    ret.appendChild(this->attach().toQDomElement(doc));

    return ret;
}

QEventFeed QEventFeed::fromQDomElement(const QDomElement& fr, const QString& accountId, const FeedType tp)
{
    QEventFeed ret;

    if (fr.elementsByTagName(NODE_EVENT_OWNERID).count() > 0)
        ret.setOwnerId(fr.elementsByTagName(NODE_EVENT_OWNERID).at(0).firstChild().toText().data());

    if (fr.elementsByTagName(NODE_EVENT_OWNER_NAME).count() > 0)
        ret.setOwnerName(fr.elementsByTagName(NODE_EVENT_OWNER_NAME).at(0).firstChild().toText().data());

    if (fr.elementsByTagName(NODE_EVENT_TEXT).count() > 0)
        ret.setText(fr.elementsByTagName(NODE_EVENT_TEXT).at(0).firstChild().toText().data());

    if (fr.elementsByTagName(NODE_EVENT_CREATED).count() > 0)
        ret.setCreated(QDateTime::fromString(fr.elementsByTagName(NODE_EVENT_CREATED).at(0).firstChild().toText().data(), DATE_TIME_FORMAT));

    ret.setAttach(QAttachment::fromQDomElement(fr.firstChildElement(NODE_ATTACH_ROOT), accountId));

    ret.setAccountId(accountId);

    return ret;
}

Photo QEventFeed::toPhoto() const
{
    Photo ret;
    ret.setAccountId(this->accountId());
    ret.setOwnerId(this->ownerId());
    ret.setAlbumId(this->attach().albumId());
    ret.setIcon(this->attach().icon());
    ret.setIcon_url(this->attach().iconUrl());
    ret.setPhoto(this->attach().image());
    ret.setPhoto_url(this->attach().objectUrl());
    ret.setTime_create(this->created().toString(DATE_TIME_FORMAT));
    ret.setPhotoId(this->attach().id());

    return ret;
}

// Getters
const QString QEventFeed::accountId() const { return d->accountId; }
const QString QEventFeed::ownerId() const { return d->ownerId; }
const QString QEventFeed::ownerName() const { return d->ownerName; }
const QString QEventFeed::text() const { return d->text; }
const QDateTime QEventFeed::created() const { return d->created; }
const QAttachment QEventFeed::attach() const { return d->attach; }

// Setters
void QEventFeed::setAccountId(const QString& v) { d->accountId = v; }
void QEventFeed::setOwnerId(const QString& v) { d->ownerId = v; }
void QEventFeed::setOwnerName(const QString& v) { d->ownerName = v; }
void QEventFeed::setText(const QString& v) { d->text = v; }
void QEventFeed::setCreated(const QDateTime& v) { d->created = v; }
void QEventFeed::setAttach(const QAttachment& v) { d->attach = v; }
