/**
  \file profilebrowserpanel.h
  \brief ProfileBrowserPanel class description
  \author MySocials team
  \date 2011-03-23
  */
#ifndef PROFILEBROWSERPANEL_H
#define PROFILEBROWSERPANEL_H

#include <QWidget>
#include <QRadioButton>
#include <QPushButton>
#include <QMessageBox>
#include <QListWidget>
#include <QDesktopWidget>

#include "widgets/qtoppanel.h"
#include "utils/hintedit.h"

#include "datatypes/message.h"
#include "datatypes/friend.h"
#include "datatypes/account.h"

#include "widgets/profilebrowser.h"

#include "servicemgr.h"

namespace Ui {
    class ProfileBrowserPanel;
}

/*!
 \class ProfileBrowserPanel
 \brief Panel for browsing friend profiles.

 This class is based on QWidget. It allows browsing friend profiles, going to
 their albums or sending messages to them. The list of friends gets from
 Service Manager (see \a ServiceMgr).

 \sa ServiceMgr
 */
class ProfileBrowserPanel : public QWidget
{
    Q_OBJECT
public:
    /*!
      A panel constructor.

      \param parent Pointer to parent widget.
     */
    explicit ProfileBrowserPanel(QWidget *parent = 0);

    /*!
      A panel constructor.

      \param sm Pointer to Service Manager.

      \param parent Pointer to parent widget.
     */
    explicit ProfileBrowserPanel(ServiceMgr *mgr = NULL, QWidget *parent = 0);

    //! A destructor.
    ~ProfileBrowserPanel();

    void clearEditOwnr();

    /*!
     \brief Selects friend and shows his profile.

     \param fd friend's profile
     */
    void showFriendInfo(Friend fd);

    /*!
     \brief Initialization function.

     Initializes UI elements of panel and connects signals to slots.
     */
    void init();

    /*!
     Hides specified layout and all layouts contained in it.

     \param layout Pointer to layout which must be hidden.
     */
    void hideAll(QLayout *layout);

    /*!
     Shows specified layout and all layouts contained in it.

     \param layout Pointer to layout which must be shown.
     */
    void showAll(QLayout *layout);

    /*!
     \brief States of window.

     It is using for single-panel mode for defining which layout must be shown
     at current moment.
     */
    enum WindowState {
        WindowFriendList,   /*!< \brief State when list of friends is shown. */
        WindowProfile       /*!< \brief State when profile for selected friend is shown. */
    };

    /*!
     \brief Sets state of window.

     \param state The new state of window.
     */
    void setState(WindowState state);

    /*!
     \brief Updates window corresponding current state.
     */
    void setState();

signals:
    /*!
     This signal is emitted when user selects a friend.

     \param profile of friend
     */
    void openProfile(Friend profile);

    /*!
     This signal is emitted when user click albums button

     \param fd profile of current friend
     */
    void showFriendAlbums(Friend fd);

    /*!
     This signal is emitted when user click sending message button

     \param fd profile of current Friend
     */
    void sendMessage(Friend fd);

    /*!
     This signal is emitted when user click messages button

     \param fd profile of current friend
     */
    void showFriendMessages(Friend fd);

public slots:

private:    
    Ui::ProfileBrowserPanel *ui;
    ProfileBrowser *pb;
    ServiceMgr *sm;
    HintEdit *editOwnr;

    QRadioButton *peopleradb;
    QPushButton *buttonEditOwnr;
    QPushButton *buttonRefreshFriends;
    QPushButton *buttonAlbums;
    QPushButton *buttonSendMsg;
    QPushButton *buttonBack;
    QPushButton *buttonMessages;
    QFont usedFont;

    bool isFriendsUpdate;
    bool isPortraitMode;

    bool isSkipFriendUpdate;

    // current state for window
    WindowState state;

    void ownrSelected(QListWidgetItem *item, bool isNeedUpdateState);

    QListWidgetItem *createFriendWidget(Friend fr);
    void addMeToFriends();

private slots:
    void refreshFriends();
    void refreshFriends(bool isNeedUpdate);
    void buttonResetOwnr_clicked();

    void editOwnr_textChanged(QString str);
    void ownrSelected(QListWidgetItem *item);

    void hideProfile();

    void showFriendAlbums();
    void showFriendMessages();
    void sendMessageToFriend();
    void orientationChanged();

    void updateListFriends(FriendList list, bool isLastUpdate);
    void updateProfile(Friend profile);
    void updateAccounts(AccountList list);
    void disableFriendSkipUpdate();
};

#endif // PROFILEBROWSERPANEL_H
