#include "messagebrowserpanel.h"
#include "ui_messagebrowserpanel.h"

MessageBrowserPanel::MessageBrowserPanel(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::MessageBrowserPanel),
    currentFriend(Friend()),
    isPortraitMode(false),
    state(WindowMessageList),
    msgState(MessagesInbox)
{
    this->sm = new ServiceMgr;
    ui->setupUi(this);

    init();

    this->isInboxMessagesUpdate = false;
    this->isOutboxMessagesUpdate = false;
    this->isFriendMessages = false;

    // connect to signals from Service Manager
    connect(sm, SIGNAL(updateInboxMessageList(MessageList,bool)), this,
            SLOT(updateListInboxMessages(MessageList,bool)));
    connect(sm, SIGNAL(updateOutboxMessageList(MessageList,bool)), this,
            SLOT(updateListOutboxMessages(MessageList,bool)));
    connect(sm, SIGNAL(updateAccounts(AccountList)), this,
            SLOT(updateAccounts(AccountList)));

    // check that one acount must be setup
    if (this->sm->getAccounts().length() != 0) {
        this->refreshListInboxMessages();
    }

    this->setState();
}

MessageBrowserPanel::MessageBrowserPanel(ServiceMgr *mgr, QWidget *parent) :
        QWidget(parent),
        ui(new Ui::MessageBrowserPanel),
        currentFriend(Friend()),
        isPortraitMode(false),
        state(WindowMessageList),
        msgState(MessagesOutbox)
{
    this->sm = mgr;
    ui->setupUi(this);

    init();

    this->isInboxMessagesUpdate = false;
    this->isOutboxMessagesUpdate = false;
    this->isFriendMessages = false;

    // connect to signals from Service Manager
    connect(sm, SIGNAL(updateInboxMessageList(MessageList, bool)), this,
            SLOT(updateListInboxMessages(MessageList, bool)));
    connect(sm, SIGNAL(updateOutboxMessageList(MessageList, bool)), this,
            SLOT(updateListOutboxMessages(MessageList, bool)));
    connect(sm, SIGNAL(updateAccounts(AccountList)), this,
            SLOT(updateAccounts(AccountList)));

    // check that one acount must be setup
    if (this->sm->getAccounts().length() != 0) {
        this->refreshListInboxMessages();;
    }

    this->setState();
}

MessageBrowserPanel::~MessageBrowserPanel()
{
    delete ui;
}

void MessageBrowserPanel::init()
{
    mb = new MessageBrowser(this);
    ui->layoutMessageBrowserBottom->addWidget(mb, 0);

#ifdef Q_WS_MAEMO_5
    ui->listMessages->setHeaderHidden(TRUE);
#endif

    buttonBack = new QPushButton(this);
    buttonBack->setIcon(QIcon::fromTheme("general_back",
                                         QIcon(":/res/general_back.png")));
    buttonBack->setFlat(TRUE);
    ui->layoutMessageBrowserTop->addWidget(buttonBack, 0);
    buttonBack->setVisible(FALSE);

    ui->layoutMessageBrowserTop->addWidget(buttonNewMsg = new QPushButton(this), 0);
    buttonNewMsg->setFlat(TRUE);
    buttonNewMsg->setIcon(QIcon(":/res/msg_new.png"));
    buttonNewMsg->setIconSize(QSize(24, 24));

    ui->layoutMessageBrowserTop->addWidget(buttonReply = new QPushButton(this), 0);
    buttonReply->setFlat(TRUE);
    buttonReply->setEnabled(FALSE);
    buttonReply->setIcon(QIcon(":/res/msg_reply.png"));
    buttonReply->setIconSize(QSize(24, 24));

    ui->layoutMessageBrowserTop->addWidget(buttonEditDraft = new QPushButton(this), 0);
    buttonEditDraft->setFlat(TRUE);
    buttonEditDraft->setIcon(QIcon(":/res/msg_edit.png"));
    buttonEditDraft->setEnabled(FALSE);
    buttonEditDraft->setIconSize(QSize(24, 24));

    ui->layoutMessageBrowserTop->addWidget(buttonDeleteMsg = new QPushButton(this), 0);
    buttonDeleteMsg->setFlat(TRUE);
    buttonDeleteMsg->setEnabled(FALSE);
    buttonDeleteMsg->setIcon(QIcon(":/res/msg_delete.png"));
    buttonDeleteMsg->setIconSize(QSize(24, 24));

    ui->layoutMessagesTop->addWidget(buttonInbox = new QPushButton(this), 0);
    buttonInbox->setFlat(TRUE);
    buttonInbox->setIcon(QIcon(":/res/inbox.png"));
    buttonInbox->setIconSize(QSize(24, 24));

    ui->layoutMessagesTop->addWidget(buttonOutbox = new QPushButton(this), 0);
    buttonOutbox->setFlat(TRUE);
    buttonOutbox->setIcon(QIcon(":/res/outbox.png"));
    buttonOutbox->setIconSize(QSize(24, 24));

    ui->layoutMessagesTop->addWidget(buttonDrafts = new QPushButton(this), 0);
    buttonDrafts->setFlat(TRUE);
    buttonDrafts->setIcon(QIcon(":/res/drafts.png"));
    buttonDrafts->setIconSize(QSize(24, 24));

    QStringList headers;
    headers << QString(tr("Title")) << QString(tr("Date"));
    ui->listMessages->setColumnCount(2);
    ui->listMessages->setHeaderLabels(headers);
    ui->listMessages->header()->setResizeMode(0, QHeaderView::Stretch);

    connect(buttonBack, SIGNAL(clicked()), this, SLOT(hideMessage()));
    connect(buttonInbox, SIGNAL(clicked()), this, SLOT(refreshListInboxMessages()));
    connect(buttonOutbox, SIGNAL(clicked()), this, SLOT(refreshListOutboxMessages()));
    connect(buttonDrafts, SIGNAL(clicked()), this, SLOT(showListDraftMessages()));
    connect(buttonDeleteMsg, SIGNAL(clicked()), this, SLOT(deleteMessageClicked()));
    connect(buttonEditDraft, SIGNAL(clicked()), this, SLOT(editDraftMessage()));
    connect(buttonNewMsg, SIGNAL(clicked()), this, SLOT(newMessageClicked()));
    connect(buttonReply, SIGNAL(clicked()), this, SLOT(replyMessageClicked()));
    connect(ui->listMessages, SIGNAL(itemClicked(QTreeWidgetItem*, int)), this,
            SLOT(messageSelected(QTreeWidgetItem*)));
    connect(ui->listMessages, SIGNAL(itemActivated(QTreeWidgetItem*,int)), this,
            SLOT(messageSelected(QTreeWidgetItem*)));

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
}

void MessageBrowserPanel::updateAccounts(AccountList )
{
    if (this->sm->getAccounts().length() != 0) {
        switch (this->msgState) {
        case MessagesInbox:
            refreshListInboxMessages();
            break;
        case MessagesOutbox:
            refreshListOutboxMessages();
            break;
        default:
            return;
        }
    }
}

void MessageBrowserPanel::updateListInboxMessages(MessageList list, bool isLastUpdate)
{
    qDebug() << "Start update Message list process. isLastUpdate=" << isLastUpdate;

    qDebug() << "list.count: " << list.count();

    if (isLastUpdate) {
        this->isInboxMessagesUpdate = false;
    } else {
        this->isInboxMessagesUpdate = true;
    }

    if (list.isEmpty()) {
        ui->listMessages->clear();
        QStringList row;
        if (isLastUpdate) {
            row << "No Messages" << "";
            new QTreeWidgetItem(ui->listMessages, row);
        } else {
            row << "Downloading message list...";
            new QTreeWidgetItem(ui->listMessages, row);
        }
        return;
    }

    MessageList newMessages(list);
    QList<QTreeWidgetItem *> dels;

    QTreeWidgetItem* root = ui->listMessages->invisibleRootItem();

    // check that all current messages are in new list
    for (int i = 0; i < root->childCount(); i++) {
        bool isFound = false;

        QTreeWidgetItem *item = root->child(i);
        QVariant v = item->data(0, Qt::UserRole);

        if (v.canConvert<Message>()) {
            Message oldMsg = v.value<Message>();

            int removedVal = -1;

            for (int j = 0; j < newMessages.length(); j++) {
                Message curMsg = newMessages.at(j);
                if (oldMsg.messageId == curMsg.messageId &&
                    oldMsg.accountId == curMsg.accountId) {
                    removedVal = j;

                    v.setValue(curMsg);
                    item->setData(0, Qt::UserRole, v);

                    isFound = true;
                    break;
                }
            }

            if (isFound && removedVal >= 0) {
                newMessages.removeAt(removedVal);
            } else {
                qDebug() << "prepare to remove message with id " << oldMsg.messageId << " from " << oldMsg.accountId;
            }
        }

        if (!isFound)
            dels.append(item);
    }

    qDebug() << "Will be add " << newMessages.length() << " messages and removed " << dels.length() << " records. Total widgets: " << root->childCount();

    // del old messages
    for (int i = 0; i < dels.length(); i++) {
        if (dels.at(i)->isSelected()) {
            root->child(0)->setSelected(true);
        }
        delete dels.at(i);
    }

    // add new Messages with asceding sorting
    for (int i = 0; i < newMessages.length(); i++) {
        qDebug() << "added new Message with id " << newMessages.at(i).messageId << " from " << newMessages.at(i).accountId;

        QTreeWidgetItem *item = createMessageWidget(newMessages.at(i));
        if (root->childCount() <= 1)
            root->addChild(item);

        bool flag = false;
        for (int j = 1; j < root->childCount(); j++) {
            if (root->child(j)->text(1).compare(item->text(1)) > 0) {
                root->insertChild(j, item);
                flag = true;
                break;
            }
        }
        if (!flag)
            root->addChild(item);
    }

    ui->listMessages->sortItems(1, Qt::AscendingOrder);

    if (isFriendMessages) {
        showFriendMessages(this->currentFriend);
    } else {
        this->ui->listMessages->clearSelection();
        root->child(0)->setSelected(true);
        this->ui->listMessages->selectedItems().append(root->child(0));
#ifndef Q_WS_MAEMO_5
        this->messageSelected(root->child(0));
#endif
    }
}

void MessageBrowserPanel::updateListOutboxMessages(MessageList list, bool isLastUpdate)
{
    qDebug() << "Start update Message list process. isLastUpdate=" << isLastUpdate;

    if (isLastUpdate) {
        this->isOutboxMessagesUpdate = false;
    } else {
        this->isOutboxMessagesUpdate = true;
    }

    if (list.isEmpty()) {
        ui->listMessages->clear();
        QStringList row;
        if (isLastUpdate) {
            row << "No Messages" << "";
            new QTreeWidgetItem(ui->listMessages, row);
        } else {
            row << "Downloading message list..." << "";
            new QTreeWidgetItem(ui->listMessages, row);
        }
        return;
    }

    MessageList newMessages(list);
    QList<QTreeWidgetItem *> dels;

    QTreeWidgetItem* root = ui->listMessages->invisibleRootItem();

    // check that all current messages are in new list
    for (int i = 0; i < root->childCount(); i++) {
        bool isFound = false;

        QTreeWidgetItem *item = root->child(i);
        QVariant v = item->data(0, Qt::UserRole);

        if (v.canConvert<Message>()) {
            Message oldMsg = v.value<Message>();

            int removedVal = -1;

            for (int j = 0; j < newMessages.length(); j++) {
                Message curMsg = newMessages.at(j);
                if (oldMsg.messageId == curMsg.messageId &&
                    oldMsg.accountId == curMsg.accountId) {
                    removedVal = j;

                    v.setValue(curMsg);
                    item->setData(0, Qt::UserRole, v);

                    isFound = true;
                    break;
                }
            }

            if (isFound && removedVal >= 0) {
                newMessages.removeAt(removedVal);
            } else {
                qDebug() << "prepare to remove message with id " << oldMsg.messageId << " from " << oldMsg.accountId;
            }
        }

        if (!isFound)
            dels.append(item);
    }

    qDebug() << "Will be add " << newMessages.length() << " messages and removed " << dels.length() << " records. Total widgets: " << root->childCount();

    // del old messages
    for (int i = 0; i < dels.length(); i++) {
        if (dels.at(i)->isSelected()) {
            root->child(0)->setSelected(true);
        }
        delete dels.at(i);
    }

    // add new Messages with asceding sorting
    for (int i = 0; i < newMessages.length(); i++) {
        qDebug() << "added new Message with id " << newMessages.at(i).messageId << " from " << newMessages.at(i).accountId;

        QTreeWidgetItem *item = createMessageWidget(newMessages.at(i));
        if (root->childCount() <= 1)
            root->addChild(item);

        bool flag = false;
        for (int j = 1; j < root->childCount(); j++) {
            if (root->child(j)->text(1).compare(item->text(1)) > 0) {
                root->insertChild(j, item);
                flag = true;
                break;
            }
        }
        if (!flag)
            root->addChild(item);
    }

    ui->listMessages->sortItems(1, Qt::AscendingOrder);

    if (isFriendMessages) {
        showFriendMessages(this->currentFriend);
    } else {
        this->ui->listMessages->clearSelection();
        root->child(0)->setSelected(true);
        this->ui->listMessages->selectedItems().append(root->child(0));
#ifndef Q_WS_MAEMO_5
        this->messageSelected(root->child(0));
#endif
    }
}

void MessageBrowserPanel::showListDraftMessages()
{
    qDebug() << __FILE__ << __FUNCTION__;

    setMessagesState(MessagesDrafts);

    QTreeWidgetItem* root = ui->listMessages->invisibleRootItem();
    this->ui->listMessages->clear();
    this->mb->resetMessage();

    AccountList list = this->sm->getAccounts();

    for (int i = 0; i < list.count(); i++) {
        MessageList listDrafts = list.at(i)->getDraftMessages();

        for (int j = 0; j < listDrafts.count(); j++) {
            QTreeWidgetItem* item = this->createMessageWidget(listDrafts.at(j));
            root->addChild(item);
        }
    }

    if (root->childCount() == 0) {
        QStringList row;
        row << "No saved messages" << "";
        new QTreeWidgetItem(ui->listMessages, row);
        return;
    } else {
        this->ui->listMessages->clearSelection();
        root->child(0)->setSelected(true);
        this->ui->listMessages->selectedItems().append(root->child(0));
#ifndef Q_WS_MAEMO_5
        this->messageSelected(root->child(0));
#endif
    }
}

void MessageBrowserPanel::messageSelected(QTreeWidgetItem *item)
{
    QVariant v = item->data(0, Qt::UserRole);
    if (v.canConvert<Message>()) {
        Message msg = v.value<Message>();

        if ((this->msgState != MessagesDrafts) && (msg.status.toInt() == 0)) {
            this->sm->readMessage(msg.accountId, msg.messageId);
            item->setIcon(0, QIcon(":/res/read_message.png"));
        }

        // recepient for VKontakte service
        if ((this->msgState == MessagesInbox) &&
            (msg.recepientList.count() == 0)) {
            AccountList list = this->sm->getAccounts();

            for (int i = 0; i < list.count(); i++) {
                if (list.at(i)->accountId() == msg.accountId) {
                    Friend profile = list.at(i)->getProfile(false);

                    Recepient rec;
                    rec.name = profile.name();
                    rec.id = profile.ownerId;
                    msg.recepientList.append(rec);
                }
            }
        }

        // sender for VKontakte service
        if ((this->msgState == MessagesOutbox) &&
            (msg.senderName.isEmpty())) {
            AccountList list = this->sm->getAccounts();

            for (int i = 0; i < list.count(); i++) {
                if (list.at(i)->accountId() == msg.accountId) {
                    Friend profile = list.at(i)->getProfile(false);

                    msg.senderId = profile.ownerId;
                    msg.senderName = profile.name();
                }
            }
        }

        mb->setMessage(msg);

        switch (msgState) {
        case MessagesInbox:
            this->buttonReply->setEnabled(TRUE);
            this->buttonEditDraft->setEnabled(FALSE);
            break;
        case MessagesOutbox:
            this->buttonReply->setEnabled(FALSE);
            this->buttonEditDraft->setEnabled(FALSE);
            break;
        case MessagesDrafts:
            this->buttonEditDraft->setEnabled(TRUE);
            this->buttonReply->setEnabled(FALSE);
        }

        this->buttonDeleteMsg->setEnabled(TRUE);
        this->setState(WindowMessage);

        emit openMessage(msg);
    }
}

void MessageBrowserPanel::refreshListInboxMessages()
{
    this->refreshListInboxMessages(true);
}

void MessageBrowserPanel::refreshListOutboxMessages()
{
    this->refreshListOutboxMessages(true);
}

void MessageBrowserPanel::refreshListInboxMessages(bool isNeedUpdate)
{
    qDebug() << __FILE__ << __FUNCTION__ << " isNeedUpdate = " << isNeedUpdate;

    this->mb->resetMessage();
    setMessagesState(MessagesInbox);

    if (!this->isInboxMessagesUpdate) {
        this->isInboxMessagesUpdate = true;
        sm->getInboxMessages(isNeedUpdate, true);
    }
}

void MessageBrowserPanel::refreshListOutboxMessages(bool isNeedUpdate)
{
    qDebug() << __FILE__ << __FUNCTION__ << " isNeedUpdate = " << isNeedUpdate;

    this->mb->resetMessage();
    setMessagesState(MessagesOutbox);

    if (!this->isOutboxMessagesUpdate) {
        this->isOutboxMessagesUpdate = true;
        sm->getOutboxMessages(isNeedUpdate, true);
    }
}

void MessageBrowserPanel::deleteMessageClicked()
{
    QList<QTreeWidgetItem*> selectedItems = ui->listMessages->selectedItems();

    for (int k = 0; k < selectedItems.count(); k++) {
        QTreeWidgetItem* item = selectedItems.at(k);
        if (item != NULL) {
            QVariant v = item->data(0, Qt::UserRole);
            if (v.canConvert<Message>()) {
                Message msg = v.value<Message>();

                if (msgState != MessagesDrafts) {
                    this->sm->deleteMessage(msg);

                    emit deleteMessage(msg);
                } else {
                    AccountList list = this->sm->getAccounts();

                    for (int i = 0; i < list.count(); i++) {
                        if (list.at(i)->accountId() == msg.accountId) {
                            MessageList list2 = list.at(i)->getDraftMessages();

                            for (int j = 0; j < list2.count(); j++) {
                                if (list2.at(j).messageId == msg.messageId)
                                    list2.removeAt(j);
                            }
                            list.at(i)->setDraftMessages(list2);
                            break;
                        }
                    }
                }
            }
        }
    }

    switch (msgState) {
    case MessagesInbox:
        refreshListInboxMessages();
        break;
    case MessagesOutbox:
        refreshListOutboxMessages();
        break;
    case MessagesDrafts:
        showListDraftMessages();
        break;
    default:
        return;
    }
}

void MessageBrowserPanel::newMessageClicked()
{
    emit newMessage();
}

void MessageBrowserPanel::replyMessageClicked()
{
    QTreeWidgetItem *item = ui->listMessages->selectedItems().first();
    if (item != NULL) {
        QVariant v = item->data(0, Qt::UserRole);
        if (v.canConvert<Message>()) {
            Message msg = v.value<Message>();

            emit replyMessage(msg);
        }
    }
}

void MessageBrowserPanel::editDraftMessage()
{
    QTreeWidgetItem *item = ui->listMessages->currentItem();
    if (item != NULL) {
        QVariant v = item->data(0, Qt::UserRole);
        if (v.canConvert<Message>()) {
            Message msg = v.value<Message>();

            emit editDraftMessage(msg);
        }
    }
}

void MessageBrowserPanel::setMessagesState(MessagesState state)
{
    this->msgState = state;    
}

void MessageBrowserPanel::setState()
{
    this->setState(this->state);
}

void MessageBrowserPanel::setState(WindowState state) {
    this->state = state;
    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;

    switch(this->state) {
    case WindowMessageList:
        showAll(ui->layoutMessages);
        hideAll(ui->layoutMessageBrowser);

        if (isMultiWindow) {
            showAll(ui->layoutMessageBrowser);
            buttonBack->setVisible(FALSE);
        } else {
            hideAll(ui->layoutMessageBrowser);
        }
        break;
    case WindowMessage:
        showAll(ui->layoutMessageBrowser);
        hideAll(ui->layoutMessages);

        if (isMultiWindow) {
            showAll(ui->layoutMessages);
            buttonBack->setVisible(FALSE);
        } else {
            hideAll(ui->layoutMessages);
            buttonBack->setVisible(TRUE);
        }
        break;
    default:
        QMessageBox::warning(this, tr("Error occured"), tr("Unknown window state"));
        break;
    }
}

void MessageBrowserPanel::hideAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            hideAll(layout->itemAt(i)->layout());
        } else {
            layout->itemAt(i)->widget()->setVisible(FALSE);
        }
    }
}
void MessageBrowserPanel::showAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            showAll(layout->itemAt(i)->layout());
        } else {
            layout->itemAt(i)->widget()->setVisible(TRUE);
        }
    }
}

void MessageBrowserPanel::hideMessage()
{
    this->setState(WindowMessageList);
}

QTreeWidgetItem* MessageBrowserPanel::createMessageWidget(Message msg)
{
    QStringList row;
    row << msg.subject() << msg.time;
    QTreeWidgetItem *item = new QTreeWidgetItem(row);

    // For Outbox and Drafts messages status is read as default
    if ((msg.status.toInt() == 1) || (msgState != MessagesInbox))
        item->setIcon(0, QIcon(":/res/read_message.png"));
    else
        item->setIcon(0, QIcon(":/res/message.png"));

    QVariant v;
    v.setValue(msg);
    item->setData(0, Qt::UserRole, v);

    return item;
}

void MessageBrowserPanel::showFriendMessages(Friend fd)
{
    this->isFriendMessages = true;
    this->currentFriend = fd;
    this->mb->resetMessage();

    buttonReply->setEnabled(FALSE);
    buttonDeleteMsg->setEnabled(FALSE);
    buttonEditDraft->setEnabled(FALSE);

    if (this->msgState == MessagesDrafts)
        return;

    QTreeWidgetItem* root = ui->listMessages->invisibleRootItem();

    bool isFirstItem = true;
    for (int i = 0; i < root->childCount(); i++) {
        QTreeWidgetItem *item = root->child(i);
        QVariant v = item->data(0, Qt::UserRole);

        if (v.canConvert<Message>()) {
            Message msg = v.value<Message>();

            bool isFound = false;
            for (int j = 0; j < msg.recepientList.count(); j++) {
                if (msg.recepientList.at(j).id == fd.ownerId)
                    isFound = true;
            }

            if ((msg.senderId != fd.ownerId) && !isFound) {
                item->setHidden(TRUE);
            } else {
                item->setHidden(FALSE);                

                if (isFirstItem)  {
                    this->ui->listMessages->clearSelection();
                    item->setSelected(TRUE);
#ifndef Q_WS_MAEMO_5
                    this->messageSelected(item);
#endif
                    isFirstItem = false;
                }
            }
        }
    }
}

void MessageBrowserPanel::showAllMessages()
{
    if (isFriendMessages) {
        this->isFriendMessages = false;

        QTreeWidgetItem* root = ui->listMessages->invisibleRootItem();

        for (int i = 0; i < root->childCount(); i++)
            root->child(i)->setHidden(FALSE);

        this->ui->listMessages->clearSelection();
        root->child(0)->setSelected(true);
        this->ui->listMessages->selectedItems().append(root->child(0));
        this->messageSelected(root->child(0));
    }
}

void MessageBrowserPanel::orientationChanged()
{
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
        this->isPortraitMode = false;
    else
        this->isPortraitMode = true;
}
