#include "account.h"

Account::Account(QString libraryName) : isLibraryInit(false)
{
    this->transport = new QTransport(libraryName, NULL);
    if (!(isLibraryInit = (this->transport!=NULL && this->transport->ready())))
        return;
    QObject::connect(this->transport, SIGNAL(settingsReceived(QString, QString)), this, SLOT(gotSettings(QString, QString)));
    this->isFirstRequest = true;
    this->isNetworkEnabled = false;
}

Account::Account(QString libraryName, QString accountId) : isLibraryInit(false)
{
    this->transport = new QTransport(libraryName, accountId);
    if (!(isLibraryInit = (this->transport!=NULL && this->transport->ready())))
        return;
    QObject::connect(this->transport, SIGNAL(settingsReceived(QString, QString)), this, SLOT(gotSettings(QString, QString)));
    this->isFirstRequest = true;
    this->isNetworkEnabled = false;
}

Account::~Account()
{
    if (this->transport != NULL) {
        this->transport->close();
        delete this->transport;
    }
}

bool Account::ready() const
{
    return isLibraryInit;
}

void Account::setProxy(const QString& proxyHost, const int proxyPort)
{
    this->transport->init(proxyHost, proxyPort);
}

// store settings to account
void Account::gotSettings(QString accountId, QString settings) {
    qDebug() << __FILE__ << ":" << __LINE__ << ": I got settings for " << accountId << ":" << settings;
    if (this->transport->accountId == accountId) {
        this->settings = settings;
        this->saveAccount();
    }
}

void Account::setProfile(const Friend& data)
{
    qDebug() << __FILE__ << ":" << __LINE__ <<  ": I got profile \"" << data.name() << "\" in " << data.objectName();
    this->isNetworkEnabled = true;
    this->profile = data;
    qDebug() << "SetUp profile for " << this->profile.name() << " = " << data.name();
}

void Account::saveAccount()
{
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_ACCOUNT_ROOT);
    out.appendChild(profile);

    QDomElement node = out.createElement(NODE_ACCOUNT_SETTINGS);
    if (this->settings.isEmpty()) {
        this->settings = this->oldSettings;
        qDebug() << "Uses old settings to store";
    }
    QDomText t = out.createTextNode(this->settings);
    profile.appendChild(node);
    node.appendChild(t);

    node = out.createElement(NODE_ACCOUNT_LIBRARY);
    t = out.createTextNode(this->transport->libraryName);
    profile.appendChild(node);
    node.appendChild(t);

    profile.appendChild(Friend::toQDomElement(out, this->profile, true));

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(this->transport->accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + "/" + QString(FILE_ACCOUNT_DATA));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    //file.write(output.toUtf8());
    file.write(out.toByteArray(1));
    file.close();
}

Account* Account::loadAccount(const QString& accountId)
{
    QFile file(Utils::getAccountDir(accountId) + QString(FILE_ACCOUNT_DATA));
    if (file.open(QIODevice::ReadOnly) == false)
        return NULL;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    QString libName = doc.documentElement().elementsByTagName(NODE_ACCOUNT_LIBRARY).at(0).firstChild().toText().data();

    Account *ret = new Account(libName, accountId);
    if (ret == NULL)
        return NULL;
    if (!ret->ready()) {
        delete ret;
        return NULL;
    }

    ret->oldSettings =  doc.documentElement().elementsByTagName(NODE_ACCOUNT_SETTINGS).at(0).firstChild().toText().data();

    if (!ret->oldSettings.isNull())
        ret->transport->setSettings(ret->oldSettings);

    ret->profile = Friend::fromQDomElement(doc.documentElement().elementsByTagName(NODE_FRIEND_ROOT).at(0).toElement(), accountId, ret->serviceName());

    return ret;
}

FriendList Account::getFriendList()
{
    if (!this->friends.isEmpty())
        return this->friends;

    // try to load friend list
    QFile file(Utils::getAccountDir(this->transport->accountId) + QString(FILE_FRIENDS_DATA));
    if (file.open(QIODevice::ReadOnly) == false)
        return this->friends;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    this->friends.clear();
    QDomNodeList lst = doc.elementsByTagName(NODE_FRIEND_ROOT);

    qDebug() << "Start load of " << lst.length() << " friends";
    for (uint i = 0; i < lst.length(); i++) {
        this->friends.append(Friend::fromQDomElement(lst.at(i).toElement(), this->accountId(), this->serviceName()));
    }

    return this->friends;
}

void Account::setFriendList(const FriendList& newFriends)
{
    this->friends = newFriends;

    // store friends into cache
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_FRIENDS_ROOT);
    out.appendChild(profile);

    for (int i = 0; i < newFriends.length(); i++) {
        profile.appendChild(Friend::toQDomElement(out, newFriends[i], false));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(this->transport->accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + "/" + QString(FILE_FRIENDS_DATA));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    //file.write(output.toUtf8());
    file.write(out.toByteArray(1));
    file.close();
}

AlbumList Account::getAlbumList()
{
    if (!this->albums.isEmpty())
        return this->albums;

    this->albums = this->profile.getAlbumList();
    return this->albums;
}

void Account::setAlbumList(const AlbumList& list)
{
    this->albums = list;
    this->profile.setAlbumList(list, this->transport->accountId, this->profile.ownerId);
}

MessageList Account::getInboxMessageList()
{
    if (!this->inboxMessages.isEmpty())
        return this->inboxMessages;

    // try to load message list
    QFile file(Utils::getAccountDir(this->transport->accountId) + QString(FILE_INBOX_DATA));
    if (file.open(QIODevice::ReadOnly) == false)
        return this->inboxMessages;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    this->inboxMessages.clear();
    QDomNodeList lst = doc.elementsByTagName(NODE_MESSAGE_ROOT);

    qDebug() << "Start load of " << lst.length() << " messages";
    for (uint i = 0; i < lst.length(); i++) {
        this->inboxMessages.append(Message::fromQDomElement(lst.at(i).toElement(), this->transport->accountId));
    }

    return this->inboxMessages;
}

void Account::setInboxMessageList(const MessageList& newMessages)
{
    this->inboxMessages = newMessages;

    // store friends into cache
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_MESSAGES_ROOT);
    out.appendChild(profile);

    for (int i = 0; i < newMessages.length(); i++) {
        profile.appendChild(Message::toQDomElement(out, newMessages[i]));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(this->transport->accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + QString(FILE_INBOX_DATA));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }

    file.write(out.toByteArray(1));
    file.close();
}

MessageList Account::getOutboxMessageList()
{
    if (!this->outboxMessages.isEmpty())
        return this->outboxMessages;

    // try to load message list
    QFile file(Utils::getAccountDir(this->transport->accountId) + QString(FILE_OUTBOX_DATA));
    if (file.open(QIODevice::ReadOnly) == false)
        return this->outboxMessages;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    this->outboxMessages.clear();
    QDomNodeList lst = doc.elementsByTagName(NODE_MESSAGE_ROOT);

    qDebug() << "Start load of " << lst.length() << " messages";
    for (uint i = 0; i < lst.length(); i++) {
        this->outboxMessages.append(Message::fromQDomElement(lst.at(i).toElement(), this->transport->accountId));
    }

    return this->outboxMessages;
}

void Account::setOutboxMessageList(const MessageList& newMessages)
{
    this->outboxMessages = newMessages;

    // store friends into cache
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_MESSAGES_ROOT);
    out.appendChild(profile);

    for (int i = 0; i < newMessages.length(); i++) {
        profile.appendChild(Message::toQDomElement(out, newMessages[i]));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(this->transport->accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + QString(FILE_OUTBOX_DATA));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }

    file.write(out.toByteArray(1));
    file.close();
}

void Account::setDraftMessages(const MessageList &newMessages)
{
    this->draftMessages = newMessages;

    QDomDocument out("MyDoc");
    QDomElement msgList = out.createElement(NODE_MESSAGES_ROOT);
    out.appendChild(msgList);

    for (int i = 0; i < newMessages.length(); i++) {
        msgList.appendChild(Message::toQDomElement(out, newMessages[i]));
    }

    QDir dir;

    dir.setPath(dir.cleanPath(Utils::getAccountDir(this->transport->accountId)));
    if (!dir.exists())
        dir.mkpath(dir.absolutePath());

    QFile file(dir.absolutePath() + QString(FILE_DRAFTS_DATA));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }

    file.write(out.toByteArray(1));
    file.close();
}

MessageList Account::getDraftMessages()
{
    if (!this->draftMessages.isEmpty())
        return this->draftMessages;

    // try to load message list
    QFile file(Utils::getAccountDir(this->transport->accountId) + QString(FILE_DRAFTS_DATA));
    if (file.open(QIODevice::ReadOnly) == false)
        return this->draftMessages;;

    QDomDocument doc;
    doc.setContent(&file);
    file.close();

    this->draftMessages.clear();
    QDomNodeList lst = doc.elementsByTagName(NODE_MESSAGE_ROOT);

    qDebug() << "Start load of " << lst.length() << " messages";
    for (uint i = 0; i < lst.length(); i++) {
        this->draftMessages.append(Message::fromQDomElement(lst.at(i).toElement(), this->transport->accountId));
    }

    return this->draftMessages;
}

Friend Account::getProfile(bool isNeedUpdate)
{
    // if first check was failed then return old profile
    if (!this->isFirstRequest && !this->isNetworkEnabled)
        return this->profile;

    if (this->profile.ownerId.isEmpty() || isNeedUpdate || this->isFirstRequest) {
        qDebug() << "Request for download profile for " << this->transport->accountId;
        this->isFirstRequest = false;
        QtConcurrent::run(this->transport, &QTransport::getProfile);
    }

    return this->profile;
}

QString Account::accountId() const
{
    if (this->transport != NULL)
        return this->transport->accountId;
    return QString::null;
}

QString Account::serviceName() const
{
    if (this->transport != NULL)
        return this->transport->serviceName();
    return QString::null;
}
