#include "mediaplayer.h"
#include "../../base/playbacklistmodel.h"
#include "../../base/session.h"
#include "../../base/soundcloud.h"
#include "../../base/notifications.h"
#include "../../base/settings.h"
#include "../../base/lastfm.h"
#include "../../base/database.h"
#include "../../base/utils.h"
#include <gst/gst.h>
#include <gst/gstelementfactory.h>
#include <gst/gstpipeline.h>
#include <QTimer>

MediaPlayer* MediaPlayer::self = 0;

MediaPlayer::MediaPlayer(QObject *parent) :
    QObject(parent),
    m_player(0),
    m_bus(0),
    m_busTimer(new QTimer(this)),
    m_index(0),
    m_position(0),
    m_duration(0),
    m_volume(0),
    m_muted(false),
    m_state(Media::Stopped),
    m_playbackMode(Media::Sequential)
{
    if (!self) {
        self = this;
    }

    gst_init(NULL, NULL);
    m_player = gst_element_factory_make("playbin", "play");
    m_bus = gst_pipeline_get_bus(GST_PIPELINE(m_player));
    m_busTimer->setInterval(500);
    m_busTimer->setSingleShot(false);

#ifdef MEDIA_PLAYERS_ENABLED
    this->connect(Settings::instance(), SIGNAL(mediaPlayerChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
#endif

#if (defined (MEDIA_PLAYERS_ENABLED) && (defined (QML_USER_INTERFACE)))
    this->onMediaPlayerChanged(Settings::instance()->mediaPlayer());
#else
    if (PlaybackListModel::playbackQueue()->rowCount() > 0) {
        this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(0));
    }
#endif

#ifdef MEEGO_EDITION_HARMATTAN
    m_resourceSet = new ResourcePolicy::ResourceSet("player", this);
    m_audioResource = new ResourcePolicy::AudioResource("player");
    m_audioResource->setProcessID(QCoreApplication::applicationPid());
    m_audioResource->setStreamTag("media.name", "*");
    m_audioResource->setOptional(false);
    m_resourceSet->addResourceObject(m_audioResource);

    this->connect(m_resourceSet, SIGNAL(resourcesGranted(QList<ResourcePolicy::ResourceType>)), this, SLOT(onResourcesGranted()));
    this->connect(m_resourceSet, SIGNAL(resourcesReleased()), this, SLOT(onResourcesReleased()));
    this->connect(m_resourceSet, SIGNAL(resourcesReleasedByManager()), this, SLOT(onResourcesReleased()));
    this->connect(m_resourceSet, SIGNAL(resourcesDenied()), this, SLOT(onResourcesDenied()));
    this->connect(m_resourceSet, SIGNAL(lostResources()), this, SLOT(onResourcesLost()));
#endif

    this->connect(m_busTimer, SIGNAL(timeout()), this, SLOT(checkBusForMessages()));
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
}

MediaPlayer::~MediaPlayer() {
#ifdef MEEGO_EDITION_HARMATTAN
    m_resourceSet->release();
#endif
    this->stop();
    gst_object_unref(GST_OBJECT(m_player));
    gst_object_unref(GST_OBJECT(m_bus));
}

MediaPlayer* MediaPlayer::instance() {
    return !self ? new MediaPlayer(Session::instance()) : self;
}

#ifdef QML_USER_INTERFACE
void MediaPlayer::playTrack(TrackItem *track, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTrackFromQML(track, false);
    this->setCurrentIndex(0, playImmediately);
}

void MediaPlayer::playTracks(QList<TrackItem*> tracks, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTracksFromQML(tracks, false);
    this->setCurrentIndex(0, playImmediately);
}
#else
void MediaPlayer::playTrack(QSharedPointer<TrackItem> track, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTrack(track, false);
    this->setCurrentIndex(0, playImmediately);
}

void MediaPlayer::playTracks(QList<QSharedPointer<TrackItem> > tracks, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTracks(tracks, false);
    this->setCurrentIndex(0, playImmediately);
}
#endif
void MediaPlayer::removeTrack(int row) {
    PlaybackListModel::playbackQueue()->removeTrack(row);

    if (row == this->currentIndex()) {
        if (PlaybackListModel::playbackQueue()->rowCount() > 0) {
            this->setCurrentIndex(row);
        }
    }
    else if (row < this->currentIndex()) {
        m_index--;
        emit currentIndexChanged(this->currentIndex());
    }
}

void MediaPlayer::clearTracks() {
    PlaybackListModel::playbackQueue()->clear();
}

void MediaPlayer::setCurrentTrack(QSharedPointer<TrackItem> track) {
    m_track = track;
#ifdef QML_USER_INTERFACE
    emit currentTrackChanged(track.data());
#else
    emit currentTrackChanged(track);
#endif
    if (track.isNull()) {
        return;
    }

    switch (track.data()->service()) {
    case Services::NoService:
        g_object_set(G_OBJECT(m_player), "uri", qPrintable(this->currentTrack().data()->url().toString()), NULL);

        if ((Settings::instance()->archiveOnline()) && (!track.data()->favourite()) && (!track.data()->id().isEmpty()) && (SoundCloud::instance()->userSignedIn())) {
            SoundCloud::instance()->trackIsFavourite(track.data()->id());
        }

        return;
    default:
        g_object_set(G_OBJECT(m_player), "uri", qPrintable(SoundCloud::instance()->getStreamUrl(this->currentTrack().data()->url()).toString()), NULL);
        return;
    }
}

void MediaPlayer::setCurrentIndex(int index, bool playImmediately) {
    m_index = index;
    emit currentIndexChanged(index);

    if ((index >= 0) && (index < PlaybackListModel::playbackQueue()->rowCount())) {
        this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(index));

        if (playImmediately) {
            this->play();
        }
    }
}

void MediaPlayer::play() {
#ifdef MEEGO_EDITION_HARMATTAN
    m_resourceSet->acquire();
#else
    gst_element_set_state(m_player, GST_STATE_PLAYING);
    this->startTick();
#endif
}

void MediaPlayer::togglePlayPause() {
    if (this->playing()) {
        this->pause();
    }
    else {
        this->play();
    }
}

void MediaPlayer::pause() {
    gst_element_set_state(m_player, GST_STATE_PAUSED);
    this->stopTick();
}

void MediaPlayer::stop() {
    gst_element_set_state(m_player, GST_STATE_NULL);
    this->stopTick();
}

void MediaPlayer::setPlaybackMode(Media::PlaybackMode mode) {
    if (mode != this->playbackMode()) {
        m_playbackMode = mode;
        emit playbackModeChanged(mode);
    }
}

void MediaPlayer::setSequentialPlaybackMode() {
    this->setPlaybackMode(Media::Sequential);
}

void MediaPlayer::setRepeatAllPlaybackMode() {
    this->setPlaybackMode(Media::RepeatAll);
}

void MediaPlayer::setRepeatOnePlaybackMode() {
    this->setPlaybackMode(Media::RepeatOne);
}

void MediaPlayer::togglePlaybackMode() {
    switch (this->playbackMode()) {
    case Media::Sequential:
        this->setPlaybackMode(Media::RepeatAll);
        return;
    case Media::RepeatAll:
        this->setPlaybackMode(Media::RepeatOne);
        return;
    default:
        this->setPlaybackMode(Media::Sequential);
        return;
    }
}

bool MediaPlayer::playing() const {
    switch (this->state()) {
    case Media::Playing:
        return true;
    default:
        return false;
    }
}

bool MediaPlayer::paused() const {
    switch (this->state()) {
    case Media::Paused:
        return true;
    default:
        return false;
    }
}

void MediaPlayer::next() {
    switch (this->playbackMode()) {
    case Media::RepeatAll:
        if (this->currentIndex() < PlaybackListModel::playbackQueue()->rowCount() - 1) {
            this->setCurrentIndex(this->currentIndex() + 1);
        }
        else {
            this->setCurrentIndex(0);
        }

        return;
    case Media::RepeatOne:
        this->play();
        return;
    default:
        if (this->currentIndex() < PlaybackListModel::playbackQueue()->rowCount() - 1) {
            this->setCurrentIndex(this->currentIndex() + 1);
        }

        return;
    }
}

void MediaPlayer::previous() {
    if (this->currentIndex() > 0) {
        this->setCurrentIndex(this->currentIndex() - 1);
    }
}

qint64 MediaPlayer::position() const {
    return m_position;
}

void MediaPlayer::setPosition(qint64 position) {
    if (position != this->position()) {
        m_position = position;
        emit positionChanged(position);
    }
}

qint64 MediaPlayer::duration() const {
    qint64 d = m_duration;

    if ((d <= 0) && (!this->currentTrack().isNull())) {
        d = this->currentTrack().data()->duration();
    }

    return d;
}

qreal MediaPlayer::volume() const {
    return m_volume;
}

void MediaPlayer::setVolume(qreal volume) {
    if (volume != this->volume()) {
        m_volume = volume;
        emit volumeChanged(volume);
    }
}

bool MediaPlayer::muted() const {
    return m_muted;
}

void MediaPlayer::setMuted(bool muted) {
    if (muted != this->muted()) {
        m_muted = muted;
        emit mutedChanged(muted);
    }
}

void MediaPlayer::toggleMuted() {
    this->setMuted(!this->muted());
}

void MediaPlayer::startTick() {
    m_busTimer->start();
}

void MediaPlayer::stopTick() {
    m_busTimer->stop();
}

Media::State MediaPlayer::state() const {
    return m_state;
}

void MediaPlayer::setState(Media::State state) {
    if (state != this->state()) {
        m_state = state;
        emit stateChanged(state);
    }
}

void MediaPlayer::checkBusForMessages() {
    GstMessage *message;
    gst_object_ref(m_bus);

    while((message = gst_bus_pop(m_bus)) != NULL) {
        switch (GST_MESSAGE_TYPE(message)) {
        case GST_MESSAGE_BUFFERING:
            // buffer progress
            this->onBufferProgressChanged(message);
            break;
        case GST_MESSAGE_SEGMENT_DONE:
            // update position
            this->onPositionChanged(message);
            break;
        case GST_MESSAGE_DURATION_CHANGED:
            // update duration
            this->onDurationChanged(message);
            break;
        case GST_MESSAGE_STATE_CHANGED:
            // update state
            this->onStateChanged(message);
            break;
        case GST_MESSAGE_EOS:
            this->onPlaybackFinished();
            break;
        case GST_MESSAGE_WARNING:
            // error
            this->onError(message);
            break;
        default:
            break;
        }
    }

    gst_message_unref(message);
    gst_object_unref(m_bus);
}

void MediaPlayer::onBufferProgressChanged(GstMessage *message) {
    gint percent;
    gst_message_parse_buffering(message, &percent);

    if (percent < 100) {
        this->pause();
        this->setState(Media::Buffering);
    }
    else {
        this->play();
    }

//    g_free(percent);
}

void MediaPlayer::onPositionChanged(GstMessage *message) {
    GstFormat *format;
    gint64 *position;
    gst_message_parse_segment_done(message, format, position);

//    switch (format) {
//    case GST_FORMAT_PERCENT:
//        qDebug() << "PostionChanged: Percent";
//        break;
//    case GST_FORMAT_TIME:
//        qDebug() << "PostionChanged: Time";
//        break;
//    case GST_FORMAT_BYTES:
//        qDebug() << "PostionChanged: Bytes";
//        break;
//    default:
//        qDebug() << "PostionChanged: Other";
//        break;
//    }

    qDebug() << position;

    gst_object_unref(GST_OBJECT(format));
    g_free(position);
}

void MediaPlayer::onDurationChanged(GstMessage *message) {
    Q_UNUSED(message)
}

void MediaPlayer::onStateChanged(GstMessage *message) {
    GstState *newState;
    gst_message_parse_state_changed(message, NULL, newState, NULL);

    switch (GST_STATE(newState)) {
    case GST_STATE_PLAYING:
        this->setState(Media::Playing);
        break;
    case GST_STATE_PAUSED:
        this->setState(Media::Paused);
#ifdef MEEGO_EDITION_HARMATTAN
        m_resourceSet->release();
#endif
        break;
    case GST_STATE_NULL:
        this->setState(Media::Stopped);
#ifdef MEEGO_EDITION_HARMATTAN
        m_resourceSet->release();
#endif
        break;
    default:
        break;
    }

    gst_object_unref(GST_OBJECT(newState));
}

void MediaPlayer::onPlaybackFinished() {
    this->onTrackPlayed(this->currentTrack());
    this->setPosition(0);
    this->next();
}

void MediaPlayer::onError(GstMessage *message) {
    GError *error;
    gchar *debug;
    gst_message_parse_warning(message, &error, &debug);

    Notifications::instance()->onError(QString("%1: %2").arg(tr("Playback error")).arg(QString(debug)));

    g_error_free(error);
    g_free(debug);
}

#ifdef MEDIA_PLAYERS_ENABLED
void MediaPlayer::onMediaPlayerChanged(const QString &mediaPlayer) {
    if (mediaPlayer != "musikloud") {
        this->stop();
#ifdef QML_USER_INTERFACE
        this->setCurrentIndex(0);
        this->setCurrentTrack(QSharedPointer<TrackItem>());
        this->disconnect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
#else
        this->deleteLater();
#endif
    }
#ifdef QML_USER_INTERFACE
    else {
        this->onCountChanged(PlaybackListModel::playbackQueue()->rowCount());
        this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    }
#endif
}
#endif

void MediaPlayer::onTrackPlayed(QSharedPointer<TrackItem> track) {
    track.data()->setPlayCount(track.data()->playCount() + 1);
    track.data()->setLastPlayed(Utils::currentMSecsSinceEpoch());

    switch (track.data()->service()) {
    case Services::NoService:
        QMetaObject::invokeMethod(Database::instance(), "updateTrack", Q_ARG(QUrl, track.data()->url()), Q_ARG(QString, "playCount"), Q_ARG(QVariant, track.data()->playCount()));
        QMetaObject::invokeMethod(Database::instance(), "updateTrack", Q_ARG(QUrl, track.data()->url()), Q_ARG(QString, "lastPlayed"), Q_ARG(QVariant, track.data()->lastPlayed()));
        break;
    default:
        break;
    }

    if ((Settings::instance()->scrobbleTracks()) && (Lastfm::instance()->userSignedIn()) && (!track.data()->artist().isEmpty())) {
        Lastfm::instance()->scrobbleTrack(track.data()->artist(), track.data()->title());
    }
}

void MediaPlayer::onCountChanged(int count) {
    switch (count) {
    case 0:
        this->stop();
        this->setCurrentIndex(0);
        this->setCurrentTrack(QSharedPointer<TrackItem>());
        return;
    default:
        if (this->currentTrack().isNull()) {
            this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(0));
        }

        return;
    }
}

#ifdef MEEGO_EDITION_HARMATTAN
void MediaPlayer::onResourcesGranted() {
    gst_element_set_state(m_player, GST_STATE_PLAYING);
    this->startTick();
}

void MediaPlayer::onResourcesReleased() {
    qDebug() << "Resources released";
    this->stop();
}

void MediaPlayer::onResourcesDenied() {
    qDebug() << "Resources denied";
    this->stop();
}

void MediaPlayer::onResourcesLost() {
    qDebug() << "Resources lost";
    this->stop();
}
#endif
