#include "homewindow.h"
#include "aboutdialog.h"
#include "settingsdialog.h"
#include "accountsdialog.h"
#include "searchdialog.h"
#include "exploredialog.h"
#include "querydialog.h"
#include "mychannelwindow.h"
#include "tracklistwindow.h"
#include "playlistlistwindow.h"
#include "playlisttrackswindow.h"
#include "trackinfowindow.h"
#include "userlistwindow.h"
#include "grouplistwindow.h"
#include "playbackqueuewindow.h"
#include "transferlistwindow.h"
#include "archivewindow.h"
#include "activitieswindow.h"
#include "nowplayingwindow.h"
#include "channelwindow.h"
#include "navbutton.h"
#include "nowplayingbutton.h"
#include "../base/feedurls.h"
#include "../base/soundcloud.h"
#include "windowstack.h"
#include "../base/settings.h"
#include "../base/clipboardmonitor.h"
#include "../mediaplayer/mobility/mediaplayer.h"
#include "../base/tracklauncher.h"
#include "../base/playbacklistmodel.h"
#include <QGridLayout>
#include <QAction>
#include <QMenuBar>

HomeWindow::HomeWindow(QWidget *parent) :
    StackedWindow(parent),
    m_myChannelButton(new NavButton(tr("You"), ":/images/you.png", this)),
    m_activitiesButton(new NavButton(tr("Dashboard"), ":/images/dashboard.png", this)),
    m_exploreButton(new NavButton(tr("Explore"), ":/images/explore.png", this)),
    m_searchButton(new NavButton(tr("Search"), ":/images/search.png", this)),
    m_queueButton(new NavButton(tr("Queue"), ":/images/queue.png", this)),
    m_transfersButton(new NavButton(tr("Transfers"), ":/images/transfers.png", this)),
    m_archiveButton(new NavButton(tr("Archive"), ":/images/archive.png", this)),
    m_accountsButton(new NavButton(tr("Accounts"), ":/images/accounts.png", this)),
    m_settingsButton(new NavButton(tr("Settings"), ":/images/settings.png", this)),
    m_nowPlayingButton(0),
    m_settingsAction(this->menuBar()->addAction(tr("Settings"), this, SLOT(showSettingsDialog()))),
    m_aboutAction(this->menuBar()->addAction(tr("About"), this, SLOT(showAboutDialog()))),
    m_grid(0)
{    
    this->setWindowTitle("MusiKloud");
    this->setAttribute(Qt::WA_DeleteOnClose, false);
    this->setCentralWidget(new QWidget(this));
    m_grid = new QGridLayout(this->centralWidget());
    m_grid->setContentsMargins(0, 0, 0, 0);

    this->connect(m_myChannelButton, SIGNAL(clicked()), this, SLOT(showMyChannel()));
    this->connect(m_activitiesButton, SIGNAL(clicked()), this, SLOT(showActivities()));
    this->connect(m_exploreButton, SIGNAL(clicked()), this, SLOT(showExploreDialog()));
    this->connect(m_searchButton, SIGNAL(clicked()), this, SLOT(showSearchDialog()));
    this->connect(m_queueButton, SIGNAL(clicked()), this, SLOT(showPlaybackQueue()));
    this->connect(m_transfersButton, SIGNAL(clicked()), this, SLOT(showTransfers()));
    this->connect(m_archiveButton, SIGNAL(clicked()), this, SLOT(showArchive()));
    this->connect(m_accountsButton, SIGNAL(clicked()), this, SLOT(showAccountsDialog()));
    this->connect(m_settingsButton, SIGNAL(clicked()), this, SLOT(showSettingsDialog()));
    this->connect(ClipboardMonitor::instance(), SIGNAL(gotClipboardUrl(QString)), this, SLOT(displayResource(QString)));

    this->onMediaPlayerChanged(Settings::instance()->mediaPlayer());
    this->onOrientationChanged();
}

bool HomeWindow::event(QEvent *event) {
    switch (event->type()) {
    case QEvent::ApplicationDeactivate:
        Settings::instance()->saveSettings();
        break;
    default:
        break;
    }

    return StackedWindow::event(event);
}

void HomeWindow::setLandscapeLayout() {
    m_settingsButton->hide();
    m_settingsAction->setEnabled(true);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_myChannelButton, 0, 0);
        m_grid->addWidget(m_activitiesButton, 0, 1);
        m_grid->addWidget(m_exploreButton, 0, 2);
        m_grid->addWidget(m_searchButton, 0, 3);
        m_grid->addWidget(m_queueButton, 1, 0);
        m_grid->addWidget(m_transfersButton, 1, 1);
        m_grid->addWidget(m_archiveButton, 1, 2);
        m_grid->addWidget(m_accountsButton, 1, 3);
    }
    else {
        m_grid->removeWidget(m_searchButton);
        m_grid->removeWidget(m_queueButton);
        m_grid->removeWidget(m_transfersButton);
        m_grid->removeWidget(m_archiveButton);
        m_grid->removeWidget(m_accountsButton);
        m_grid->addWidget(m_searchButton, 0, 3);
        m_grid->addWidget(m_queueButton, 1, 0);
        m_grid->addWidget(m_transfersButton, 1, 1);
        m_grid->addWidget(m_archiveButton, 1, 2);
        m_grid->addWidget(m_accountsButton, 1, 3);
    }
}

void HomeWindow::setPortraitLayout() {
    m_settingsButton->show();
    m_settingsAction->setEnabled(false);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_myChannelButton, 0, 0);
        m_grid->addWidget(m_activitiesButton, 0, 1);
        m_grid->addWidget(m_exploreButton, 0, 2);
        m_grid->addWidget(m_searchButton, 1, 0);
        m_grid->addWidget(m_queueButton, 1, 1);
        m_grid->addWidget(m_transfersButton, 1, 2);
        m_grid->addWidget(m_archiveButton, 2, 0);
        m_grid->addWidget(m_accountsButton, 2, 1);
        m_grid->addWidget(m_settingsButton, 2, 2);
    }
    else {
        m_grid->removeWidget(m_searchButton);
        m_grid->removeWidget(m_queueButton);
        m_grid->removeWidget(m_transfersButton);
        m_grid->removeWidget(m_archiveButton);
        m_grid->removeWidget(m_accountsButton);
        m_grid->removeWidget(m_settingsButton);
        m_grid->addWidget(m_searchButton, 1, 0);
        m_grid->addWidget(m_queueButton, 1, 1);
        m_grid->addWidget(m_transfersButton, 1, 2);
        m_grid->addWidget(m_archiveButton, 2, 0);
        m_grid->addWidget(m_accountsButton, 2, 1);
        m_grid->addWidget(m_settingsButton, 2, 2);
    }
}

void HomeWindow::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(this);
    dialog->open();
}

void HomeWindow::showAboutDialog() {
    AboutDialog *dialog = new AboutDialog(this);
    dialog->open();
}

void HomeWindow::showAccountsDialog() {
    AccountsDialog *dialog = new AccountsDialog(this);
    dialog->open();
}

void HomeWindow::showMyChannel() {
    if (SoundCloud::instance()->userSignedIn()) {
        MyChannelWindow *window = new MyChannelWindow(this);
        window->show();
    }
    else {
        this->showSignInDialog();
    }
}

void HomeWindow::showActivities() {
    if (SoundCloud::instance()->userSignedIn()) {
        ActivitiesWindow *window = new ActivitiesWindow(this);
        window->show();
    }
    else {
        this->showSignInDialog();
    }
}

void HomeWindow::showExploreDialog() {
    ExploreDialog *dialog = new ExploreDialog(this);
    dialog->open();
    this->connect(dialog, SIGNAL(categoryChosen(Queries::QueryType,QString,QUrl)), this, SLOT(showExploreWindow(Queries::QueryType,QString,QUrl)));
}

void HomeWindow::showExploreWindow(Queries::QueryType type, const QString &title, const QUrl &feed) {
    switch (type) {
    case Queries::Tracks:
    {
        TrackListWindow *tracksWindow = new TrackListWindow(this);
        tracksWindow->setTrackFeed(feed);
        tracksWindow->setWindowTitle(title);
        tracksWindow->show();
    }
        return;
    case Queries::Users:
    {
        UserListWindow *usersWindow = new UserListWindow(this);
        usersWindow->setUserFeed(feed);
        usersWindow->setWindowTitle(title);
        usersWindow->show();
    }
        return;
    case Queries::Groups:
    {
        GroupListWindow *groupsWindow = new GroupListWindow(this);
        groupsWindow->setGroupFeed(feed);
        groupsWindow->setWindowTitle(title);
        groupsWindow->show();
    }
        return;
    default:
        qWarning() << "HomeWindow::showExploreWindow(): Invalid type";
        return;
    }
}

void HomeWindow::showSearchDialog() {
    SearchDialog *dialog = new SearchDialog(this);
    dialog->open();

    this->connect(dialog, SIGNAL(search(QString,Queries::QueryType,Queries::QueryOrder)), this, SLOT(search(QString,Queries::QueryType,Queries::QueryOrder)));
    this->connect(dialog, SIGNAL(fromUrl(QString)), this, SLOT(displayResource(QString)));
}

void HomeWindow::search(const QString &query, Queries::QueryType type, Queries::QueryOrder order) {
    switch (type) {
    case Queries::Tracks:
        this->trackSearch(query, order);
        return;
    case Queries::Playlists:
        this->playlistSearch(query, order);
        return;
    case Queries::Users:
        this->userSearch(query);
        return;
    case Queries::Groups:
        this->groupSearch(query, order);
        return;
    default:
        qWarning() << "HomeWindow::search(): Invalid query type.";
        return;
    }
}

void HomeWindow::trackSearch(const QString &query, Queries::QueryOrder order) {
    TrackListWindow *window = new TrackListWindow(this);
    window->search(query, order);
    window->show();
}

void HomeWindow::playlistSearch(const QString &query, Queries::QueryOrder order) {
    PlaylistListWindow *window = new PlaylistListWindow(this);
    window->search(query, order);
    window->show();
}

void HomeWindow::groupSearch(const QString &query, Queries::QueryOrder order) {
    GroupListWindow *window = new GroupListWindow(this);
    window->search(query, order);
    window->show();
}

void HomeWindow::userSearch(const QString &query) {
    UserListWindow *window = new UserListWindow(this);
    window->search(query);
    window->show();
}

void HomeWindow::displayTrack(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->displayTrack(urls.first());
    }
}

void HomeWindow::displayTrack(const QString &url) {
    WindowStack::instance()->unwindWindowStack();
    this->activateWindow();

    SoundCloud::instance()->getFullTrack(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(showTrack(QSharedPointer<TrackItem>)));
}

void HomeWindow::playTrack(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->playTrack(urls.first());
    }
}

void HomeWindow::playTrack(const QString &url) {
    WindowStack::instance()->unwindWindowStack();
    this->activateWindow();

    SoundCloud::instance()->getFullTrack(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(playTrack(QSharedPointer<TrackItem>)));
}

void HomeWindow::displayPlaylist(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->displayPlaylist(urls.first());
    }
}

void HomeWindow::displayPlaylist(const QString &url) {
    WindowStack::instance()->unwindWindowStack();
    this->activateWindow();

    SoundCloud::instance()->getFullPlaylist(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotPlaylist(QSharedPointer<PlaylistItem>)), this, SLOT(showPlaylist(QSharedPointer<PlaylistItem>)));
}

void HomeWindow::displayUser(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->displayUser(urls.first());
    }
}

void HomeWindow::displayUser(const QString &url) {
    WindowStack::instance()->unwindWindowStack();
    this->activateWindow();

    SoundCloud::instance()->resolve(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotResourceId(QString,Queries::QueryType)), this, SLOT(showUserChannel(QString)));
}

void HomeWindow::showTrack(QSharedPointer<TrackItem> track) {
    TrackInfoWindow *window = new TrackInfoWindow(track, this);
    window->show();
}

void HomeWindow::playTrack(QSharedPointer<TrackItem> track) {
    if (Settings::instance()->mediaPlayer() == "musikloud") {
        MediaPlayer::instance()->playTrack(track);
        NowPlayingWindow *window = new NowPlayingWindow(this);
        window->show();
    }
    else {
        TrackLauncher::launchTrack(SoundCloud::instance()->getStreamUrl(track.data()->streamUrl()));
    }
}

void HomeWindow::showPlaylist(QSharedPointer<PlaylistItem> playlist) {
    PlaylistTracksWindow *window = new PlaylistTracksWindow(playlist, this);
    window->show();
}

void HomeWindow::showUserChannel(const QString &id) {
    ChannelWindow *window = new ChannelWindow(id, this);
    window->show();
}

void HomeWindow::displayResource(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->displayResource(urls.first());
    }
}

void HomeWindow::displayResource(const QString &url) {
    WindowStack::instance()->unwindWindowStack();
    this->activateWindow();

    SoundCloud::instance()->resolve(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotResourceId(QString,Queries::QueryType)), this, SLOT(onResourceResolved(QString,Queries::QueryType)));
}

void HomeWindow::onResourceResolved(const QString &id, Queries::QueryType type) {
    switch (type) {
    case Queries::Tracks:
        this->displayTrack(id);
        return;
    case Queries::Playlists:
        this->displayPlaylist(id);
        return;
    case Queries::Users:
        this->showUserChannel(id);
        return;
    default:
        qWarning() << "HomeWindow::onResourceResolved(): Invalid resource type.";
        return;
    }
}

void HomeWindow::showPlaybackQueue() {
    if ((Settings::instance()->mediaPlayer() == "musikloud") && (PlaybackListModel::playbackQueue()->rowCount() > 0)) {
        NowPlayingWindow *nowPlayingWindow = new NowPlayingWindow(this);
        nowPlayingWindow->show();
    }
    else {
        PlaybackQueueWindow *queueWindow = new PlaybackQueueWindow(this);
        queueWindow->show();
    }
}

void HomeWindow::showTransfers() {
    TransferListWindow *window = new TransferListWindow(this);
    window->show();
}

void HomeWindow::showArchive() {
    ArchiveWindow *window = new ArchiveWindow(this);
    window->show();
}

void HomeWindow::showSignInDialog() {
    QueryDialog *dialog = new QueryDialog(this);
    dialog->setWindowTitle(tr("Sign in?"));
    dialog->setMessage(tr("You are not signed in to an account. Would you like to sign in now?"));
    dialog->open();

    this->connect(dialog, SIGNAL(accepted()), this, SLOT(showAccountsDialog()));
}

void HomeWindow::onMediaPlayerChanged(const QString &mediaPlayer) {
    if (mediaPlayer == "musikloud") {
        if (!m_nowPlayingButton) {
            m_nowPlayingButton = new NowPlayingButton(this);
        }

        m_nowPlayingButton->poke();
    }
    else if (m_nowPlayingButton) {
        m_nowPlayingButton->deleteLater();
        m_nowPlayingButton = 0;
    }
}
