#include "archivetrackstab.h"
#include "listview.h"
#include "filterbox.h"
#include "querydialog.h"
#include "nowplayingwindow.h"
#include "nowplayingbutton.h"
#include "archivetrackdelegate.h"
#include "trackinfowindow.h"
#include "trackselectionwindow.h"
#include "windowstack.h"
#include "../base/archivetrackmodel.h"
#include "../base/tracklistmodel.h"
#include "../base/playbacklistmodel.h"
#include "../base/settings.h"
#include "../base/tracklauncher.h"
#include "../mediaplayer/mobility/mediaplayer.h"
#include <QLabel>
#include <QMenu>
#include <QAction>
#include <QVBoxLayout>
#include <QMaemo5Style>

ArchiveTracksTab::ArchiveTracksTab(QWidget *parent) :
    AbstractTab(parent),
    m_model(new ArchiveTrackModel(this)),
    m_view(new ListView(this)),
    m_filterBox(new FilterBox(this)),
    m_queueAction(new QAction(tr("Add to playback queue"), this)),
    m_deleteAction(new QAction(tr("Delete"), this)),
    m_contextMenu(new QMenu(this)),
    m_contextQueueAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueTrack()))),
    m_contextDeleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(showConfirmDeleteDialog()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No tracks found")), this)),
    m_vbox(new QVBoxLayout(this))
{
    this->addAction(m_queueAction);
    this->addAction(m_deleteAction);

    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);
    m_vbox->addWidget(m_filterBox);
    m_filterBox->hide();
    m_noResultsLabel->hide();
    m_view->setModel(m_model);
    m_view->setMonitorScrolling(true);

    ArchiveTrackDelegate *delegate = new ArchiveTrackDelegate(m_view);
    m_view->setItemDelegate(delegate);

    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playTrack(QModelIndex)));
    this->connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    this->connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));
    this->connect(m_queueAction, SIGNAL(triggered()), this, SLOT(queueTracks()));
    this->connect(m_deleteAction, SIGNAL(triggered()), this, SLOT(deleteTracks()));

    if (Settings::instance()->mediaPlayer() == "musikloud") {
        this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
        this->connect(m_view, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
    }

    m_model->getTracks();
    this->onOrientationChanged();
}

void ArchiveTracksTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void ArchiveTracksTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void ArchiveTracksTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible(!count);
}

void ArchiveTracksTab::onItemClicked(const QModelIndex &index) {
    TrackInfoWindow *window = new TrackInfoWindow(m_model->get(index.row()), WindowStack::instance()->currentWindow());
    window->show();
}

void ArchiveTracksTab::playTrack(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "musikloud") {
        MediaPlayer::instance()->playTrack(m_model->get(index.row()));
        NowPlayingWindow *window = new NowPlayingWindow(this);
        window->show();
    }
    else {
        TrackLauncher::launchTrack(index.data(TrackListModel::UrlRole).toUrl());
    }
}

void ArchiveTracksTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_contextQueueAction);
}

void ArchiveTracksTab::queueTrack() {
    PlaybackListModel::playbackQueue()->addTrack(m_model->get(m_view->currentIndex().row()));
}

void ArchiveTracksTab::showConfirmDeleteDialog() {
    QueryDialog *dialog = new QueryDialog(QString("%1 '%2'?").arg(tr("Do you wish to delete")).arg(m_model->data(m_view->currentIndex(), TrackListModel::TitleRole).toString()), this);
    dialog->setWindowTitle(tr("Delete?"));
    dialog->open();
    this->connect(dialog, SIGNAL(accepted()), this, SLOT(deleteTrack()));
}

void ArchiveTracksTab::deleteTrack() {
    m_model->deleteTrack(m_view->currentIndex().row());
}

void ArchiveTracksTab::queueTracks() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Add to queue"), SLOT(queueTracks()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_view->indexAt(QPoint(0, 0)));
    window->show();
}

void ArchiveTracksTab::deleteTracks() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Add to queue"), SLOT(deleteTracks()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_view->indexAt(QPoint(0, 0)));
    window->show();
}

void ArchiveTracksTab::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        this->showFilterBox();
        m_filterBox->setFilterText(event->text());
    }
}

void ArchiveTracksTab::showFilterBox() {
    m_filterBox->show();
    m_filterBox->setFocus(Qt::OtherFocusReason);
}
