#include "tracklistwindow.h"
#include "tracklistdelegate.h"
#include "trackinfowindow.h"
#include "mainwindow.h"
#include "../base/tracklistmodel.h"
#include "../base/playbacklistmodel.h"
#include "../base/transfermanager.h"
#include "../base/soundcloud.h"
#include "../mediaplayer/gstreamer/mediaplayer.h"
#include <QListView>
#include <QMenu>
#include <QVBoxLayout>

TrackListWindow::TrackListWindow(QWidget *parent) :
    QWidget(parent),
    m_model(new TrackListModel(this)),
    m_view(new QListView(this)),
    m_contextMenu(new QMenu(this)),
    m_viewAction(m_contextMenu->addAction(tr("View"), this, SLOT(showSelectedTrack()))),
    m_playAction(m_contextMenu->addAction(tr("Play"), this, SLOT(playSelectedTrack()))),
    m_queueAction(m_contextMenu->addAction(tr("Queue"), this, SLOT(queueSelectedTrack()))),
    m_downloadAction(m_contextMenu->addAction(tr("Download"), this, SLOT(downloadSelectedTrack()))),
    m_favouriteAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(toggleFavouriteSelectedTrack()))),
    m_separatorAction(m_contextMenu->addSeparator()),
    m_moreResultsAction(m_contextMenu->addAction(tr("Load more tracks"), m_model, SLOT(getMoreTracks()))),
    m_vbox(new QVBoxLayout(this))
{
    this->setAttribute(Qt::WA_DeleteOnClose);
    this->setFocusProxy(m_view);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new TrackListDelegate(m_view));
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);

    m_favouriteAction->setEnabled(SoundCloud::instance()->userSignedIn());

    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);

    this->connect(m_view, SIGNAL(activated(QModelIndex)), this, SLOT(onListItemActivated(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
}

TrackListWindow::~TrackListWindow() {}

void TrackListWindow::setTrackFeed(const QUrl &feed) {
    m_model->getTracks(feed);
}

void TrackListWindow::getGroupTracks(QSharedPointer<GroupItem> group) {
    this->setWindowTitle(group.data()->title());
    m_model->getGroupTracks(group.data()->id());
}

void TrackListWindow::search(const QString &query, Queries::QueryOrder order) {
    this->setWindowTitle(QString("%1 ('%2')").arg(tr("Search")).arg(query));
    m_model->search(query, order);
}

void TrackListWindow::onListItemActivated(const QModelIndex &index) {
    if (index.isValid()) {
        TrackInfoWindow *window = new TrackInfoWindow(m_model->get(index.row()), MainWindow::instance());
        MainWindow::instance()->push(window);
    }
}

void TrackListWindow::showContextMenu(const QPoint &pos) {
    if (m_view->currentIndex().isValid()) {
        m_favouriteAction->setText(m_view->currentIndex().data(TrackListModel::FavouriteRole).toBool() ? tr("Delete from favourites")
                                                                                                       : tr("Add to favourites"));
        m_moreResultsAction->setEnabled(m_model->moreResults());
        m_contextMenu->setActiveAction(m_viewAction);
        m_contextMenu->popup(this->mapToGlobal(pos), m_viewAction);
    }
}

void TrackListWindow::showSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        TrackInfoWindow *window = new TrackInfoWindow(m_model->get(m_view->currentIndex().row()), MainWindow::instance());
        MainWindow::instance()->push(window);
    }
}

void TrackListWindow::playSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        MediaPlayer::instance()->playTrack(m_model->get(m_view->currentIndex().row()));
    }
}

void TrackListWindow::queueSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        PlaybackListModel::playbackQueue()->addTrack(m_model->get(m_view->currentIndex().row()));
    }
}

void TrackListWindow::downloadSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        TransferManager::instance()->addDownloadTransfer(m_model->get(m_view->currentIndex().row()));
    }
}

void TrackListWindow::toggleFavouriteSelectedTrack() {
    if (m_view->currentIndex().isValid()) {
        if (m_view->currentIndex().data(TrackListModel::FavouriteRole).toBool()) {
            SoundCloud::instance()->deleteFromFavourites(QStringList(m_view->currentIndex().data(TrackListModel::IdRole).toString()));
        }
        else {
            SoundCloud::instance()->addToFavourites(QStringList(m_view->currentIndex().data(TrackListModel::IdRole).toString()));
        }
    }
}
