#include "settingsdialog.h"
#include "../base/settings.h"
#include "../base/lastfm.h"
#include "../base/selectionmodels.h"
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QCheckBox>
#include <QComboBox>
#include <QDialogButtonBox>
#include <QFileDialog>
#include <QTabWidget>
#include <QGridLayout>

SettingsDialog::SettingsDialog(QWidget *parent) :
    QDialog(parent),
    m_folderEdit(new QLineEdit(this)),
    m_folderButton(new QPushButton(tr("Browse"), this)),
    m_subfoldersCheckBox(new QCheckBox(tr("Create artist subfolders"), this)),
    m_clipboardCheckBox(new QCheckBox(tr("Monitor clipboard for URLs"), this)),
    m_scrobbleCheckBox(new QCheckBox(tr("Scrobble tracks to Last.fm"), this)),
    m_onlineArchiveCheckBox(new QCheckBox(tr("Online access in archive"), this)),
    m_transfersCheckBox(new QCheckBox(tr("Start transfers automatically"), this)),
    m_formatComboBox(new QComboBox(this)),
    m_transfersComboBox(new QComboBox(this)),
    m_languageComboBox(new QComboBox(this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Save | QDialogButtonBox::Cancel, Qt::Horizontal, this)),
    m_tabWidget(new QTabWidget(this)),
    m_mediaTab(new QWidget(this)),
    m_transfersTab(new QWidget(this)),
    m_otherTab(new QWidget(this)),
    m_vbox(new QVBoxLayout(this))
{
    this->setWindowTitle(tr("Settings"));
    this->setAttribute(Qt::WA_DeleteOnClose);

    LanguageModel *languageModel = new LanguageModel(m_languageComboBox);

    QDir dir("/opt/cutetube/translations");
    QStringList availableLanguages = dir.entryList(QStringList("*.qm"), QDir::Files);
    QList<int> rowsToRemove;

    for (int i = 0; i < languageModel->rowCount(); i++) {
        QString lang = languageModel->value(i).toString();

        if (!availableLanguages.contains(QString("musikloud_%1.qm").arg(lang))) {
            rowsToRemove.prepend(i);
        }
        else {
            availableLanguages.removeOne(lang);
        }
    }

    while (!rowsToRemove.isEmpty()) {
        languageModel->removeRows(rowsToRemove.takeFirst(), 1);
    }

    m_formatComboBox->setModel(new DownloadFormatModel(m_formatComboBox));
    m_transfersComboBox->setModel(new ConcurrentTransfersModel(m_transfersComboBox));
    m_languageComboBox->setModel(languageModel);

    QGridLayout *mediaGrid = new QGridLayout(m_mediaTab);
    mediaGrid->addWidget(new QLabel(tr("Download folder") + ":", this), 0, 0);
    mediaGrid->addWidget(m_folderEdit, 0, 1);
    mediaGrid->addWidget(m_folderButton, 0, 2);
    mediaGrid->addWidget(m_subfoldersCheckBox, 1, 0, 1, 3);
    mediaGrid->addWidget(new QLabel(tr("Download format") + ":", this), 2, 0);
    mediaGrid->addWidget(m_formatComboBox, 2, 1, 1, 2);
    mediaGrid->addWidget(m_clipboardCheckBox, 3, 0, 1, 3);
    mediaGrid->addWidget(m_scrobbleCheckBox, 4, 0, 1, 3);
    mediaGrid->addWidget(m_onlineArchiveCheckBox, 5, 0, 1, 3);

    QGridLayout *transfersGrid = new QGridLayout(m_transfersTab);
    transfersGrid->addWidget(new QLabel(tr("Concurrent transfers") + ":", this), 0, 0);
    transfersGrid->addWidget(m_transfersComboBox, 0, 1);
    transfersGrid->addWidget(m_transfersCheckBox, 1, 0, 1, 2);

    QGridLayout *otherGrid = new QGridLayout(m_otherTab);
    otherGrid->addWidget(new QLabel(tr("Language") + ":", this), 0, 0);
    otherGrid->addWidget(m_languageComboBox, 0, 1);

    m_tabWidget->addTab(m_mediaTab, tr("Media"));
    m_tabWidget->addTab(m_transfersTab, tr("Transfers"));
    m_tabWidget->addTab(m_otherTab, tr("Other"));

    m_vbox->addWidget(m_tabWidget);
    m_vbox->addWidget(m_buttonBox);

    this->loadSettings();
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(saveSettings()));
    this->connect(m_buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
    this->connect(m_folderButton, SIGNAL(clicked()), this, SLOT(showFileDialog()));
}

SettingsDialog::~SettingsDialog() {}

void SettingsDialog::loadSettings() {
    m_folderEdit->setText(Settings::instance()->downloadPath());
    m_subfoldersCheckBox->setChecked(Settings::instance()->createArtistSubfolders());
    m_formatComboBox->setCurrentIndex(m_formatComboBox->findData(Settings::instance()->downloadFormat(), Qt::UserRole + 1));
    m_clipboardCheckBox->setChecked(Settings::instance()->monitorClipboard());
    m_scrobbleCheckBox->setChecked(Settings::instance()->scrobbleTracks());
    m_scrobbleCheckBox->setEnabled(Lastfm::instance()->userSignedIn());
    m_onlineArchiveCheckBox->setChecked(Settings::instance()->archiveOnline());
    m_transfersComboBox->setCurrentIndex(m_transfersComboBox->findData(Settings::instance()->maximumConcurrentTransfers(), Qt::UserRole + 1));
    m_transfersCheckBox->setChecked(Settings::instance()->startTransfersAutomatically());
    m_languageComboBox->setCurrentIndex(m_languageComboBox->findData(Settings::instance()->language(), Qt::UserRole + 1));
}

void SettingsDialog::saveSettings() {
    Settings::instance()->setDownloadPath(m_folderEdit->text());
    Settings::instance()->setCreateArtistSubfolders(m_subfoldersCheckBox->isChecked());
    Settings::instance()->setDownloadFormat(static_cast<AudioFormats::Format>(m_formatComboBox->itemData(m_formatComboBox->currentIndex(), Qt::UserRole + 1).toInt()));
    Settings::instance()->setMonitorClipboard(m_clipboardCheckBox->isChecked());
    Settings::instance()->setScrobbleTracks(m_scrobbleCheckBox->isChecked());
    Settings::instance()->setArchiveOnline(m_onlineArchiveCheckBox->isChecked());
    Settings::instance()->setMaximumConcurrentTransfers(m_transfersComboBox->itemData(m_formatComboBox->currentIndex(), Qt::UserRole + 1).toInt());
    Settings::instance()->setStartTransfersAutomatically(m_transfersCheckBox->isChecked());
    Settings::instance()->setLanguage(m_languageComboBox->itemData(m_languageComboBox->currentIndex(), Qt::UserRole + 1).toString());
    Settings::instance()->saveSettings();

    this->accept();
}

void SettingsDialog::showFileDialog() {
    QString folder = QFileDialog::getExistingDirectory(this, tr("Download folder"), Settings::instance()->downloadPath());

    if (!folder.isEmpty()) {
        m_folderEdit->setText(folder);
    }
}
