#include "mainwindow.h"
#include "buttonbar.h"
#include "mediaplayerbutton.h"
#include "toolbar.h"
#include "toolbutton.h"
#include "volumecontrol.h"
#include "windowstack.h"
#include "navigationbar.h"
#include "homescreendelegate.h"
#include "nowplayingwindow.h"
#include "tracklistwindow.h"
#include "grouplistwindow.h"
#include "playlistlistwindow.h"
#include "userlistwindow.h"
#include "channelwindow.h"
#include "mychannelwindow.h"
#include "activitylistwindow.h"
#include "transferlistwindow.h"
#include "archivewindow.h"
#include "trackinfowindow.h"
#include "searchdialog.h"
#include "exploredialog.h"
#include "settingsdialog.h"
#include "accountsdialog.h"
#include "aboutdialog.h"
#include "informationbanner.h"
#include "informationnote.h"
#include "../base/homescreenmodel.h"
#include "../base/soundcloud.h"
#include "../base/settings.h"
#include "../base/playbacklistmodel.h"
#include "../base/utils.h"
#include "../mediaplayer/gstreamer/mediaplayer.h"
#include <QMenuBar>
#include <QListView>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QApplication>
#include <QLabel>
#include <QTimer>
#include <QKeyEvent>

MainWindow* MainWindow::self = 0;

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    m_searchAction(this->menuBar()->addAction(tr("Search"), this, SLOT(showSearchDialog()))),
    m_transfersAction(this->menuBar()->addAction(tr("Transfers"), this, SLOT(showTransfers()))),
    m_accountsAction(this->menuBar()->addAction(tr("Accounts"), this, SLOT(showAccountsDialog()))),
    m_settingsAction(this->menuBar()->addAction(tr("Settings"), this, SLOT(showSettingsDialog()))),
    m_playbackMenu(this->menuBar()->addMenu(tr("Playback"))),
    m_playPauseAction(m_playbackMenu->addAction(tr("Play"), MediaPlayer::instance(), SLOT(togglePlayPause()))),
    m_stopAction(m_playbackMenu->addAction(tr("Stop"), MediaPlayer::instance(), SLOT(stop()))),
    m_playbackSeparator(m_playbackMenu->addSeparator()),
    m_previousAction(m_playbackMenu->addAction(tr("Previous"), MediaPlayer::instance(), SLOT(previous()))),
    m_nextAction(m_playbackMenu->addAction(tr("Next"), MediaPlayer::instance(), SLOT(next()))),
    m_playbackSeparator2(m_playbackMenu->addSeparator()),
    m_playbackModeMenu(m_playbackMenu->addMenu(tr("Mode"))),
    m_playbackModeSeparator(m_playbackMenu->addSeparator()),
    m_sequentialAction(m_playbackModeMenu->addAction(tr("Sequential"), MediaPlayer::instance(), SLOT(setSequentialPlaybackMode()))),
    m_repeatAllAction(m_playbackModeMenu->addAction(tr("Repeat all"), MediaPlayer::instance(), SLOT(setRepeatAllPlaybackMode()))),
    m_repeatOneAction(m_playbackModeMenu->addAction(tr("Repeat one"), MediaPlayer::instance(), SLOT(setRepeatOnePlaybackMode()))),
    m_muteAction(m_playbackMenu->addAction(tr("Mute"), MediaPlayer::instance(), SLOT(setMuted(bool)))),
    m_playbackModeGroup(new QActionGroup(this)),
    m_viewMenu(this->menuBar()->addMenu(tr("View"))),
    m_homeAction(m_viewMenu->addAction(tr("Main"), this, SLOT(toggleCurrentTab()))),
    m_nowPlayingAction(m_viewMenu->addAction(tr("Now playing"), this, SLOT(toggleCurrentTab()))),
    m_viewModeSeparator(m_viewMenu->addSeparator()),
    m_fullscreenAction(m_viewMenu->addAction(tr("Full screen"), this, SLOT(setFullScreen(bool)))),
    m_viewModeGroup(new QActionGroup(this)),
    m_aboutAction(this->menuBar()->addAction(tr("About"), this, SLOT(showAboutDialog()))),
    m_quitAction(this->menuBar()->addAction(tr("Close"), QApplication::instance(), SLOT(quit()))),
    m_toolBar(new ToolBar(this)),
    m_toggleVolumeButton(new ToolButton(m_toolBar)),
    m_playbackModeButton(new ToolButton(m_toolBar)),
    m_positionLabel(new QLabel("--:--", m_toolBar)),
    m_slider(new QSlider(Qt::Horizontal, m_toolBar)),
    m_durationLabel(new QLabel("--:--", m_toolBar)),
    m_mediaButtonBar(new ButtonBar(this)),
    m_previousButton(new MediaPlayerButton(MediaPlayerButton::Previous, m_mediaButtonBar)),
    m_playPauseButton(new MediaPlayerButton(MediaPlayerButton::PlayPause, m_mediaButtonBar)),
    m_stopButton(new MediaPlayerButton(MediaPlayerButton::Stop, m_mediaButtonBar)),
    m_nextButton(new MediaPlayerButton(MediaPlayerButton::Next, m_mediaButtonBar)),
    m_volumeControl(new VolumeControl(this)),
    m_volumeTimer(new QTimer(this)),
    m_windowStack(new WindowStack(this)),
    m_hbox(0),
    m_vbox(0),
    m_homeWindow(new QWidget(this)),
    m_nowPlayingWindow(0),
    m_navigationBar(new NavigationBar(m_homeWindow)),
    m_view(new QListView(m_homeWindow))
{
    if (!self) {
        self = this;
    }
    else {
        qWarning() << "MainWindow already exists";
    }

    this->setWindowTitle("MusiKloud");
    this->setCentralWidget(new QWidget(this));
    this->addToolBar(m_toolBar);

    m_searchAction->setShortcut(Qt::CTRL + Qt::Key_S);
    m_transfersAction->setShortcut(Qt::CTRL + Qt::Key_T);
    m_playPauseAction->setShortcut(Qt::Key_Space);
    m_quitAction->setShortcut(Qt::CTRL + Qt::Key_Q);

    m_playPauseAction->setEnabled(false);
    m_stopAction->setEnabled(false);
    m_previousAction->setEnabled(false);
    m_nextAction->setEnabled(false);

    m_sequentialAction->setCheckable(true);
    m_sequentialAction->setChecked(true);
    m_repeatAllAction->setCheckable(true);
    m_repeatOneAction->setCheckable(true);
    m_muteAction->setCheckable(true);

    m_playbackModeGroup->addAction(m_sequentialAction);
    m_playbackModeGroup->addAction(m_repeatAllAction);
    m_playbackModeGroup->addAction(m_repeatOneAction);
    m_playbackModeGroup->setExclusive(true);

    m_homeAction->setCheckable(true);
    m_homeAction->setChecked(true);
    m_nowPlayingAction->setCheckable(true);

    m_viewModeGroup->addAction(m_homeAction);
    m_viewModeGroup->addAction(m_nowPlayingAction);
    m_viewModeGroup->setExclusive(true);

    m_fullscreenAction->setCheckable(true);

    m_toolBar->addWidget(m_toggleVolumeButton);
    m_toolBar->addWidget(m_playbackModeButton);
    m_toolBar->addWidget(m_positionLabel);
    m_toolBar->addWidget(m_slider);
    m_toolBar->addWidget(m_durationLabel);
    m_toolBar->setTabsEnabled(true);

    m_toggleVolumeButton->setCheckable(true);
    m_toggleVolumeButton->setIcon(QIcon("/usr/share/icons/hicolor/26x26/hildon/qgn_medi_volume_unmuted.png"));
    m_playbackModeButton->setIcon(QIcon("/usr/share/icons/hicolor/26x26/hildon/qgn_medi_random_off_repeat_off.png"));
    m_playbackModeButton->setMenu(m_playbackModeMenu);
    m_playbackModeButton->setPopupMode(QToolButton::InstantPopup);

    m_positionLabel->setFixedWidth(100);
    m_durationLabel->setFixedWidth(100);
    m_positionLabel->setAlignment(Qt::AlignCenter);
    m_durationLabel->setAlignment(Qt::AlignCenter);

    m_mediaButtonBar->addButton(m_previousButton);
    m_mediaButtonBar->addButton(m_playPauseButton);
    m_mediaButtonBar->addButton(m_stopButton);
    m_mediaButtonBar->addButton(m_nextButton);

    m_volumeControl->hide();

    m_volumeTimer->setInterval(5000);
    m_volumeTimer->setSingleShot(true);

    m_view->setWindowTitle(tr("Home"));
    m_view->setModel(new HomescreenModel(m_view));
    m_view->setItemDelegate(new HomescreenDelegate(m_view));
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    m_vbox = new QVBoxLayout(m_homeWindow);
    m_vbox->addWidget(m_navigationBar);
    m_vbox->addWidget(m_windowStack);

    m_hbox = new QHBoxLayout(this->centralWidget());
    m_hbox->setContentsMargins(12, 0, 0, 0);
    m_hbox->addWidget(m_mediaButtonBar);
    m_hbox->addWidget(m_homeWindow);

    this->connect(m_view, SIGNAL(activated(QModelIndex)), this, SLOT(onListItemActivated(QModelIndex)));
    this->connect(m_toolBar, SIGNAL(currentTabChanged(int)), this, SLOT(onCurrentTabChanged(int)));
    this->connect(m_windowStack, SIGNAL(pushed(QString)), m_navigationBar, SLOT(push(QString)));
    this->connect(m_navigationBar, SIGNAL(popped(int)), m_windowStack, SLOT(pop(int)));
    this->connect(m_toggleVolumeButton, SIGNAL(clicked(bool)), this, SLOT(showVolumeControl(bool)));
    this->connect(m_muteAction, SIGNAL(triggered(bool)), m_volumeControl, SLOT(setMuted(bool)));
    this->connect(m_previousButton, SIGNAL(clicked()), m_previousAction, SLOT(trigger()));
    this->connect(m_playPauseButton, SIGNAL(clicked()), m_playPauseAction, SLOT(trigger()));
    this->connect(m_stopButton, SIGNAL(clicked()), m_stopAction, SLOT(trigger()));
    this->connect(m_nextButton, SIGNAL(clicked()), m_nextAction, SLOT(trigger()));
    this->connect(m_volumeControl, SIGNAL(valueChanged(qreal)), MediaPlayer::instance(), SLOT(setVolume(qreal)));
    this->connect(m_volumeControl, SIGNAL(mutedChanged(bool)), m_muteAction, SLOT(trigger()));
    this->connect(m_volumeControl, SIGNAL(pressed()), m_volumeTimer, SLOT(stop()));
    this->connect(m_volumeControl, SIGNAL(released()), m_volumeTimer, SLOT(start()));
    this->connect(m_volumeTimer, SIGNAL(timeout()), this, SLOT(showVolumeControl()));
    this->connect(MediaPlayer::instance(), SIGNAL(stateChanged(Media::State)), this, SLOT(onMediaStateChanged(Media::State)));
    this->connect(MediaPlayer::instance(), SIGNAL(mutedChanged(bool)), this, SLOT(onVolumeMutedChanged(bool)));
    this->connect(MediaPlayer::instance(), SIGNAL(positionChanged(qint64)), this, SLOT(onPlaybackPositionChanged(qint64)));
    this->connect(MediaPlayer::instance(), SIGNAL(durationChanged(qint64)), this, SLOT(onPlaybackDurationChanged(qint64)));
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onPlaybackQueueChanged(int)));

    m_windowStack->push(m_view);
}

MainWindow::~MainWindow() {}

MainWindow* MainWindow::instance() {
    return !self ? new MainWindow : self;
}

bool MainWindow::event(QEvent *event) {
    switch (event->type()) {
    case QEvent::WindowStateChange:
        m_fullscreenAction->setChecked(this->isFullScreen());
        break;
    case QEvent::ContextMenu:
        return false;
    default:
        break;
    }

    return QMainWindow::event(event);
}

void MainWindow::keyPressEvent(QKeyEvent *event) {
    if (!event->isAutoRepeat()) {
        if (event->modifiers() & Qt::ControlModifier) {
            switch (event->key()) {
            case Qt::Key_S:
                m_searchAction->trigger();
                event->accept();
                return;
            case Qt::Key_T:
                m_transfersAction->trigger();
                event->accept();
                return;
            case Qt::Key_Q:
                m_quitAction->trigger();
                event->accept();
                return;
            case Qt::Key_I:
            {
                InformationNote *note = new InformationNote(this);
                note->showInformation(tr("This is an information note"));
            }
            default:
                QMainWindow::keyPressEvent(event);
                return;
            }
        }

        switch (event->key()) {
        case Qt::Key_Up:
            m_navigationBar->setFocus(Qt::OtherFocusReason);
            event->accept();
            return;
        case Qt::Key_Down:
            m_windowStack->setFocus(Qt::OtherFocusReason);
            event->accept();
            return;
        case Qt::Key_Space:
            m_playPauseAction->trigger();
            event->accept();
            return;
        case Qt::Key_Escape:
            this->pop();
            event->accept();
            return;
        default:
            break;
        }
    }

    QMainWindow::keyPressEvent(event);
}

void MainWindow::onMediaStateChanged(Media::State state) {
    switch (state) {
    case Media::Playing:
        m_playPauseAction->setText(tr("Pause"));
        m_stopAction->setEnabled(true);
        m_playPauseButton->update();
        m_stopButton->update();
        return;
    case Media::Paused:
        m_playPauseAction->setText(tr("Play"));
        m_stopAction->setEnabled(true);
        m_playPauseButton->update();
        m_stopButton->update();
    default:
        m_playPauseAction->setText(tr("Play"));
        m_stopAction->setEnabled(false);
        m_playPauseButton->update();
        m_stopButton->update();
        return;
    }
}

void MainWindow::onPlaybackQueueChanged(int count) {
    switch (count) {
    case 0:
        m_playPauseAction->setEnabled(false);
        m_previousAction->setEnabled(false);
        m_nextAction->setEnabled(false);
        m_previousButton->setEnabled(false);
        m_nextButton->setEnabled(false);
        return;
    case 1:
        m_playPauseAction->setEnabled(true);
        m_previousAction->setEnabled(false);
        m_nextAction->setEnabled(false);
        m_playPauseButton->setEnabled(true);
        m_previousButton->setEnabled(false);
        m_nextButton->setEnabled(false);
        return;
    default:
        m_playPauseAction->setEnabled(true);
        m_previousAction->setEnabled(true);
        m_nextAction->setEnabled(true);
        m_playPauseButton->setEnabled(true);
        m_previousButton->setEnabled(true);
        m_nextButton->setEnabled(true);
        return;
    }
}

void MainWindow::onVolumeMutedChanged(bool muted) {
    m_toggleVolumeButton->setIcon(QIcon(QString("/usr/share/icons/hicolor/26x26/hildon/qgn_medi_volume_%1muted.png").arg(muted ? "" : "un")));
}

void MainWindow::onPlaybackPositionChanged(qint64 position) {
    m_durationLabel->setText(Utils::durationFromMSecs(position));
    m_slider->setValue(position);
}

void MainWindow::onPlaybackDurationChanged(qint64 duration) {
    m_durationLabel->setText(!duration ? "--:--" : Utils::durationFromMSecs(duration));
    m_slider->setMaximum(duration);
}

void MainWindow::setFullScreen(bool fullScreen) {
    if (fullScreen) {
        this->showFullScreen();
    }
    else {
        this->showNormal();
    }
}

void MainWindow::push(QWidget *window) {
    m_windowStack->push(window);
    m_toolBar->setCurrentTab(0);
}

void MainWindow::pop(int to) {
    m_navigationBar->pop(to);
    m_toolBar->setCurrentTab(0);
}

void MainWindow::showExploreWindow(Queries::QueryType type, const QString &title, const QUrl &feed) {
    switch (type) {
    case Queries::Tracks:
    {
        TrackListWindow *tracksWindow = new TrackListWindow(this);
        tracksWindow->setTrackFeed(feed);
        tracksWindow->setWindowTitle(title);
        this->push(tracksWindow);
    }
        return;
    case Queries::Users:
    {
        UserListWindow *usersWindow = new UserListWindow(this);
        usersWindow->setUserFeed(feed);
        usersWindow->setWindowTitle(title);
        this->push(usersWindow);
    }
        return;
    case Queries::Groups:
    {
        GroupListWindow *groupsWindow = new GroupListWindow(this);
        groupsWindow->setGroupFeed(feed);
        groupsWindow->setWindowTitle(title);
        this->push(groupsWindow);
    }
        return;
    default:
        qWarning() << "MainWindow::showExploreWindow(): Invalid type";
        return;
    }
}

void MainWindow::search(const QString &query, Queries::QueryType type, Queries::QueryOrder order) {
    this->pop(0);

    switch (type) {
    case Queries::Tracks:
        this->trackSearch(query, order);
        return;
    case Queries::Playlists:
        this->playlistSearch(query, order);
        return;
    case Queries::Users:
        this->userSearch(query);
        return;
    case Queries::Groups:
        this->groupSearch(query, order);
        return;
    default:
        qWarning() << "MainWindow::search(): Invalid query type.";
        return;
    }
}

void MainWindow::trackSearch(const QString &query, Queries::QueryOrder order) {
    TrackListWindow *window = new TrackListWindow(this);
    window->search(query, order);
    this->push(window);
}

void MainWindow::playlistSearch(const QString &query, Queries::QueryOrder order) {
    PlaylistListWindow *window = new PlaylistListWindow(this);
    window->search(query, order);
    this->push(window);
}

void MainWindow::groupSearch(const QString &query, Queries::QueryOrder order) {
    GroupListWindow *window = new GroupListWindow(this);
    window->search(query, order);
    this->push(window);
}

void MainWindow::userSearch(const QString &query) {
    UserListWindow *window = new UserListWindow(this);
    window->search(query);
    this->push(window);
}

void MainWindow::displayTrack(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->displayTrack(urls.first());
    }
}

void MainWindow::displayTrack(const QString &url) {
    this->pop(0);
    this->activateWindow();

    SoundCloud::instance()->getFullTrack(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(showTrack(QSharedPointer<TrackItem>)));
}

void MainWindow::playTrack(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->playTrack(urls.first());
    }
}

void MainWindow::playTrack(const QString &url) {
    this->pop(0);
    this->activateWindow();

    SoundCloud::instance()->getFullTrack(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotTrack(QSharedPointer<TrackItem>)), this, SLOT(playTrack(QSharedPointer<TrackItem>)));
}

void MainWindow::displayPlaylist(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->displayPlaylist(urls.first());
    }
}

void MainWindow::displayPlaylist(const QString &url) {
    this->pop(0);
    this->activateWindow();

    SoundCloud::instance()->getFullPlaylist(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotPlaylist(QSharedPointer<PlaylistItem>)), this, SLOT(showPlaylist(QSharedPointer<PlaylistItem>)));
}

void MainWindow::displayUser(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->displayUser(urls.first());
    }
}

void MainWindow::displayUser(const QString &url) {
    this->pop(0);
    this->activateWindow();

    SoundCloud::instance()->resolve(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotResourceId(QString,Queries::QueryType)), this, SLOT(showUserChannel(QString)));
}

void MainWindow::showTrack(QSharedPointer<TrackItem> track) {
    TrackInfoWindow *window = new TrackInfoWindow(track, this);
    this->push(window);
}

void MainWindow::playTrack(QSharedPointer<TrackItem> track) {
    MediaPlayer::instance()->playTrack(track);
}

void MainWindow::showPlaylist(QSharedPointer<PlaylistItem> playlist) {
    Q_UNUSED(playlist)
}

void MainWindow::showUserChannel(const QString &id) {
    ChannelWindow *window = new ChannelWindow(id, this);
    this->push(window);
}

void MainWindow::displayResource(const QStringList &urls) {
    if (!urls.isEmpty()) {
        this->displayResource(urls.first());
    }
}

void MainWindow::displayResource(const QString &url) {
    this->pop(0);
    this->activateWindow();

    SoundCloud::instance()->resolve(url);
    this->connect(SoundCloud::instance(), SIGNAL(gotResourceId(QString,Queries::QueryType)), this, SLOT(onResourceResolved(QString,Queries::QueryType)));
}

void MainWindow::onResourceResolved(const QString &id, Queries::QueryType type) {
    switch (type) {
    case Queries::Tracks:
        this->displayTrack(id);
        return;
    case Queries::Playlists:
        this->displayPlaylist(id);
        return;
    case Queries::Users:
        this->showUserChannel(id);
        return;
    default:
        qWarning() << "MainWindow::onResourceResolved(): Invalid resource type.";
        return;
    }
}

void MainWindow::onListItemActivated(const QModelIndex &index) {
    switch (index.row()) {
    case 0:
        this->showMyChannel();
        return;
    case 1:
        this->showDashboard();
        return;
    case 2:
        this->showExploreDialog();
        return;
    case 3:
        this->showArchive();
        return;
    default:
        qDebug() << index.row();
        return;
    }
}

void MainWindow::toggleCurrentTab() {
    m_toolBar->setCurrentTab(m_nowPlayingAction->isChecked() ? 1 : 0);
}

void MainWindow::onCurrentTabChanged(int index) {
    switch (index) {
    case 1:
        this->showNowPlayingWindow();
        return;
    default:
        this->showHomeWindow();
        return;
    }
}

void MainWindow::showVolumeControl(bool show) {
    if (show) {
        m_mediaButtonBar->hide();
        m_hbox->insertWidget(0, m_volumeControl);
        m_volumeControl->show();
        m_volumeTimer->start();
    }
    else {
        m_toggleVolumeButton->setChecked(false);
        m_volumeControl->hide();
        m_hbox->insertWidget(0, m_mediaButtonBar);
        m_mediaButtonBar->show();

        if (m_volumeTimer->isActive()) {
            m_volumeTimer->stop();
        }
    }
}

void MainWindow::showHomeWindow() {
    if (m_nowPlayingWindow) {
        m_nowPlayingWindow->hide();
    }

    m_hbox->addWidget(m_homeWindow);
    m_homeWindow->show();
    m_homeAction->setChecked(true);
}

void MainWindow::showNowPlayingWindow() {
    if (!m_nowPlayingWindow) {
        m_nowPlayingWindow = new NowPlayingWindow(this);
    }

    m_homeWindow->hide();
    m_hbox->addWidget(m_nowPlayingWindow);
    m_nowPlayingWindow->show();
    m_nowPlayingAction->setChecked(true);
}

void MainWindow::showMyChannel() {
    if (SoundCloud::instance()->userSignedIn()) {
        MyChannelWindow *window = new MyChannelWindow(this);
        this->push(window);
    }
    else {
        InformationBanner::showInformation(this, tr("Please connect your SoundCloud account"));
    }
}

void MainWindow::showDashboard() {
    if (SoundCloud::instance()->userSignedIn()) {
        ActivityListWindow *window = new ActivityListWindow(this);
        this->push(window);
    }
    else {
        InformationBanner::showInformation(this, tr("Please connect your SoundCloud account"));
    }
}

void MainWindow::showTransfers() {
//    this->pop(0);

    TransferListWindow *window = new TransferListWindow(this);
    this->push(window);
}

void MainWindow::showArchive() {
    ArchiveWindow *window = new ArchiveWindow(this);
    this->push(window);
}

void MainWindow::showExploreDialog() {
    ExploreDialog *dialog = new ExploreDialog(this);
    dialog->open();
    this->connect(dialog, SIGNAL(categoryChosen(Queries::QueryType,QString,QUrl)), this, SLOT(showExploreWindow(Queries::QueryType,QString,QUrl)));
}

void MainWindow::showSearchDialog() {
    SearchDialog *dialog = new SearchDialog(this);
    dialog->open();
    this->connect(dialog, SIGNAL(search(QString,Queries::QueryType,Queries::QueryOrder)), this, SLOT(search(QString,Queries::QueryType,Queries::QueryOrder)));
    this->connect(dialog, SIGNAL(fromUrl(QString)), this, SLOT(displayResource(QString)));
}

void MainWindow::showAccountsDialog() {
    AccountsDialog *dialog = new AccountsDialog(this);
    dialog->open();
}

void MainWindow::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(this);
    dialog->open();
}

void MainWindow::showAboutDialog() {
    AboutDialog *dialog = new AboutDialog(this);
    dialog->open();
}
