#include "informationnote.h"
#include "../base/utils.h"
#include <QPaintEvent>
#include <QPainter>
#include <QLabel>
#include <QDialogButtonBox>
#include <QGridLayout>

InformationNote::InformationNote(QWidget *parent) :
    QDialog(parent),
    m_noteType(Information),
    m_iconLabel(new QLabel(this)),
    m_messageLabel(new QLabel(this)),
    m_buttonBox(new QDialogButtonBox(Qt::Horizontal, this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowFlags(Qt::Dialog | Qt::FramelessWindowHint);
    this->setMinimumWidth(400);

    m_iconLabel->setFixedSize(50, 50);
    m_messageLabel->setWordWrap(true);

    m_grid->setContentsMargins(24, 24, 24, 24);
    m_grid->addWidget(m_iconLabel, 0, 0);
    m_grid->addWidget(m_messageLabel, 0, 1);
    m_grid->addWidget(m_buttonBox, 1, 0, 1, 2, Qt::AlignHCenter);

    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
    this->connect(m_buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
}

InformationNote::~InformationNote() {}

void InformationNote::paintEvent(QPaintEvent *event) {
    QPainter painter(this);

    int width = this->rect().width() / 2;
    int height = this->rect().height() / 2;

    Utils::drawBorderImage(&painter, QRect(0, 0, width, height), QImage("/usr/share/themes/default/images/qgn_plat_note_frame_corner_top_left.png"), 8, 8, 8, 8);
    Utils::drawBorderImage(&painter, QRect(width - 1, 0, width, height), QImage("/usr/share/themes/default/images/qgn_plat_note_frame_corner_top_right.png"), 8, 8, 8, 8);
    Utils::drawBorderImage(&painter, QRect(0, height - 1, width, height), QImage("/usr/share/themes/default/images/qgn_plat_note_frame_corner_bottom_left.png"), 8, 8, 8, 8);
    Utils::drawBorderImage(&painter, QRect(width - 1, height - 1, width, height), QImage("/usr/share/themes/default/images/qgn_plat_note_frame_corner_bottom_right.png"), 8, 8, 8, 8);

    event->accept();
}

InformationNote::NoteType InformationNote::noteType() const {
    return m_noteType;
}

void InformationNote::setNoteType(NoteType type) {
    if (!this->isVisible()) {
        m_noteType = type;

        switch (type) {
        case Question:
            m_iconLabel->setPixmap(QPixmap("/usr/share/icons/hicolor/50x50/hildon/qgn_note_confirm.png"));
            m_buttonBox->setStandardButtons(QDialogButtonBox::Yes | QDialogButtonBox::No);
            return;
        case Warning:
            m_iconLabel->setPixmap(QPixmap("/usr/share/icons/hicolor/50x50/hildon/qgn_note_gene_syswarning.png"));
            m_buttonBox->setStandardButtons(QDialogButtonBox::Ok);
            return;
        default:
            m_iconLabel->setPixmap(QPixmap("/usr/share/icons/hicolor/50x50/hildon/qgn_note_info.png"));
            m_buttonBox->setStandardButtons(QDialogButtonBox::Ok);
            return;
        }
    }
}

QString InformationNote::message() const {
    return m_messageLabel->text();
}

void InformationNote::setMessage(const QString &message) {
    if (!this->isVisible()) {
        m_messageLabel->setText(message);
    }
}

void InformationNote::showInformation(const QString &message) {
    this->setNoteType(Information);
    this->setMessage(message);
    this->open();
}

void InformationNote::showQuestion(const QString &message) {
    this->setNoteType(Question);
    this->setMessage(message);
    this->open();
}

void InformationNote::showWarning(const QString &message) {
    this->setNoteType(Warning);
    this->setMessage(message);
    this->open();
}

InformationNote* InformationNote::showInformation(QWidget *parent, const QString &message) {
    InformationNote *note = new InformationNote(parent);
    note->setAttribute(Qt::WA_DeleteOnClose);
    note->showInformation(message);

    return note;
}

InformationNote* InformationNote::showQuestion(QWidget *parent, const QString &message) {
    InformationNote *note = new InformationNote(parent);
    note->setAttribute(Qt::WA_DeleteOnClose);
    note->showQuestion(message);

    return note;
}

InformationNote* InformationNote::showWarning(QWidget *parent, const QString &message) {
    InformationNote *note = new InformationNote(parent);
    note->setAttribute(Qt::WA_DeleteOnClose);
    note->showWarning(message);

    return note;
}
