#include "soundcloud.h"
#include "json.h"
#include "utils.h"
#include <QRegExp>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QStringList>
#include <QDateTime>
#include <QUrl>
#include <QDebug>
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
#include <QQmlEngine>
#else
#include <QDeclarativeEngine>
#endif
#endif
#if QT_VERSION >= 0x050000
#include <QUrlQuery>
#endif

const QString CLIENT_ID("aada80a82c91fb42d0bb1cb321b59034");
const QString CLIENT_SECRET("dcc9c80ce05614d4988cebb82ddde152");
const QString REDIRECT_URI("http://www.musikloud.co.uk");

SoundCloud* SoundCloud::self = 0;

using namespace QtJson;

SoundCloud::SoundCloud(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_busy(false),
    m_cancelled(false)
{
    if (!self) {
        self = this;
    }
}

SoundCloud::~SoundCloud() {}

SoundCloud* SoundCloud::instance() {
    return !self ? new SoundCloud : self;
}

void SoundCloud::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            this->setCancelled(false);
            emit busy(message, numberOfOperations);
        }
        else if (!this->cancelled()) {
            emit busyProgressChanged(numberOfOperations);
        }

        emit busyChanged(isBusy);
    }
}

void SoundCloud::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, 0);
    this->disconnect(this, SIGNAL(gotResourceId(QString,Queries::QueryType)), 0, 0);
#ifdef QML_USER_INTERFACE
    this->disconnect(this, SIGNAL(gotTrack(TrackItem*)), 0, 0);
    this->disconnect(this, SIGNAL(gotPlaylist(PlaylistItem*)), 0, 0);
#else
    this->disconnect(this, SIGNAL(gotTrack(QSharedPointer<TrackItem>)), 0, 0);
    this->disconnect(this, SIGNAL(gotPlaylist(QSharedPointer<PlaylistItem>)), 0, 0);
#endif
    emit currentOperationCancelled();
}

QUrl SoundCloud::authUrl() {
    QUrl url("https://soundcloud.com/connect");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("client_id", CLIENT_ID);
    query.addQueryItem("response_type", "code");
    query.addQueryItem("redirect_uri", REDIRECT_URI);
    query.addQueryItem("scope", "non-expiring");
    query.addQueryItem("display", "popup");
    url.setQuery(query);
#else
    url.addQueryItem("client_id", CLIENT_ID);
    url.addQueryItem("response_type", "code");
    url.addQueryItem("redirect_uri", REDIRECT_URI);
    url.addQueryItem("scope", "non-expiring");
    url.addQueryItem("display", "popup");
#endif
    return url;
}

QNetworkReply* SoundCloud::createReply(QUrl url, int offset) {
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);

    if (this->userSignedIn()) {
        url.setScheme("https");
        query.addQueryItem("oauth_token", this->accessToken());
    }
    else {
        url.setScheme("http");
        query.addQueryItem("client_id", CLIENT_ID);
    }

    if (offset) {
        query.addQueryItem("offset", QString::number(offset));
    }

    url.setQuery(query);
#else
    if (this->userSignedIn()) {
        url.setScheme("https");
        url.addQueryItem("oauth_token", this->accessToken());
    }
    else {
        url.setScheme("http");
        url.addQueryItem("client_id", CLIENT_ID);
    }

    if (offset) {
        url.addQueryItem("offset", QString::number(offset));
    }
#endif
    qDebug() << url;

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());

    return this->networkAccessManager()->get(request);
}

QNetworkReply* SoundCloud::createSearchReply(int queryType, const QString &query, int order, int offset) {
    QString qs(query);
    qs.replace(' ', '+');

    QUrl url;

    switch (queryType) {
    case Queries::Tracks:
        url = TRACKS_BASE_FEED;
        break;
    case Queries::Playlists:
        url = PLAYLISTS_BASE_FEED;
        break;
    case Queries::Users:
        url = USERS_BASE_FEED;
        break;
    case Queries::Groups:
        url = GROUPS_BASE_FEED;
        break;
    default:
        qWarning() << "SoundCloud::createSearchReply(): Invalid query type.";
        break;
    }
#if QT_VERSION >= 0x050000
    QUrlQuery urlQuery(url);

    switch (order) {
    case Queries::Date:
        urlQuery.addQueryItem("order", "created_at");
        break;
    default:
        urlQuery.addQueryItem("order", "hotness");
        break;
    }

    urlQuery.addQueryItem("q", "'" + query + "'|" + qs);
    urlQuery.addQueryItem("client_id", CLIENT_ID);

    if (offset) {
        urlQuery.addQueryItem("offset", QString::number(offset));
    }

    url.setQuery(urlQuery);
#else
    switch (order) {
    case Queries::Date:
        url.addQueryItem("order", "created_at");
        break;
    default:
        url.addQueryItem("order", "hotness");
        break;
    }

    url.addQueryItem("q", "'" + query + "'|" + qs);
    url.addQueryItem("client_id", CLIENT_ID);

    if (offset) {
        url.addQueryItem("offset", QString::number(offset));
    }
#endif
    qDebug() << url;

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());

    return this->networkAccessManager()->get(request);
}

QUrl SoundCloud::getStreamUrl(QUrl url) {
    if (url.scheme() == "https") {
        url.setScheme("http");
    }
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);

    if (!query.hasQueryItem("client_id")) {
        query.addQueryItem("client_id", CLIENT_ID);
    }

    url.setQuery(query);
#else
    if (!url.hasQueryItem("client_id")) {
        url.addQueryItem("client_id", CLIENT_ID);
    }
#endif
    return url;
}

QUrl SoundCloud::getDownloadUrl(QUrl url) {
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);

    if (url.scheme() == "https") {
        if (this->userSignedIn()) {
            query.addQueryItem("oauth_token", this->accessToken());
        }
        else {
            url.setScheme("http");
            query.addQueryItem("client_id", CLIENT_ID);
        }
    }
    else {
        query.addQueryItem("client_id", CLIENT_ID);
    }

    url.setQuery(query);
#else
    if (url.scheme() == "https") {
        if (this->userSignedIn()) {
            url.addQueryItem("oauth_token", this->accessToken());
        }
        else {
            url.setScheme("http");
            url.addQueryItem("client_id", CLIENT_ID);
        }
    }
    else {
        url.addQueryItem("client_id", CLIENT_ID);
    }
#endif
    return url;
}

void SoundCloud::resolve(const QString &urlToResolve) {
    this->setBusy(true, tr("Resolving URL"));
    QUrl url("https://api.soundcloud.com/resolve");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("url", urlToResolve);
    query.addQueryItem("client_id", CLIENT_ID);
    url.setQuery(query);
#else
    url.addQueryItem("url", urlToResolve);
    url.addQueryItem("client_id", CLIENT_ID);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkResolvedUrl()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void SoundCloud::checkResolvedUrl() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString resource = reply->header(QNetworkRequest::LocationHeader).toString();
    QString id = resource.section('/', -1).section('?', 0, 0);

    if (!id.isEmpty()) {
        QString resourceType = resource.section('/', -2, -2);

        if (resourceType == "tracks") {
            emit gotResourceId(id, Queries::Tracks);
        }
        else if (resourceType == "playlists") {
            emit gotResourceId(id, Queries::Playlists);
        }
        else if (resourceType == "users") {
            emit gotResourceId(id, Queries::Users);
        }
        else {
            emit error(tr("Unable to resolve URL"));
        }
    }
    else {
        emit error(tr("Unable to resolve URL"));
    }

    this->disconnect(this, SIGNAL(gotResourceId(QString,Queries::QueryType)), 0, 0);

    reply->deleteLater();
}

void SoundCloud::getFullTrack(const QString &id) {
    this->setBusy(true, tr("Retrieving track details"));

    if (id.contains("soundcloud.com")) {
        this->resolve(id);
        this->connect(this, SIGNAL(gotResourceId(QString,Queries::QueryType)), this, SLOT(getFullTrack(QString)));
        return;
    }

    QUrl url;
#if QT_VERSION >= 0x050000
    QUrlQuery query;

    if (this->userSignedIn()) {
        url.setUrl("https://api.soundcloud.com/tracks/" + id + ".json");
        query.addQueryItem("oauth_token", this->accessToken());
    }
    else {
        url.setUrl("http://api.soundcloud.com/tracks/" + id + ".json");
        query.addQueryItem("client_id", CLIENT_ID);
    }

    url.setQuery(query);
#else
    if (this->userSignedIn()) {
        url.setUrl("https://api.soundcloud.com/tracks/" + id + ".json");
        url.addQueryItem("oauth_token", this->accessToken());
    }
    else {
        url.setUrl("http://api.soundcloud.com/tracks/" + id + ".json");
        url.addQueryItem("client_id", CLIENT_ID);
    }
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkTrack()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void SoundCloud::checkTrack() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

        if (statusCode != 200) {
            emit error(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        }
        else {
            emit error(tr("Cannot parse server response"));
        }
    }
    else {
        TrackItem *track = new TrackItem(result);
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
        QQmlEngine::setObjectOwnership(track, QQmlEngine::JavaScriptOwnership);
#else
        QDeclarativeEngine::setObjectOwnership(track, QDeclarativeEngine::JavaScriptOwnership);
#endif
        emit gotTrack(track);
#else
        emit gotTrack(QSharedPointer<TrackItem>(track));
#endif
    }
#ifdef QML_USER_INTERFACE
    this->disconnect(this, SIGNAL(gotTrack(TrackItem*)), 0, 0);
#else
    this->disconnect(this, SIGNAL(gotTrack(QSharedPointer<TrackItem>)), 0, 0);
#endif
    reply->deleteLater();
}

void SoundCloud::getFullPlaylist(const QString &id) {
    this->setBusy(true, tr("Retrieving set details"));

    if (id.contains("soundcloud.com")) {
        this->resolve(id);
        this->connect(this, SIGNAL(gotResourceId(QString,Queries::QueryType)), this, SLOT(getFullPlaylist(QString)));
        return;
    }

    QUrl url;
#if QT_VERSION >= 0x050000
    QUrlQuery query;

    if (this->userSignedIn()) {
        url.setUrl("https://api.soundcloud.com/playlists/" + id + ".json");
        query.addQueryItem("oauth_token", this->accessToken());
    }
    else {
        url.setUrl("http://api.soundcloud.com/playlists/" + id + ".json");
        query.addQueryItem("client_id", CLIENT_ID);
    }

    url.setQuery(query);
#else
    if (this->userSignedIn()) {
        url.setUrl("https://api.soundcloud.com/playlists/" + id + ".json");
        url.addQueryItem("oauth_token", this->accessToken());
    }
    else {
        url.setUrl("http://api.soundcloud.com/playlists/" + id + ".json");
        url.addQueryItem("client_id", CLIENT_ID);
    }
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkPlaylist()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void SoundCloud::checkPlaylist() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

        if (statusCode != 200) {
            emit error(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        }
        else {
            emit error(tr("Cannot parse server response"));
        }
    }
    else {
        PlaylistItem *playlist = new PlaylistItem(result);
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
        QQmlEngine::setObjectOwnership(playlist, QQmlEngine::JavaScriptOwnership);
#else
        QDeclarativeEngine::setObjectOwnership(playlist, QDeclarativeEngine::JavaScriptOwnership);
#endif
        emit gotPlaylist(playlist);
#else
        emit gotPlaylist(QSharedPointer<PlaylistItem>(playlist));
#endif
    }
#ifdef QML_USER_INTERFACE
    this->disconnect(this, SIGNAL(gotPlaylist(PlaylistItem*)), 0, 0);
#else
    this->disconnect(this, SIGNAL(gotPlaylist(QSharedPointer<PlaylistItem>)), 0, 0);
#endif
    reply->deleteLater();
}

void SoundCloud::getUserProfile(const QString &id) {
    QUrl url(QString("http://api.soundcloud.com/users/%1.json?client_id=%2").arg(id).arg(CLIENT_ID));
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUser()));
}

void SoundCloud::checkUser() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit error(tr("Cannot parse server response"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode != 200) {
        emit error(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
    }
    else {
        UserItem *user = new UserItem(result);

        if (!user->followed()) {
            this->isUserFollowed(user->id());
        }
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
        QQmlEngine::setObjectOwnership(user, QQmlEngine::JavaScriptOwnership);
#else
        QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
#endif
        emit gotUser(user);
#else
        emit gotUser(QSharedPointer<UserItem>(user));
#endif
    }
#ifdef QML_USER_INTERFACE
    this->disconnect(this, SIGNAL(gotUser(UserItem*)), 0, 0);
#else
    this->disconnect(this, SIGNAL(gotUser(QSharedPointer<UserItem>)), 0, 0);
#endif
    reply->deleteLater();
}

void SoundCloud::getCurrentUserProfile() {
    QUrl url("https://api.soundcloud.com/me.json?oauth_token=" + this->accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCurrentUser()));
}

void SoundCloud::checkCurrentUser() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

        if (statusCode != 200) {
            emit error(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());

            if (statusCode == 401) {
                emit signedIn(QString());
            }
        }
        else {
            emit error(tr("Cannot parse server response"));
        }
    }

    UserItem *user = new UserItem(result);

    if (this->username().isEmpty()) {
        this->setUsername(user->username());
        emit gotUsername(user->username());
    }
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
        QQmlEngine::setObjectOwnership(user, QQmlEngine::JavaScriptOwnership);
#else
        QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
#endif
    emit gotUser(user);
#else
    emit gotUser(QSharedPointer<UserItem>(user));
#endif
    reply->deleteLater();
}

void SoundCloud::isUserFollowed(const QString &id) {
    m_actionId = id;
    QUrl url(QString("https://api.soundcloud.com/me/followings/%1.json?oauth_token=%2").arg(id).arg(this->accessToken()));
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfUserFollowed()));
}

void SoundCloud::checkIfUserFollowed() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    emit followingChanged(m_actionId, reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt() != 404);
    reply->deleteLater();
}

void SoundCloud::trackIsFavourite(const QString &id) {
    m_actionId = id;
    QUrl url(QString("https://api.soundcloud.com/me/favorites/%1.json?oauth_token=%2").arg(id).arg(this->accessToken()));
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfFavourite()));
}

void SoundCloud::checkIfFavourite() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    emit favouriteChanged(m_actionId, reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt() != 404);
    reply->deleteLater();
}

void SoundCloud::signIn(const QUrl &response) {
#if QT_VERSION >= 0x050000
    QUrlQuery query(response);
    QString authCode = query.queryItemValue("code");

    if (authCode.isEmpty()) {

        QString errorString = query.queryItemValue("error");

        if (errorString == "access_denied") {
            emit info(tr("You have denied access to your SoundCloud account"));
        }
        else if (errorString.isEmpty()){
            emit error(tr("No authorisation code received from SoundCloud"));
        }
        else {
            emit error(query.queryItemValue("error_description").replace('+', ' '));
        }

        return;
    }
#else
    QString authCode = response.queryItemValue("code");

    if (authCode.isEmpty()) {

        QString errorString = response.queryItemValue("error");

        if (errorString == "access_denied") {
            emit info(tr("You have denied access to your SoundCloud account"));
        }
        else if (errorString.isEmpty()){
            emit error(tr("No authorisation code received from SoundCloud"));
        }
        else {
            emit error(response.queryItemValue("error_description").replace('+', ' '));
        }

        return;
    }
#endif
    this->setBusy(true, tr("Signing in"));
    QNetworkRequest request(QUrl("https://api.soundcloud.com/oauth2/token"));
    QByteArray data(
                "client_id=" + CLIENT_ID.toUtf8()
                + "&client_secret=" + CLIENT_SECRET.toUtf8()
                + "&redirect_uri=" + REDIRECT_URI.toUtf8()
                + "&grant_type=authorization_code"
                + "&code=" + authCode.toUtf8()
                );

    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->post(request, data);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAccessToken()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void SoundCloud::signIn(const QString &user, const QString &pass) {
    this->setBusy(true, tr("Signing in"));
    QNetworkRequest request(QUrl("https://api.soundcloud.com/oauth2/token"));
    QByteArray data(
                "client_id=" + CLIENT_ID.toUtf8()
                + "&client_secret=" + CLIENT_SECRET.toUtf8()
                + "&scope=non-expiring"
                + "&grant_type=password"
                + "&username=" + user.toUtf8()
                + "&password=" + pass.toUtf8()
                );

    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->post(request, data);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAccessToken()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void SoundCloud::checkAccessToken() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit error(tr("Error parsing server response"));
    }
    else {
        QString token = result.value("access_token").toString();

        if (token.isEmpty()) {

            QString errorString = result.value("error").toString();

            if (!errorString.isEmpty()) {
                emit error(errorString);
            }
            else {
                emit error(tr("Error obtaining access token"));
            }
        }
        else {
            emit signedIn(token);
            emit alert(tr("You are connected to your SoundCloud account"));
        }
    }

    reply->deleteLater();
}

void SoundCloud::setCredentials(const QString &user, const QString &token) {
    this->setUsername(user);
    this->setAccessToken(token);
}

void SoundCloud::setUsername(const QString &user) {
    m_username = user;
    emit usernameChanged();
}

void SoundCloud::setAccessToken(const QString &token) {
    m_accessToken = token;
    emit accessTokenChanged();
    emit userSignedInChanged(this->userSignedIn());
}

void SoundCloud::postRequest(const QUrl &url, const QByteArray &data) {
    qDebug() << url;
    qDebug() << data;
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->post(request, data);
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void SoundCloud::putRequest(const QUrl &url) {
    qDebug() << url;
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("User-Agent", QString("MusiKloud/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->put(request, QByteArray());
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void SoundCloud::deleteRequest(const QUrl &url) {
    qDebug() << url;
    QNetworkRequest request(url);
#if QT_VERSION >= 0x040600
    QNetworkReply *reply = this->networkAccessManager()->deleteResource(request);
#else
    QNetworkReply *reply = this->networkAccessManager()->post(request, "");
#endif
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void SoundCloud::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());
    if (!reply) {
        emit error("Network error");
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 200) || (statusCode == 201)) {
        emit postSuccessful(QString(reply->readAll()));
    }
    else {
        emit postFailed(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
    }

    reply->deleteLater();
}

void SoundCloud::addComment(const QString &id, const QString &comment, int timestamp) {
    QUrl url(QString("https://api.soundcloud.com/tracks/%1/comments.json?oauth_token=%2").arg(id).arg(this->accessToken()));
    QByteArray data("&comment[body]=" + comment.toUtf8().toPercentEncoding());

    if (timestamp) {
        data += "&comment[timestamp]=" + QByteArray::number(timestamp);
    }

    this->postRequest(url, data);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
}

void SoundCloud::onCommentAdded(const QString &response) {
    bool ok;
    QVariantMap comment = Json::parse(response, ok).toMap();

    if (ok) {
        emit commentAdded(QSharedPointer<CommentItem>(new CommentItem(comment)));
    }
    else {
        emit error(tr("Comment response invalid"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
}

void SoundCloud::addToFavourites(const QStringList &ids) {
    if (!ids.isEmpty()) {
        m_actionIds = ids;
        m_actionsProcessed = 0;
        this->setBusy(true, tr("Adding track(s) to favourites"), m_actionIds.size());
        this->addToFavourites();
    }
}

void SoundCloud::addToFavourites() {
    if (!m_actionIds.isEmpty()) {
        QUrl url(QString("https://api.soundcloud.com/me/favorites/%1.json?oauth_token=%2").arg(m_actionIds.first()).arg(this->accessToken()));
        this->putRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToFavourites()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onTrackActionError(QString)));
    }
}

void SoundCloud::onAddedToFavourites() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_actionIds.isEmpty()) {
        emit favouriteChanged(m_actionIds.takeFirst(), true);
    }
    if (!m_actionIds.isEmpty()) {
        this->addToFavourites();
    }
    else {
        this->setBusy(false);
        emit alert(tr("Track(s) added to favourites"));

        this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToFavourites()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onTrackActionError(QString)));
    }
}

void SoundCloud::deleteFromFavourites(const QStringList &ids) {
    if (!ids.isEmpty()) {
        m_actionIds = ids;
        m_actionsProcessed = 0;
        this->setBusy(true, tr("Deleting track(s) from favourites"), m_actionIds.size());
        this->deleteFromFavourites();
    }
}

void SoundCloud::deleteFromFavourites() {
    if (!m_actionIds.isEmpty()) {
        QUrl url(QString("https://api.soundcloud.com/me/favorites/%1.json?oauth_token=%2").arg(m_actionIds.first()).arg(this->accessToken()));
        this->deleteRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromFavourites()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onTrackActionError(QString)));
    }
}

void SoundCloud::onDeletedFromFavourites() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_actionIds.isEmpty()) {
        emit favouriteChanged(m_actionIds.takeFirst(), false);
    }
    if (!m_actionIds.isEmpty()) {
        this->deleteFromFavourites();
    }
    else {
        this->setBusy(false);
        emit alert(tr("Track(s) deleted from favourites"));

        this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromFavourites()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onTrackActionError(QString)));
    }
}

void SoundCloud::onTrackActionError(const QString &errorString) {
    this->setBusy(false);
    emit error(errorString);
    m_actionIds.clear();
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onTrackActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void SoundCloud::followUser(const QString &id) {
    m_actionId = id;
    QUrl url(QString("https://api.soundcloud.com/me/followings/%1.json?oauth_token=%2").arg(id).arg(this->accessToken()));
    this->putRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUserFollowed()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void SoundCloud::onUserFollowed() {
    emit followingChanged(m_actionId, true);
    emit alert(tr("You have followed this user"));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUserFollowed()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void SoundCloud::unfollowUser(const QString &id) {
    m_actionId = id;
    QUrl url(QString("https://api.soundcloud.com/me/followings/%1.json?oauth_token=%2").arg(id).arg(this->accessToken()));
    this->deleteRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUserUnfollowed()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void SoundCloud::onUserUnfollowed() {
    emit followingChanged(m_actionId, false);
    emit alert(tr("You no longer follow this user"));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUserUnfollowed()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void SoundCloud::onUserActionError(const QString &errorString) {
    emit error(errorString);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void SoundCloud::updateProfile(const QVariantMap &profile) {
    QUrl url("https://api.soundcloud.com/me.json");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("oauth_token", this->accessToken());
    query.addQueryItem("user[full_name]", profile.value("fullname").toString());
    query.addQueryItem("user[discogs_name]", profile.value("bandname").toString());
    query.addQueryItem("user[website]", profile.value("websiteUrl").toString());
    query.addQueryItem("user[website_name]", profile.value("websiteTitle").toString());
    query.addQueryItem("user[description]", profile.value("description").toString());
    url.setQuery(query);
#else
    url.addQueryItem("oauth_token", this->accessToken());
    url.addQueryItem("user[full_name]", profile.value("fullname").toString());
    url.addQueryItem("user[discogs_name]", profile.value("bandname").toString());
    url.addQueryItem("user[website]", profile.value("websiteUrl").toString());
    url.addQueryItem("user[website_name]", profile.value("websiteTitle").toString());
    url.addQueryItem("user[description]", profile.value("description").toString());
#endif
    this->putRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onProfileUpdated(QString)));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void SoundCloud::onProfileUpdated(const QString &response) {
    bool ok;
    QVariantMap profile = Json::parse(response, ok).toMap();

    if (ok) {
        emit profileUpdated(profile.value("id").toString(), profile);
        emit alert(tr("Your profile has been updated"));
    }
    else {
        emit error(tr("Cannot obtain new profile"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onProfileUpdated(QString)));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void SoundCloud::shareTrack(const QString &id, const QStringList &connections, const QString &message) {
    QUrl url(QString("https://api.soundcloud.com/tracks/%1/shared-to/connections?oauth_token=%2").arg(id).arg(this->accessToken()));
    QByteArray data(
                "&connections[][id]=" + connections.join("[id]=").toUtf8()
                + "&sharing_note=" + message.toUtf8().toPercentEncoding(QByteArray("+"))
                );
    this->postRequest(url, data);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onTrackShared()));
}

void SoundCloud::onTrackShared() {
    emit alert(tr("Track shared to your connections"));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onTrackShared()));
}
