#ifndef ACTIVITYLISTMODEL_H
#define ACTIVITYLISTMODEL_H

#include "activityitem.h"
#include <QSharedPointer>
#include <QAbstractListModel>

#ifndef QML_USER_INTERFACE
class ThumbnailCache;
#endif

class ActivityListModel : public QAbstractListModel
{
    Q_OBJECT

#if QT_VERSION >= 0x040700
    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY loadingChanged)
    Q_PROPERTY(QUrl feed
               READ feed
               WRITE setFeed
               NOTIFY feedChanged)
    Q_PROPERTY(bool refreshRequired
               READ refreshRequired
               NOTIFY refreshRequiredChanged)
#endif

public:
    enum Roles {
        IdRole = Qt::UserRole + 1,
        TitleRole,
        ArtistRole,
        ArtistIdRole,
        ArtistAvatarUrlRole,
        ThumbnailUrlRole,
#ifndef QML_USER_INTERFACE
        ArtistAvatarRole,
        ThumbnailRole,
        ArtistAvatarPressedRole,
        ThumbnailPressedRole,
#endif
        BodyRole,
        DateRole,
        ActivityTypeRole
    };

public:
    explicit ActivityListModel(QObject *parent = 0);
    ~ActivityListModel();
#if QT_VERSION >= 0x050000
    QHash<int, QByteArray> roleNames() const;
#endif
    inline QUrl feed() const { return m_feed; }
    void setFeed(const QUrl &feed);
    inline bool loading() const { return m_loading; }
    inline bool refreshRequired() const { return m_refresh; }
    inline bool moreResults() const { return m_moreResults; }
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant data(const QModelIndex &index, int role) const;
#if QT_VERSION >= 0x040600
    Q_INVOKABLE QVariant data(int row, const QByteArray &role) const;
#endif
#ifndef QML_USER_INTERFACE
    bool setData(const QModelIndex &index, const QVariant &value, int role);
#endif
    QSharedPointer<ActivityItem> get(int row) const;
#ifdef QML_USER_INTERFACE
    Q_INVOKABLE ActivityItem* getFromQML(int row) const;
#endif

public slots:
    void clear();
    void reset();
    void getActivities(QUrl feed = QUrl());
    void getMoreActivities();
    void insertActivity(int row, QSharedPointer<ActivityItem> activity);
    void appendActivity(QSharedPointer<ActivityItem> activity);
    bool removeActivity(const QVariant &value, int role = IdRole);

private:
    inline void setLoading(bool loading) { m_loading = loading; emit loadingChanged(loading); }
    inline void setMoreResults(bool more) { m_moreResults = more; }
    inline bool error() const { return m_error; }
    inline void setError(bool error) { m_error = error; }
    inline QString cursor() const { return m_cursor; }
    inline void setCursor(const QString &cursor) { m_cursor = cursor; }
    inline void setRefreshRequired(bool refresh) { m_refresh = refresh; emit refreshRequiredChanged(refresh); }
    bool removeRow(int row, const QModelIndex &parent = QModelIndex());

private slots:
    void addActivities();
    inline void onError() { this->setError(true); this->setLoading(false); }
    inline void stopLoading() { this->setLoading(false); emit countChanged(this->rowCount()); }
#ifndef QML_USER_INTERFACE
    void onThumbnailReady();
#endif

signals:
    void loadingChanged(bool loading);
    void countChanged(int count);
    void refreshRequiredChanged(bool refresh);
    void feedChanged(const QUrl &feed);
    
private:
#ifndef QML_USER_INTERFACE
    ThumbnailCache *m_cache;
    int m_artistAvatarPressedRow;
    int m_thumbnailPressedRow;
#endif
    QUrl m_feed;
    QString m_cursor;
    bool m_loading;
    bool m_moreResults;
    bool m_error;
    bool m_refresh;
    QList< QSharedPointer<ActivityItem> >  m_list;
#if QT_VERSION >= 0x040600
    QHash<int, QByteArray> m_roleNames;
#endif
};

#endif // ACTIVITYLISTMODEL_H
