#include "mediaplayer.h"
#include "../../base/playbacklistmodel.h"
#include "../../base/session.h"
#include "../../base/soundcloud.h"
#include "../../base/notifications.h"
#include "../../base/settings.h"
#include "../../base/lastfm.h"
#include "../../base/database.h"
#include "../../base/utils.h"

MediaPlayer* MediaPlayer::m_instance = 0;

MediaPlayer::MediaPlayer(QObject *parent) :
    QObject(parent),
    m_mediaPlayer(new Phonon::MediaObject(this)),
    m_audioOutput(new Phonon::AudioOutput(Phonon::MusicCategory, this)),
    m_index(0),
    m_playbackMode(PlaybackMode::Sequential)
{
    if (!m_instance) {
        m_instance = this;
    }

    Phonon::createPath(m_mediaPlayer, m_audioOutput);

    m_mediaPlayer->setTickInterval(1000);

    this->connect(m_mediaPlayer, SIGNAL(tick(qint64)), this, SIGNAL(positionChanged(qint64)));
    this->connect(m_mediaPlayer, SIGNAL(totalTimeChanged(qint64)), this, SIGNAL(durationChanged(qint64)));
    this->connect(m_mediaPlayer, SIGNAL(stateChanged(Phonon::State,Phonon::State)), this, SLOT(onStateChanged(Phonon::State)));
    this->connect(m_mediaPlayer, SIGNAL(finished()), this, SLOT(onPlaybackFinished()));
    this->connect(m_audioOutput, SIGNAL(volumeChanged(qreal)), this, SIGNAL(volumeChanged(qreal)));
    this->connect(m_audioOutput, SIGNAL(mutedChanged(bool)), this, SIGNAL(mutedChanged(bool)));
#ifdef MEDIA_PLAYERS_ENABLED
    this->connect(Settings::instance(), SIGNAL(mediaPlayerChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
#endif
    this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
#if (defined (MEDIA_PLAYERS_ENABLED) && (defined (QML_USER_INTERFACE)))
    this->onMediaPlayerChanged(Settings::instance()->mediaPlayer());
#else
    if (PlaybackListModel::playbackQueue()->rowCount() > 0) {
        this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(0));
    }
#endif

#ifdef MEEGO_EDITION_HARMATTAN
    m_resourceSet = new ResourcePolicy::ResourceSet("player", this);
    m_audioResource = new ResourcePolicy::AudioResource("player");
    m_audioResource->setProcessID(QCoreApplication::applicationPid());
    m_audioResource->setStreamTag("media.name", "*");
    m_audioResource->setOptional(false);
    m_resourceSet->addResourceObject(m_audioResource);

    this->connect(m_resourceSet, SIGNAL(resourcesGranted(QList<ResourcePolicy::ResourceType>)), this, SLOT(onResourcesGranted()));
    this->connect(m_resourceSet, SIGNAL(resourcesReleased()), this, SLOT(onResourcesReleased()));
    this->connect(m_resourceSet, SIGNAL(resourcesReleasedByManager()), this, SLOT(onResourcesReleased()));
    this->connect(m_resourceSet, SIGNAL(resourcesDenied()), this, SLOT(onResourcesDenied()));
    this->connect(m_resourceSet, SIGNAL(lostResources()), this, SLOT(onResourcesLost()));
#endif
}

MediaPlayer::~MediaPlayer() {
#ifdef MEEGO_EDITION_HARMATTAN
    m_resourceSet->release();
#endif
}

MediaPlayer* MediaPlayer::instance() {
    return !m_instance ? new MediaPlayer(Session::instance()) : m_instance;
}

Phonon::MediaObject* MediaPlayer::mediaObject() const {
    return m_mediaPlayer;
}

Phonon::AudioOutput* MediaPlayer::audioOutput() const {
    return m_audioOutput;
}

#ifdef QML_USER_INTERFACE
void MediaPlayer::playTrack(TrackItem *track, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTrackFromQML(track, false);
    this->setCurrentIndex(0, playImmediately);
}

void MediaPlayer::playTracks(QList<TrackItem*> tracks, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTracksFromQML(tracks, false);
    this->setCurrentIndex(0, playImmediately);
}
#else
void MediaPlayer::playTrack(QSharedPointer<TrackItem> track, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTrack(track, false);
    this->setCurrentIndex(0, playImmediately);
}

void MediaPlayer::playTracks(QList<QSharedPointer<TrackItem> > tracks, bool playImmediately) {
    PlaybackListModel::playbackQueue()->clear();
    PlaybackListModel::playbackQueue()->addTracks(tracks, false);
    this->setCurrentIndex(0, playImmediately);
}
#endif
void MediaPlayer::removeTrack(int row) {
    PlaybackListModel::playbackQueue()->removeTrack(row);

    if (row == this->currentIndex()) {
        if (PlaybackListModel::playbackQueue()->rowCount() > 0) {
            this->setCurrentIndex(row);
        }
    }
    else if (row < this->currentIndex()) {
        m_index--;
        emit currentIndexChanged(this->currentIndex());
    }
}

void MediaPlayer::clearTracks() {
    PlaybackListModel::playbackQueue()->clear();
}

void MediaPlayer::setCurrentTrack(QSharedPointer<TrackItem> track) {
    m_track = track;
#ifdef QML_USER_INTERFACE
    emit currentTrackChanged(track.data());
#else
    emit currentTrackChanged(track);
#endif
    if (track.isNull()) {
        return;
    }

    switch (track.data()->service()) {
    case Services::NoService:
        if ((Settings::instance()->archiveOnline()) && (!track.data()->favourite()) && (!track.data()->id().isEmpty()) && (SoundCloud::instance()->userSignedIn())) {
            SoundCloud::instance()->trackIsFavourite(track.data()->id());
        }

        return;
    default:
        return;
    }
}

void MediaPlayer::setCurrentIndex(int index, bool playImmediately) {
    m_index = index;
    emit currentIndexChanged(index);

    if ((index >= 0) && (index < PlaybackListModel::playbackQueue()->rowCount())) {
        this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(index));

        if (playImmediately) {
            this->play(true);
        }
    }
}

void MediaPlayer::play(bool resetSource) {
    if (((m_mediaPlayer->currentSource().type() == Phonon::MediaSource::Empty) || (resetSource)) && (!this->currentTrack().isNull())) {
        switch (this->currentTrack().data()->service()) {
        case Services::NoService:
            m_mediaPlayer->setCurrentSource(this->currentTrack().data()->url());
            break;
        default:
            m_mediaPlayer->setCurrentSource(SoundCloud::instance()->getStreamUrl(this->currentTrack().data()->streamUrl()));
            break;
        }
    }
#ifdef MEEGO_EDITION_HARMATTAN
        m_resourceSet->acquire();
#else
    m_mediaPlayer->play();
#endif
}

void MediaPlayer::togglePlayPause() {
    switch (this->state()) {
    case Phonon::PlayingState:
        this->pause();
        return;
    default:
        this->play();
        return;
    }
}

void MediaPlayer::pause() {
    m_mediaPlayer->pause();
}

void MediaPlayer::stop() {
    m_mediaPlayer->stop();
}

void MediaPlayer::setPlaybackMode(PlaybackMode::Mode mode) {
    if (mode != this->playbackMode()) {
        m_playbackMode = mode;
        emit playbackModeChanged(mode);
    }
}

void MediaPlayer::setSequentialPlaybackMode() {
    this->setPlaybackMode(PlaybackMode::Sequential);
}

void MediaPlayer::setRepeatAllPlaybackMode() {
    this->setPlaybackMode(PlaybackMode::RepeatAll);
}

void MediaPlayer::setRepeatOnePlaybackMode() {
    this->setPlaybackMode(PlaybackMode::RepeatOne);
}

void MediaPlayer::togglePlaybackMode() {
    switch (this->playbackMode()) {
    case PlaybackMode::Sequential:
        this->setPlaybackMode(PlaybackMode::RepeatAll);
        return;
    case PlaybackMode::RepeatAll:
        this->setPlaybackMode(PlaybackMode::RepeatOne);
        return;
    default:
        this->setPlaybackMode(PlaybackMode::Sequential);
        return;
    }
}

bool MediaPlayer::playing() const {
    switch (this->state()) {
    case Phonon::PlayingState:
        return true;
    default:
        return false;
    }
}

bool MediaPlayer::paused() const {
    switch (this->state()) {
    case Phonon::PausedState:
        return true;
    default:
        return false;
    }
}

void MediaPlayer::next() {
    switch (this->playbackMode()) {
    case PlaybackMode::RepeatAll:
        if (this->currentIndex() < PlaybackListModel::playbackQueue()->rowCount() - 1) {
            this->setCurrentIndex(this->currentIndex() + 1);
        }
        else {
            this->setCurrentIndex(0);
        }

        return;
    case PlaybackMode::RepeatOne:
        this->play();
        return;
    default:
        if (this->currentIndex() < PlaybackListModel::playbackQueue()->rowCount() - 1) {
            this->setCurrentIndex(this->currentIndex() + 1);
        }

        return;
    }
}

void MediaPlayer::previous() {
    if (this->currentIndex() > 0) {
        this->setCurrentIndex(this->currentIndex() - 1);
    }
}

qint64 MediaPlayer::position() const {
    return m_mediaPlayer->currentTime();
}

void MediaPlayer::setPosition(qint64 position) {
    m_mediaPlayer->seek(position);
}

qint64 MediaPlayer::duration() const {
    qint64 d = 0;

    if (m_mediaPlayer->totalTime() > 0) {
        d = m_mediaPlayer->totalTime();
    }
    else if (!this->currentTrack().isNull()) {
        d = this->currentTrack().data()->duration();
    }

    return d;
}

qreal MediaPlayer::volume() const {
    return m_audioOutput->volume();
}

void MediaPlayer::setVolume(qreal volume) {
    m_audioOutput->setVolume(volume);
}

bool MediaPlayer::muted() const {
    return m_audioOutput->isMuted();
}

void MediaPlayer::setMuted(bool muted) {
    m_audioOutput->setMuted(muted);
}

void MediaPlayer::toggleMuted() {
    this->setMuted(!this->muted());
}

void MediaPlayer::startTick() {
    m_mediaPlayer->setTickInterval(1000);
}

void MediaPlayer::stopTick() {
    m_mediaPlayer->setTickInterval(0);
}

Phonon::State MediaPlayer::state() const {
    return m_mediaPlayer->state();
}

void MediaPlayer::onStateChanged(Phonon::State state) {
    switch (state) {
    case Phonon::PausedState:
#ifdef MEEGO_EDITION_HARMATTAN
        m_resourceSet->release();
#endif
        break;
    case Phonon::StoppedState:
#ifdef MEEGO_EDITION_HARMATTAN
        m_resourceSet->release();
#endif
        break;
    case Phonon::ErrorState:
        this->onError(m_mediaPlayer->errorString());
#ifdef MEEGO_EDITION_HARMATTAN
        m_resourceSet->release();
#endif
        break;
    default:
        break;
    }

    emit stateChanged(state);
}

void MediaPlayer::onPlaybackFinished() {
    this->onTrackPlayed(this->currentTrack());
    this->setPosition(0);
    this->next();
}

void MediaPlayer::onError(const QString &errorString) {
    Notifications::instance()->onError(QString("%1: %2").arg(tr("Playback error")).arg(errorString));
}

#ifdef MEDIA_PLAYERS_ENABLED
void MediaPlayer::onMediaPlayerChanged(const QString &mediaPlayer) {
    if (mediaPlayer != "musikloud") {
        this->stop();
#ifdef QML_USER_INTERFACE
        this->setCurrentIndex(0);
        this->setCurrentTrack(QSharedPointer<TrackItem>());
        this->disconnect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
#else
        this->deleteLater();
#endif
    }
#ifdef QML_USER_INTERFACE
    else {
        this->onCountChanged(PlaybackListModel::playbackQueue()->rowCount());
        this->connect(PlaybackListModel::playbackQueue(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    }
#endif
}
#endif

void MediaPlayer::onTrackPlayed(QSharedPointer<TrackItem> track) {
    track.data()->setPlayCount(track.data()->playCount() + 1);
    track.data()->setLastPlayed(Utils::currentMSecsSinceEpoch());

    switch (track.data()->service()) {
    case Services::NoService:
        QMetaObject::invokeMethod(Database::instance(), "updateTrack", Q_ARG(QUrl, track.data()->url()), Q_ARG(QString, "playCount"), Q_ARG(QVariant, track.data()->playCount()));
        QMetaObject::invokeMethod(Database::instance(), "updateTrack", Q_ARG(QUrl, track.data()->url()), Q_ARG(QString, "lastPlayed"), Q_ARG(QVariant, track.data()->lastPlayed()));
        break;
    default:
        break;
    }

    if ((Settings::instance()->scrobbleTracks()) && (Lastfm::instance()->userSignedIn()) && (!track.data()->artist().isEmpty())) {
        Lastfm::instance()->scrobbleTrack(track.data()->artist(), track.data()->title());
    }
}

void MediaPlayer::onCountChanged(int count) {
    switch (count) {
    case 0:
        this->stop();
        this->setCurrentIndex(0);
        this->setCurrentTrack(QSharedPointer<TrackItem>());
        return;
    default:
        if (this->currentTrack().isNull()) {
            this->setCurrentTrack(PlaybackListModel::playbackQueue()->get(0));
        }

        return;
    }
}

#ifdef MEEGO_EDITION_HARMATTAN
void MediaPlayer::onResourcesGranted() {
    m_mediaPlayer->play();
}

void MediaPlayer::onResourcesReleased() {
    qDebug() << "Resources released";
}

void MediaPlayer::onResourcesDenied() {
    qDebug() << "Resources denied";
}

void MediaPlayer::onResourcesLost() {
    qDebug() << "Resources lost";
}
#endif
