#include "tracklisttab.h"
#include "trackinfowindow.h"
#include "tracklistdelegate.h"
#include "trackselectionwindow.h"
#include "nowplayingwindow.h"
#include "windowstack.h"
#include "listview.h"
#include "nowplayingbutton.h"
#include "../base/soundcloud.h"
#include "../base/settings.h"
#include "../base/transfermanager.h"
#include "../base/playbacklistmodel.h"
#include "../base/notifications.h"
#include "../base/tracklistmodel.h"
#include "../base/tracklauncher.h"
#include "../mediaplayer/mobility/mediaplayer.h"
#include <QAction>
#include <QLabel>
#include <QGridLayout>
#include <QMenu>
#include <QTimer>
#include <QMaemo5Style>

TrackListTab::TrackListTab(QWidget *parent) :
    AbstractTab(parent),
    m_model(new TrackListModel(this)),
    m_listView(new ListView(this)),
    m_addToFavouritesAction(new QAction(tr("Add to favourites"), this)),
    m_deleteFromFavouritesAction(new QAction(tr("Delete from favourites"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_trackDownloadAction(new QAction(tr("Download"), this)),
    m_contextMenu(new QMenu(this)),
    m_favouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(toggleFavourite()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueTrack()))),
    m_trackDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addTrackDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No tracks found")), this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_addToFavouritesAction);
    this->addAction(m_deleteFromFavouritesAction);
    this->addAction(m_addToQueueAction);
    this->addAction(m_trackDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    TrackListDelegate *delegate = new TrackListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_listView->setMonitorScrolling(true);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playTrack(QModelIndex)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreTracks()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_addToFavouritesAction, SIGNAL(triggered()), this, SLOT(addTracksToFavourites()));
    this->connect(m_deleteFromFavouritesAction, SIGNAL(triggered()), this, SLOT(deleteTracksFromFavourites()));
    this->connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueTracks()));
    this->connect(m_trackDownloadAction, SIGNAL(triggered()), this, SLOT(addTrackDownloads()));

    if (Settings::instance()->mediaPlayer() == "musikloud") {
        this->connect(m_listView, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
        this->connect(m_listView, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
    }

    this->onOrientationChanged();
}

void TrackListTab::showEvent(QShowEvent *event) {
    AbstractTab::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void TrackListTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void TrackListTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void TrackListTab::setTrackFeed(const QUrl &feed) {
    m_model->getTracks(feed);
    this->setMenuActions();
}

void TrackListTab::setMenuActions() {
    m_addToFavouritesAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (m_model->feed() != FAVOURITES_FEED));
    m_deleteFromFavouritesAction->setEnabled((SoundCloud::instance()->userSignedIn()) && (m_model->feed() == FAVOURITES_FEED));
    m_favouritesContextAction->setVisible((SoundCloud::instance()->userSignedIn()) && (m_model->feed() == FAVOURITES_FEED));
}

void TrackListTab::onLoadingChanged(bool loading) {
    Notifications::toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void TrackListTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void TrackListTab::addTracksToFavourites() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Add to favourites"), SLOT(addTracksToFavourites()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void TrackListTab::deleteTracksFromFavourites() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Delete from favourites"), SLOT(deleteTracksFromFavourites()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void TrackListTab::addTrackDownloads() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Download"), SLOT(addTrackDownloads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void TrackListTab::queueTracks() {
    TrackSelectionWindow *window = new TrackSelectionWindow(m_model, tr("Add to queue"), SLOT(queueTracks()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void TrackListTab::showContextMenu(const QPoint &pos) {
    if (m_model->data(m_listView->currentIndex(), TrackListModel::FavouriteRole).toBool()) {
        m_favouritesContextAction->setText(tr("Delete from favourites"));
    }
    else {
        m_favouritesContextAction->setText(tr("Add to favourites"));
    }

    m_contextMenu->popup(pos, m_favouritesContextAction);
}

void TrackListTab::toggleFavourite() {
    if (m_model->data(m_listView->currentIndex(), TrackListModel::FavouriteRole).toBool()) {
        SoundCloud::instance()->deleteFromFavourites(QStringList(m_model->data(m_listView->currentIndex(), TrackListModel::IdRole).toString()));
    }
    else {
        SoundCloud::instance()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), TrackListModel::IdRole).toString()));
    }
}

void TrackListTab::addTrackDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void TrackListTab::queueTrack() {
    PlaybackListModel::playbackQueue()->addTrack(m_model->get(m_listView->currentIndex().row()));
}

void TrackListTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showTrackInfo(m_model->get(index.row()));
    }
}

void TrackListTab::showTrackInfo(QSharedPointer<TrackItem> track) {
    TrackInfoWindow *window = new TrackInfoWindow(track, WindowStack::instance()->currentWindow());
    window->show();
}

void TrackListTab::playTrack(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "musikloud") {
        MediaPlayer::instance()->playTrack(m_model->get(index.row()));
        NowPlayingWindow *window = new NowPlayingWindow(WindowStack::instance()->currentWindow());
        window->show();
    }
    else {
        TrackLauncher::launchTrack(SoundCloud::instance()->getStreamUrl(index.data(TrackListModel::StreamUrlRole).toUrl()));
    }
}
