#include "transferlistwindow.h"
#include "transferlistdelegate.h"
#include "transferlistmodel.h"
#include <QListView>
#include <QMenu>
#include <QVBoxLayout>

TransferListWindow::TransferListWindow(QWidget *parent) :
    QWidget(parent),
    m_model(new TransferListModel(this)),
    m_view(new QListView(this)),
    m_contextMenu(new QMenu(this)),
    m_pauseResumeAction(m_contextMenu->addAction(tr("Pause"), this, SLOT(pauseOrResumeSelectedTransfer()))),
    m_cancelAction(m_contextMenu->addAction(tr("Cancel"), this, SLOT(cancelSelectedTransfer()))),
    m_separatorAction(m_contextMenu->addSeparator()),
    m_highPriorityAction(m_contextMenu->addAction(tr("High priority"), this, SLOT(setSelectedTransferPriorityToHigh()))),
    m_normalPriorityAction(m_contextMenu->addAction(tr("Normal priority"), this, SLOT(setSelectedTransferPriorityToNormal()))),
    m_lowPriorityAction(m_contextMenu->addAction(tr("Low priority"), this, SLOT(setSelectedTransferPriorityToLow()))),
    m_priorityActionGroup(new QActionGroup(m_contextMenu)),
    m_vbox(new QVBoxLayout(this))
{
    this->setWindowTitle(tr("Transfers"));

    m_view->setModel(m_model);
    m_view->setItemDelegate(new TransferListDelegate(m_view));
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);

    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);

    m_highPriorityAction->setCheckable(true);
    m_normalPriorityAction->setCheckable(true);
    m_lowPriorityAction->setCheckable(true);

    m_priorityActionGroup->setExclusive(true);
    m_priorityActionGroup->addAction(m_highPriorityAction);
    m_priorityActionGroup->addAction(m_normalPriorityAction);
    m_priorityActionGroup->addAction(m_lowPriorityAction);

    this->connect(m_view, SIGNAL(activated(QModelIndex)), this, SLOT(onListItemActivated(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
}

TransferListWindow::~TransferListWindow() {}

void TransferListWindow::onListItemActivated(const QModelIndex &index) {
    if (index.isValid()) {
        m_model->setData(index, Transfers::Queued, TransferListModel::StatusRole);
    }
}

void TransferListWindow::showContextMenu(const QPoint &pos) {
    if (m_view->currentIndex().isValid()) {
        switch (m_view->currentIndex().data(TransferListModel::StatusRole).toInt()) {
        case Transfers::Paused:
        case Transfers::Failed:
            m_pauseResumeAction->setText(tr("Resume"));
            break;
        default:
            m_pauseResumeAction->setText(tr("Pause"));
            break;
        }

        switch (m_view->currentIndex().data(TransferListModel::PriorityRole).toInt()) {
        case Transfers::HighPriority:
            m_highPriorityAction->setChecked(true);
            break;
        case Transfers::LowPriority:
            m_lowPriorityAction->setChecked(true);
            break;
        default:
            m_normalPriorityAction->setChecked(true);
            break;
        }

        m_contextMenu->setActiveAction(m_pauseResumeAction);
        m_contextMenu->popup(this->mapToGlobal(pos), m_pauseResumeAction);
    }
}

void TransferListWindow::pauseOrResumeSelectedTransfer() {
    if (m_view->currentIndex().isValid()) {
        switch (m_view->currentIndex().data(TransferListModel::StatusRole).toInt()) {
        case Transfers::Paused:
        case Transfers::Failed:
            m_model->setData(m_view->currentIndex(), Transfers::Queued, TransferListModel::StatusRole);
            return;
        default:
            m_model->setData(m_view->currentIndex(), Transfers::Paused, TransferListModel::StatusRole);
            return;
        }
    }
}

void TransferListWindow::cancelSelectedTransfer() {
    if (m_view->currentIndex().isValid()) {
        m_model->setData(m_view->currentIndex(), Transfers::Cancelled, TransferListModel::StatusRole);
    }
}

void TransferListWindow::setSelectedTransferPriorityToHigh() {
    if (m_view->currentIndex().isValid()) {
        m_model->setData(m_view->currentIndex(), Transfers::HighPriority, TransferListModel::PriorityRole);
    }
}

void TransferListWindow::setSelectedTransferPriorityToNormal() {
    if (m_view->currentIndex().isValid()) {
        m_model->setData(m_view->currentIndex(), Transfers::NormalPriority, TransferListModel::PriorityRole);
    }
}

void TransferListWindow::setSelectedTransferPriorityToLow() {
    if (m_view->currentIndex().isValid()) {
        m_model->setData(m_view->currentIndex(), Transfers::LowPriority, TransferListModel::PriorityRole);
    }
}
