#include "transferlistmodel.h"
#include "../base/transfermanager.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

TransferListModel::TransferListModel(QObject *parent) :
    QAbstractListModel(parent)
{
#if QT_VERSION >= 0x040600
    QHash<int, QByteArray> roles;
    roles[TitleRole] = "title";
    roles[DownloadPathRole] = "downloadPath";
    roles[StatusRole] = "status";
    roles[StatusTextRole] = "statusText";
    roles[StatusInfoRole] = "statusInfo";
    roles[ProgressRole] = "progress";
    roles[SizeRole] = "size";
    roles[CurrentIndexRole] = "currentIndex";
    roles[TotalFilesRole] = "totalFiles";
    roles[PriorityRole] = "priority";
    roles[PriorityTextRole] = "priorityText";
    roles[ServiceRole] = "service";
    roles[TransferTypeRole] = "transferType";
    roles[SelectedRole] = "selected";
    this->setRoleNames(roles);
#endif

    for (int i = 0; i < TransferManager::instance()->transferList()->size(); i++) {
        QSharedPointer<TransferItem> transfer = TransferManager::instance()->transferList()->at(i);
        this->connect(transfer.data(), SIGNAL(progressChanged(int)), this, SLOT(onTransferDataChanged()));
        this->connect(transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferDataChanged()));
        this->connect(transfer.data(), SIGNAL(currentIndexChanged(int)), this, SLOT(onTransferDataChanged()));
        this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferDataChanged()));
        this->connect(transfer.data(), SIGNAL(statusInfoChanged(QString)), this, SLOT(onTransferDataChanged()));
        this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::TransferPriority)), this, SLOT(onTransferDataChanged()));
    }

    this->connect(TransferManager::instance(), SIGNAL(transferRemoved(int)), this, SLOT(onTransferRemoved(int)));

    emit countChanged(this->rowCount());
}

TransferListModel::~TransferListModel() {}

int TransferListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return TransferManager::instance()->transferList()->size();
}

QVariant TransferListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case TitleRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->title();
    case StatusRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->status();
    case StatusTextRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->statusText();
    case StatusInfoRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->statusInfo();
    case ProgressRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->progress();
    case SizeRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->size();
    case CurrentIndexRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->currentIndex();
    case TotalFilesRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->totalFiles();
    case PriorityRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->priority();
    case PriorityTextRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->priorityText();
    case ServiceRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->service();
    case TransferTypeRole:
        return TransferManager::instance()->transferList()->at(index.row()).data()->transferType();
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    default:
        return QVariant();
    }
}

bool TransferListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    if (!index.isValid()) {
        return false;
    }

    switch (role) {
    case DownloadPathRole:
        TransferManager::instance()->transferList()->at(index.row()).data()->setDownloadPath(value.toString());
        break;
    case StatusRole:
        TransferManager::instance()->transferList()->at(index.row()).data()->setStatus(static_cast<Transfers::TransferStatus>(value.toInt()));
        break;
    case PriorityRole:
        TransferManager::instance()->transferList()->at(index.row()).data()->setPriority(static_cast<Transfers::TransferPriority>(value.toInt()));
        break;
    default:
        return false;
    }

    return true;
}

#if QT_VERSION >= 0x040600
QVariant TransferListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

bool TransferListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return setData(this->index(row, 0), value, this->roleNames().key(role));
}
#endif

void TransferListModel::onTransferDataChanged() {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
        int row = this->indexOf(transfer);
        emit dataChanged(this->index(row), this->index(row));
    }
}

void TransferListModel::onTransferRemoved(int row) {
    this->beginRemoveRows(QModelIndex(), row, row);
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

QSharedPointer<TransferItem> TransferListModel::get(int row) const {
    if ((row >= 0) && (row < this->rowCount())) {
        return TransferManager::instance()->transferList()->at(row);
    }

    return QSharedPointer<TransferItem>();
}

#ifdef QML_USER_INTERFACE
TransferItem* TransferListModel::getFromQML(int row) const {
    if (TransferItem *transfer = this->get(row).data()) {
        QDeclarativeEngine::setObjectOwnership(transfer, QDeclarativeEngine::CppOwnership);
        return transfer;
    }

    return 0;
}
#endif

void TransferListModel::selectRow(int row) {
    m_selectedRows.append(row);

    emit dataChanged(this->index(row), this->index(row));
    emit itemsSelectedChanged(this->itemsSelected());
}

void TransferListModel::unselectRow(int row) {
    m_selectedRows.removeOne(row);

    emit dataChanged(this->index(row), this->index(row));
    emit itemsSelectedChanged(this->itemsSelected());
}

void TransferListModel::toggleSelected(int row) {
    if (m_selectedRows.contains(row)) {
        this->unselectRow(row);
    }
    else {
        this->selectRow(row);
    }

    emit itemsSelectedChanged(this->itemsSelected());
}

void TransferListModel::selectAll() {
    m_selectedRows.clear();

    for (int i = 0; i < this->rowCount(); i++) {
        m_selectedRows.append(i);
    }

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    emit itemsSelectedChanged(true);
}

void TransferListModel::selectNone() {
    m_selectedRows.clear();

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    emit itemsSelectedChanged(false);
}

void TransferListModel::pauseSelectedTransfers() {
    QList<int> rows = m_selectedRows;
    qSort(rows.begin(), rows.end(), qLess<int>());

    foreach (int i, rows) {
        this->setData(this->index(i), Transfers::Paused, StatusRole);
    }
}

void TransferListModel::resumeSelectedTransfers() {
    QList<int> rows = m_selectedRows;
    qSort(rows.begin(), rows.end(), qLess<int>());

    foreach (int i, rows) {
        this->setData(this->index(i), Transfers::Queued, StatusRole);
    }
}

void TransferListModel::cancelSelectedTransfers() {
    QList<int> rows = m_selectedRows;
    qSort(rows.begin(), rows.end(), qGreater<int>());

    foreach (int i, rows) {
        this->setData(this->index(i), Transfers::Cancelled, StatusRole);
    }
}

int TransferListModel::indexOf(TransferItem *transfer) const {
    for (int i = 0; i < this->rowCount(); i++) {
        if (TransferManager::instance()->transferList()->at(i).data() == transfer) {
            return i;
        }
    }

    return -1;
}
