#include "navigationbar.h"
#include "../base/utils.h"
#include <QHBoxLayout>
#include <QPainter>
#include <QPaintEvent>
#include <QButtonGroup>

NavigationBar::NavigationBar(QWidget *parent) :
    QWidget(parent),
    m_backButton(new NavigationBackButton(this)),
    m_buttonGroup(new QButtonGroup(this)),
    m_hbox(new QHBoxLayout(this))
{
    this->setFixedHeight(58);
    this->setAttribute(Qt::WA_OpaquePaintEvent);

    m_buttonGroup->setExclusive(true);

    m_hbox->setContentsMargins(0, 0, 0, 0);
    m_hbox->setSpacing(0);
    m_hbox->addWidget(m_backButton);

    this->connect(m_backButton, SIGNAL(clicked()), this, SLOT(pop()));
    this->connect(m_buttonGroup, SIGNAL(buttonClicked(QAbstractButton*)), this, SLOT(onButtonClicked(QAbstractButton*)));
}

NavigationBar::~NavigationBar() {}

void NavigationBar::focusInEvent(QFocusEvent *event) {
    if (event->gotFocus()) {
        if (!m_buttonGroup->buttons().isEmpty()) {
            m_buttonGroup->buttons().last()->setFocus(Qt::OtherFocusReason);
        }
        else {
            m_backButton->setFocus(Qt::OtherFocusReason);
        }
    }

    QWidget::focusInEvent(event);
}

void NavigationBar::keyPressEvent(QKeyEvent *event) {
    switch (event->key()) {
    case Qt::Key_Left:
        if ((!m_buttonGroup->buttons().isEmpty()) && (m_buttonGroup->buttons().first()->hasFocus())) {
            m_backButton->setFocus(Qt::OtherFocusReason);
        }

        event->accept();
        return;
    case Qt::Key_Right:
        if ((m_backButton->hasFocus()) && (!m_buttonGroup->buttons().isEmpty())) {
            m_buttonGroup->buttons().first()->setFocus(Qt::OtherFocusReason);
        }

        event->accept();
        return;
    default:
        event->ignore();
        return;
    }
}

void NavigationBar::paintEvent(QPaintEvent *event) {
    QPainter painter(this);

    Utils::drawBorderImage(&painter, this->rect(), QImage("/usr/share/themes/default/images/qgn_plat_breadcrumb_end_normal.png"), 5, 5, 4, 5);

    event->accept();
}

void NavigationBar::push(const QString &title) {
    if (!m_buttonGroup->buttons().isEmpty()) {
        if (NavigationButton *navButton = qobject_cast<NavigationButton*>(m_buttonGroup->buttons().last())) {
            navButton->setLast(false);
        }
    }

    NavigationButton *button = new NavigationButton(title, this);
    button->setFirst(m_buttonGroup->buttons().isEmpty());

    m_buttonGroup->addButton(button, m_buttonGroup->buttons().size());
    m_hbox->addWidget(button);
    m_backButton->setEnabled(m_buttonGroup->buttons().size() > 1);
}

void NavigationBar::pop(int to) {
    if ((to < 0) || (to >= m_buttonGroup->buttons().size())) {
        to = m_buttonGroup->buttons().count() - 1;
    }

    while ((m_buttonGroup->buttons().size() > (to + 1)) && (m_buttonGroup->buttons().size() > 1)) {
        QAbstractButton *button = m_buttonGroup->buttons().last();
        m_buttonGroup->removeButton(button);
        m_hbox->removeWidget(button);
        button->deleteLater();
    }

    for (int i = 0; i < m_buttonGroup->buttons().size(); i++) {
        if (NavigationButton *navButton = qobject_cast<NavigationButton*>(m_buttonGroup->buttons().at(i))) {
            navButton->setFirst(i == 0);
            navButton->setLast(i == (m_buttonGroup->buttons().size() - 1));
        }
    }

    emit popped(m_buttonGroup->buttons().size() - 1);

    m_backButton->setEnabled(m_buttonGroup->buttons().size() > 1);
}

void NavigationBar::onButtonClicked(QAbstractButton *button) {
    int i = m_buttonGroup->buttons().indexOf(button);

    if (i < (m_buttonGroup->buttons().size() - 1)) {
        this->pop(i);
    }
}

NavigationBackButton::NavigationBackButton(QWidget *parent) :
    QAbstractButton(parent)
{
    this->setFixedSize(58, 58);
}

NavigationBackButton::~NavigationBackButton() {}

void NavigationBackButton::keyPressEvent(QKeyEvent *event) {
    switch (event->key()) {
    case Qt::Key_Enter:
    case Qt::Key_Return:
        this->setDown(true);
        event->accept();
        return;
    case Qt::Key_Space:
        event->ignore();
        return;
    default:
        break;
    }

    QAbstractButton::keyPressEvent(event);
}

void NavigationBackButton::keyReleaseEvent(QKeyEvent *event) {
    switch (event->key()) {
    case Qt::Key_Enter:
    case Qt::Key_Return:
        this->setDown(false);
        this->click();
        event->accept();
        return;
    case Qt::Key_Space:
        event->ignore();
        return;
    default:
        break;
    }

    QAbstractButton::keyReleaseEvent(event);
}

void NavigationBackButton::paintEvent(QPaintEvent *event) {
    QPainter painter(this);

    QString backgroundState("normal");
    QString iconState("normal");

    if (!this->isEnabled()) {
        iconState = "disabled";
    }
    else if (this->isDown()) {
        backgroundState = "pressed";
        iconState = "pressed";
    }
    else if (this->hasFocus()) {
        backgroundState = "focused";
    }

    QImage backgroundImage(QString("/usr/share/themes/default/images/qgn_plat_breadcrumb_backbutton_%1.png").arg(backgroundState));

    Utils::drawBorderImage(&painter, this->rect(), backgroundImage, 6, 6, 5, 6);

    painter.drawImage(12, 12, QImage(QString("/usr/share/themes/default/images/qgn_plat_breadcrumb_arrow_%1.png").arg(iconState)));

    event->accept();
}

NavigationButton::NavigationButton(const QString &text, QWidget *parent) :
    QAbstractButton(parent),
    m_first(false),
    m_last(true)
{
    this->setFixedHeight(58);
    this->setText(text);
}

NavigationButton::~NavigationButton() {}

void NavigationButton::keyPressEvent(QKeyEvent *event) {
    switch (event->key()) {
    case Qt::Key_Enter:
    case Qt::Key_Return:
        this->setDown(true);
        event->accept();
        return;
    case Qt::Key_Space:
        event->ignore();
        return;
    default:
        break;
    }

    QAbstractButton::keyPressEvent(event);
}

void NavigationButton::keyReleaseEvent(QKeyEvent *event) {
    switch (event->key()) {
    case Qt::Key_Enter:
    case Qt::Key_Return:
        this->setDown(false);
        this->click();
        event->accept();
        return;
    case Qt::Key_Space:
        event->ignore();
        return;
    default:
        break;
    }

    QAbstractButton::keyReleaseEvent(event);
}

void NavigationButton::paintEvent(QPaintEvent *event) {
    QPainter painter(this);

    QString backgroundState("normal");
    QString iconState("normal");
    QString position("middle");

    if (!this->isEnabled()) {
        iconState = "disabled";
    }
    else if (this->isDown()) {
        backgroundState = "pressed";
        iconState = "pressed";
    }
    else if (this->hasFocus()) {
        backgroundState = "focused";
    }

    if (this->last()) {
        position = "end";
    }

    QImage backgroundImage(QString("/usr/share/themes/default/images/qgn_plat_breadcrumb_%1_%2.png").arg(position).arg(backgroundState));

    Utils::drawBorderImage(&painter, this->rect(), backgroundImage, 5, 5, 4, 5);

    if (!this->first()) {
        painter.drawImage(12, 12, QImage(QString("/usr/share/themes/default/images/qgn_plat_breadcrumb_separator_%1.png").arg(iconState)));
    }

    QRect textRect = this->rect();
    textRect.setLeft(this->first() ? textRect.left() + 12 : textRect.left() + 40);
    textRect.setRight(textRect.right() - 12);

    painter.setClipRect(textRect);
    painter.drawText(textRect, Qt::AlignVCenter | Qt::TextSingleLine, this->text());

    event->accept();
}

QSize NavigationButton::sizeHint() const {
    if (this->last()) {
        return QAbstractButton::sizeHint();
    }

    if (this->first()) {
        return QSize(QFontMetrics(this->font()).boundingRect(this->text()).width() + 28, 58);
    }

    return QSize(QFontMetrics(this->font()).boundingRect(this->text()).width() + 52, 58);
}

bool NavigationButton::first() const {
    return m_first;
}

void NavigationButton::setFirst(bool first) {
    if (first != this->first()) {
        m_first = first;
        this->update();
    }
}

bool NavigationButton::last() const {
    return m_last;
}

void NavigationButton::setLast(bool last) {
    if (last != this->last()) {
        m_last = last;
        this->setSizePolicy(last ? QSizePolicy::Expanding : QSizePolicy::Maximum, QSizePolicy::Preferred);
        this->update();
    }
}
