#include "transfermanager.h"
#include "database.h"
#include "settings.h"
#include "trackitem.h"
#include "playlistitem.h"
#include "utils.h"

TransferManager* TransferManager::m_instance = 0;

TransferManager::TransferManager(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_transfers(new QList< QSharedPointer<TransferItem> >)
  #ifdef MEEGO_EDITION_HARMATTAN
  ,m_transferUI(new Client(this))
  #endif
{
    if (!m_instance) {
        m_instance = this;
    }
#ifdef MEEGO_EDITION_HARMATTAN
    m_transferUI->init();
#endif
}

TransferManager::~TransferManager() {
    m_transfers->clear();
    delete m_transfers;
    m_transfers = 0;
}

TransferManager* TransferManager::instance() {
    return !m_instance ? new TransferManager : m_instance;
}

void TransferManager::restoreStoredDownloads() {
    QList< QSharedPointer<TransferItem> > transfers = Database::instance()->getStoredDownloads();

    while (!transfers.isEmpty()) {
        QSharedPointer<TransferItem> transfer = transfers.takeFirst();
#ifdef MEEGO_EDITION_HARMATTAN
        transfer.data()->setHarmattanTransfer(m_transferUI->registerTransfer(transfer.data()->title(), Client::TRANSFER_TYPES_DOWNLOAD));
#endif
        transfer.data()->setDownloadPath(Settings::instance()->downloadPath() + ".incomplete/");
        transfer.data()->setNetworkAccessManager(this->networkAccessManager());
        transfer.data()->setStatus(Settings::instance()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused);
        this->transferList()->append(transfer);
        this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
        this->connect(transfer.data(), SIGNAL(trackDownloadCompleted(QSharedPointer<TrackItem>)), Database::instance(), SLOT(addTrack(QSharedPointer<TrackItem>)));
        this->connect(transfer.data(), SIGNAL(playlistDownloadCompleted(QSharedPointer<PlaylistItem>)), Database::instance(), SLOT(addPlaylist(QSharedPointer<PlaylistItem>)));

        if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
            transfer.data()->startTransfer();
        }
    }
}

QString TransferManager::generateTransferId(QString seedFileName) const {
    return QString("%1_%2").arg(QString::number(Utils::currentMSecsSinceEpoch())).arg(seedFileName);
}

void TransferManager::addDownloadTransfer(TrackItem *track, bool notify) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(QSharedPointer<TrackItem>(new TrackItem(track)), Settings::instance()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused));
#ifdef MEEGO_EDITION_HARMATTAN
    transfer.data()->setHarmattanTransfer(m_transferUI->registerTransfer(transfer.data()->title(), Client::TRANSFER_TYPES_DOWNLOAD));
#endif
    transfer.data()->setDownloadPath(Settings::instance()->downloadPath() + ".incomplete/");
    transfer.data()->setNetworkAccessManager(this->networkAccessManager());
    this->transferList()->append(transfer);
    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
    this->connect(transfer.data(), SIGNAL(trackDownloadCompleted(QSharedPointer<TrackItem>)), Database::instance(), SLOT(addTrack(QSharedPointer<TrackItem>)));

    if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
        transfer.data()->startTransfer();
    }

    QMetaObject::invokeMethod(Database::instance(), "storeDownload", Qt::QueuedConnection, Q_ARG(QSharedPointer<TransferItem>, transfer));

    if (notify) {
        emit alert(tr("Track download added to transfer queue"));
    }
}

void TransferManager::addDownloadTransfer(QSharedPointer<TrackItem> track, bool notify) {
    this->addDownloadTransfer(track.data(), notify);
}

void TransferManager::addDownloadTransfer(PlaylistItem *playlist, QList<TrackItem *> tracks, bool notify) {
    QList<QSharedPointer<TrackItem> > newTracks;

    while (!tracks.isEmpty()) {
        newTracks.append(QSharedPointer<TrackItem>(new TrackItem(tracks.takeFirst())));
    }

    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(QSharedPointer<PlaylistItem>(new PlaylistItem(playlist)), newTracks, Settings::instance()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused));
#ifdef MEEGO_EDITION_HARMATTAN
    transfer.data()->setHarmattanTransfer(m_transferUI->registerTransfer(transfer.data()->title(), Client::TRANSFER_TYPES_DOWNLOAD));
#endif
    transfer.data()->setDownloadPath(Settings::instance()->downloadPath() + ".incomplete/");
    transfer.data()->setNetworkAccessManager(this->networkAccessManager());
    this->transferList()->append(transfer);
    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
    this->connect(transfer.data(), SIGNAL(trackDownloadCompleted(QSharedPointer<TrackItem>)), Database::instance(), SLOT(addTrack(QSharedPointer<TrackItem>)));
    this->connect(transfer.data(), SIGNAL(playlistDownloadCompleted(QSharedPointer<PlaylistItem>)), Database::instance(), SLOT(addPlaylist(QSharedPointer<PlaylistItem>)));

    if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
        transfer.data()->startTransfer();
    }

    QMetaObject::invokeMethod(Database::instance(), "storeDownload", Qt::QueuedConnection, Q_ARG(QSharedPointer<TransferItem>, transfer));

    if (notify) {
        emit alert(tr("Set download added to transfer queue"));
    }
}

void TransferManager::addDownloadTransfer(QSharedPointer<PlaylistItem> playlist, QList<QSharedPointer<TrackItem> > tracks, bool notify) {
    QList<QSharedPointer<TrackItem> > newTracks;

    while (!tracks.isEmpty()) {
        newTracks.append(QSharedPointer<TrackItem>(new TrackItem(tracks.takeFirst().data())));
    }

    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(QSharedPointer<PlaylistItem>(new PlaylistItem(playlist.data())), newTracks, Settings::instance()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused));
#ifdef MEEGO_EDITION_HARMATTAN
    transfer.data()->setHarmattanTransfer(m_transferUI->registerTransfer(transfer.data()->title(), Client::TRANSFER_TYPES_DOWNLOAD));
#endif
    transfer.data()->setDownloadPath(Settings::instance()->downloadPath() + ".incomplete/");
    transfer.data()->setNetworkAccessManager(this->networkAccessManager());
    this->transferList()->append(transfer);
    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
    this->connect(transfer.data(), SIGNAL(trackDownloadCompleted(QSharedPointer<TrackItem>)), Database::instance(), SLOT(addTrack(QSharedPointer<TrackItem>)));
    this->connect(transfer.data(), SIGNAL(playlistDownloadCompleted(QSharedPointer<PlaylistItem>)), Database::instance(), SLOT(addPlaylist(QSharedPointer<PlaylistItem>)));

    if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
        transfer.data()->startTransfer();
    }

    QMetaObject::invokeMethod(Database::instance(), "storeDownload", Qt::QueuedConnection, Q_ARG(QSharedPointer<TransferItem>, transfer));

    if (notify) {
        emit alert(tr("Set download added to transfer queue"));
    }
}

void TransferManager::addDownloadTransfers(QList<TrackItem *> tracks) {
    while (!tracks.isEmpty()) {
        this->addDownloadTransfer(tracks.takeFirst(), false);
    }

    emit alert(tr("Track download(s) added to transfer queue"));
}

void TransferManager::addDownloadTransfers(QList< QSharedPointer<TrackItem> > tracks) {
    while (!tracks.isEmpty()) {
        this->addDownloadTransfer(tracks.takeFirst().data(), false);
    }

    emit alert(tr("Track download(s) added to transfer queue"));
}

QSharedPointer<TransferItem> TransferManager::getTransfer(TransferItem *transfer) const {
    for (int i = 0; i < this->transferList()->size(); i++) {
        if (this->transferList()->at(i).data() == transfer) {
            return this->transferList()->at(i);
        }
    }

    return QSharedPointer<TransferItem>();
}

QSharedPointer<TransferItem> TransferManager::getNextTransfer() const {
    int i = 0;
#ifdef MEEGO_EDITION_HARMATTAN
    while (i < this->transferList()->size()) {
        if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
            if (transfer.data()->status() == Transfers::Queued) {
                return transfer;
            }
        }

        i++;
    }
#else
    int priority = Transfers::HighPriority;

    while (priority <= Transfers::LowPriority) {
        while (i < this->transferList()->size()) {
            if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
                if ((transfer.data()->priority() == priority) && (transfer.data()->status() == Transfers::Queued)) {
                    return transfer;
                }
            }

            i++;
        }

        priority++;
        i = 0;
    }
#endif
    return QSharedPointer<TransferItem>();
}

bool TransferManager::removeTransfer(QSharedPointer<TransferItem> transfer) {
    int row = this->transferList()->indexOf(transfer);

    if (row >= 0) {
        this->transferList()->removeAt(row);
        emit transferRemoved(row);
        return true;
    }

    return false;
}

void TransferManager::onTransferStatusChanged(Transfers::TransferStatus status) {
    if ((status <= Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
        if (QSharedPointer<TransferItem> transfer = this->getNextTransfer()) {
            transfer.data()->startTransfer();
        }
    }

    if ((status == Transfers::Completed) || (status == Transfers::Cancelled)) {
        if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
            if (status == Transfers::Completed) {
                this->onTransferCompleted(this->getTransfer(transfer));
            }
            else  {
                this->onTransferCancelled(this->getTransfer(transfer));
            }
        }
    }
}

void TransferManager::onTransferCompleted(QSharedPointer<TransferItem> transfer) {
    emit alert(QString("%1 '%3' %4").arg(tr("Download of")).arg(transfer.data()->title()).arg(tr("completed")));
    this->removeTransfer(transfer);
}

void TransferManager::onTransferCancelled(QSharedPointer<TransferItem> transfer) {
    QMetaObject::invokeMethod(Database::instance(), "removeStoredDownload", Qt::QueuedConnection, Q_ARG(QSharedPointer<TransferItem>, transfer));
    this->removeTransfer(transfer);
}

void TransferManager::onMaximumConcurrentTransfersChanged(int oldMaximum, int newMaximum) {
    if (newMaximum > oldMaximum) {
        if (newMaximum > this->concurrentTransfers()) {
            if (QSharedPointer<TransferItem> transfer = this->getNextTransfer()) {
                transfer.data()->startTransfer();
            }
        }
    }
    else if (newMaximum < oldMaximum) {
        if (newMaximum < this->concurrentTransfers()) {
            int i = this->transferList()->size() - 1;
#ifdef MEEGO_EDITION_HARMATTAN
            while (i >= 0) {
                if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
                    if (transfer.data()->status() > Transfers::Queued) {
                        transfer.data()->setStatus(Transfers::Queued);
                        return;
                    }
                }

                i--;
            }
#else
            int priority = Transfers::LowPriority;

            while (priority >= Transfers::HighPriority) {
                while (i >= 0) {
                    if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
                        if ((transfer.data()->priority() == priority) && (transfer.data()->status() > Transfers::Queued)) {
                            transfer.data()->setStatus(Transfers::Queued);
                            return;
                        }
                    }

                    i--;
                }

                priority--;
                i = this->transferList()->size() - 1;
            }
#endif
        }
    }
}

int TransferManager::concurrentTransfers() const {
    int concurrent = 0;

    for (int i = 0; i < this->transferList()->size(); i++) {
        if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
            if (transfer.data()->status() > Transfers::Queued) {
                concurrent++;
            }
        }
    }

    return concurrent;
}
