#include "database.h"
#include "utils.h"
#include "settings.h"
#include <QFile>
#include <QDir>
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSqlError>
#include <QRegExp>
#include <QDateTime>
#include <QUrl>
#include <QDebug>
#include <QThread>
#include <QDesktopServices>

Database* Database::m_instance = 0;

Database::Database() :
    QObject(),
    m_thread(new QThread),
    m_database(QSqlDatabase::addDatabase("QSQLITE")),
    m_busy(false),
    m_cancelled(false)
{
    if (!m_instance) {
        m_instance = this;
    }

    this->moveToThread(m_thread);
    m_thread->start();
#if (defined (Q_WS_MAEMO_5)) || (defined (Q_WS_MAEMO_4)) || (defined (MEEGO_EDITION_HARMATTAN))
    QDir dir;
    dir.mkpath("/home/user/MusiKloud/");
    m_database.setDatabaseName("/home/user/MusiKloud/MusiKloud.sqlite");
#elif (defined (SYMBIAN_OS))
    m_database.setDatabaseName("MusiKloud.sqlite");
#else
    QDir dir(QDesktopServices::storageLocation(QDesktopServices::HomeLocation) + "/MusiKloud/");
    dir.mkpath(dir.path());
    m_database.setDatabaseName(dir.path() + "/MusiKloud.sqlite");
#endif
    this->initialize();
}

Database::~Database() {
    if (m_thread) {
        m_thread->quit();
        m_thread->deleteLater();
    }
}

Database* Database::instance() {
    return !m_instance ? new Database : m_instance;
}

void Database::initialize() {
    if (m_database.open()) {
        m_database.exec("CREATE TABLE IF NOT EXISTS tracks (filePath TEXT UNIQUE, id TEXT, artist TEXT, title TEXT, description TEXT, genre TEXT, duration INTEGER, coverUrl TEXT, waveformUrl TEXT, playlistId TEXT, trackNumber INTEGER, playCount INTEGER, lastPlayed TEXT, fileSize TEXT, fileExtension TEXT, lastModified TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS playlists (id TEXT UNIQUE, artist TEXT, title TEXT, description TEXT, coverUrl TEXT, trackCount INTEGER, duration INTEGER, lastModified TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS trackDownloads (id TEXT UNIQUE, artist TEXT, title TEXT, description TEXT, genre TEXT, duration INTEGER, format TEXT, streamUrl TEXT, downloadable INTEGER, downloadUrl TEXT, coverUrl TEXT, waveformUrl TEXT, playlistId TEXT, trackNumber INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS playlistDownloads (id TEXT UNIQUE, artist TEXT, title TEXT, description TEXT, coverUrl TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS lastfmAccount (name TEXT UNIQUE, user TEXT, token TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS soundcloudAccount (name TEXT UNIQUE, user TEXT, token TEXT)");
        m_database.exec("ALTER TABLE trackDownloads ADD COLUMN fileSize TEXT");
        m_database.exec("ALTER TABLE playlistDownloads ADD COLUMN trackCount INTEGER");
        m_database.close();
    }
    else {
        qDebug() << m_database.lastError().text();
    }
}

void Database::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            this->setCancelled(false);
            emit busy(message, numberOfOperations);
        }
        else if (!this->cancelled()) {
            emit busyProgressChanged(numberOfOperations);
        }

        emit busyChanged(isBusy);
    }
}

void Database::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    emit currentOperationCancelled();
}

void Database::restoreAccounts() {
    this->restoreSoundCloudAccount();
    this->restoreLastfmAccount();
}

bool Database::storeSoundCloudToken(const QString &token) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("INSERT OR REPLACE INTO soundcloudAccount VALUES (?, ?, ?)");
    query.addBindValue("soundcloud");
    query.addBindValue(QVariant::String);
    query.addBindValue(token);

    bool success = query.exec();

    if (success) {
        emit gotSoundCloudToken(token);
    }
    else {
        emit error(tr("Database error. Unable to store SoundCloud account details"));
    }

    m_database.close();

    return success;
}

bool Database::storeSoundCloudUsername(const QString &user) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("UPDATE soundcloudAccount SET user = ? WHERE name = ?");
    query.addBindValue(user);
    query.addBindValue("soundcloud");

    bool success = query.exec();

    if (success) {
        emit gotSoundCloudUsername(user);
    }
    else {
        emit error(tr("Database error. Unable to store SoundCloud account details"));
    }

    m_database.close();

    return success;
}

bool Database::deleteSoundCloudAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("DELETE FROM soundcloudAccount WHERE name = ?");
    query.addBindValue("soundcloud");

    bool success = query.exec();

    if (success) {
        emit gotSoundCloudUsername(QString());
        emit gotSoundCloudToken(QString());
        emit info(tr("SoundCloud account deleted. Please visit the SoundCloud website to revoke access"));
    }
    else {
        emit error(tr("Database error. Unable to delete SoundCloud account details"));
    }

    m_database.close();

    return success;
}

void Database::restoreSoundCloudAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QString user;
    QString token;
    QSqlQuery query;
    QSqlRecord record;
    query.exec("SELECT user, token FROM soundcloudAccount");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            user = query.value(0).toString();
            token = query.value(1).toString();
        }
    }

    m_database.close();

    emit gotSoundCloudAccount(user, token);
}

bool Database::storeLastfmAccount(const QString &user, const QString &token) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("INSERT OR REPLACE INTO lastfmAccount VALUES (?, ?, ?)");
    query.addBindValue("lastfm");
    query.addBindValue(user);
    query.addBindValue(token);

    bool success = query.exec();

    if (success) {
        emit gotLastfmAccount(user, token);
    }
    else {
        emit error(tr("Database error. Unable to store Lastfm account details"));
    }

    m_database.close();

    return success;
}

bool Database::deleteLastfmAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("DELETE FROM lastfmAccount WHERE name = ?");
    query.addBindValue("lastfm");

    bool success = query.exec();

    if (success) {
        emit gotLastfmAccount(QString(), QString());
        emit info(tr("Last.fm account deleted. Please visit the Last.fm website to revoke access"));
    }
    else {
        emit error(tr("Database error. Unable to delete Last.fm account details"));
    }

    m_database.close();

    return success;
}

void Database::restoreLastfmAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QString user;
    QString token;
    QSqlQuery query;
    QSqlRecord record;
    query.exec("SELECT user, token FROM lastfmAccount");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            user = query.value(0).toString();
            token = query.value(1).toString();
        }
    }

    m_database.close();

    emit gotLastfmAccount(user, token);
}

bool Database::storeDownload(QSharedPointer<TransferItem> transfer) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = this->storeTrackDownload(transfer.data()->track());

    switch (transfer.data()->transferType()) {
    case Transfers::PlaylistDownload:
        success = this->storePlaylistDownload(transfer.data()->playlist());
        break;
    default:
        return success;
    }

    int i = 0;

    while ((success) && (i < transfer.data()->playlistTracks().size())) {
        success = this->storeTrackDownload(transfer.data()->playlistTracks().at(i));
        i++;
    }

    return success;
}

bool Database::storeTrackDownload(QSharedPointer<TrackItem> track) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare("INSERT OR IGNORE INTO trackDownloads VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    query.addBindValue(track.data()->id());
    query.addBindValue(track.data()->artist());
    query.addBindValue(track.data()->title());
    query.addBindValue(track.data()->description());
    query.addBindValue(track.data()->genre());
    query.addBindValue(track.data()->duration());
    query.addBindValue(track.data()->format());
    query.addBindValue(track.data()->streamUrl());
    query.addBindValue(track.data()->downloadable());
    query.addBindValue(track.data()->downloadUrl());
    query.addBindValue(track.data()->thumbnailUrl());
    query.addBindValue(track.data()->waveformUrl());
    query.addBindValue(track.data()->playlistId());
    query.addBindValue(track.data()->trackNumber());
    query.addBindValue(track.data()->size());
    success = query.exec();
    m_database.close();

    return success;
}

bool Database::storePlaylistDownload(QSharedPointer<PlaylistItem> playlist) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare("INSERT OR IGNORE INTO playlistDownloads VALUES (?, ?, ?, ?, ?, ?)");
    query.addBindValue(playlist.data()->id());
    query.addBindValue(playlist.data()->artist());
    query.addBindValue(playlist.data()->title());
    query.addBindValue(playlist.data()->description());
    query.addBindValue(playlist.data()->thumbnailUrl());
    query.addBindValue(playlist.data()->trackCount());
    success = query.exec();
    m_database.close();

    return success;
}

bool Database::removeStoredDownload(QSharedPointer<TransferItem> transfer) {
    switch (transfer.data()->transferType()) {
    case Transfers::PlaylistDownload:
        return this->removeStoredPlaylistDownload(transfer.data()->playlist().data()->id());
    default:
        return this->removeStoredTrackDownload(transfer.data()->track().data()->id());
    }
}

bool Database::removeStoredTrackDownload(const QString &id, const QString &key) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare(QString("DELETE FROM trackDownloads WHERE %1 = ?").arg(key));
    query.addBindValue(id);
    success = query.exec();
    m_database.close();

    return success;
}

bool Database::removeStoredPlaylistDownload(const QString &id) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare("DELETE FROM playlistDownloads WHERE id = ?");
    query.addBindValue(id);
    success = query.exec();
    m_database.close();

    if (success) {
        return this->removeStoredTrackDownload(id, "playlistId");
    }

    return false;
}

QList< QSharedPointer<TransferItem> > Database::getStoredDownloads() {
    QList< QSharedPointer<TransferItem> > transfers;
    QList< QSharedPointer<TrackItem> > tracks = this->getStoredTrackDownloads();
    QList< QSharedPointer<PlaylistItem> > playlists = this->getStoredPlaylistDownloads();

    qDebug() << "Found " + QString::number(tracks.size()) + " track downloads";
    qDebug() << "Found " + QString::number(playlists.size()) + " playlist downloads";

    while (!playlists.isEmpty()) {
        QSharedPointer<PlaylistItem> playlist = playlists.takeFirst();
        QList< QSharedPointer<TrackItem> > playlistTracks;

        int i = 0;

        while (i < tracks.size()) {
            if (tracks.at(i).data()->playlistId() == playlist.data()->id()) {
                playlistTracks.append(tracks.takeAt(i));
            }
            else {
                i++;
            }
        }

        transfers.append(QSharedPointer<TransferItem>(new TransferItem(playlist, playlistTracks)));
    }

    while (!tracks.isEmpty()) {
        transfers.append(QSharedPointer<TransferItem>(new TransferItem(tracks.takeFirst())));
    }

    return transfers;
}

QList< QSharedPointer<TrackItem> > Database::getStoredTrackDownloads() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    QSqlRecord record;
    QList< QSharedPointer<TrackItem> > tracks;
    query.exec("SELECT * FROM trackDownloads ORDER BY trackNumber ASC");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            QSharedPointer<TrackItem> track(new TrackItem);
            track.data()->setId(query.value(0).toString());
            track.data()->setArtist(query.value(1).toString());
            track.data()->setTitle(query.value(2).toString());
            track.data()->setDescription(query.value(3).toString());
            track.data()->setGenre(query.value(4).toString());
            track.data()->setDuration(query.value(5).toInt());
            track.data()->setFormat(query.value(6).toString());
            track.data()->setStreamUrl(query.value(7).toString());
            track.data()->setDownloadable(query.value(8).toBool());
            track.data()->setDownloadUrl(query.value(9).toString());
            track.data()->setThumbnailUrl(query.value(10).toString());
            track.data()->setWaveformUrl(query.value(11).toString());
            track.data()->setPlaylistId(query.value(12).toString());
            track.data()->setTrackNumber(query.value(13).toInt());
            track.data()->setSize(query.value(14).toLongLong());
            tracks.append(track);
        }
    }

    m_database.close();

    return tracks;
}

QList < QSharedPointer<PlaylistItem> > Database::getStoredPlaylistDownloads() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    QSqlRecord record;
    QList < QSharedPointer<PlaylistItem> > playlists;
    query.exec("SELECT * FROM playlistDownloads");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            QSharedPointer<PlaylistItem> playlist(new PlaylistItem);
            playlist.data()->setId(query.value(0).toString());
            playlist.data()->setArtist(query.value(1).toString());
            playlist.data()->setTitle(query.value(2).toString());
            playlist.data()->setDescription(query.value(3).toString());
            playlist.data()->setThumbnailUrl(query.value(4).toString());
            playlist.data()->setTrackCount(query.value(5).toInt());
            playlists.append(playlist);
        }
    }
    m_database.close();

    return playlists;
}

bool Database::addPlaylist(QSharedPointer<PlaylistItem> playlist) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare("INSERT OR IGNORE INTO playlists VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    query.addBindValue(playlist.data()->id());
    query.addBindValue(playlist.data()->artist());
    query.addBindValue(playlist.data()->title());
    query.addBindValue(playlist.data()->description());
    query.addBindValue(playlist.data()->thumbnailUrl());
    query.addBindValue(playlist.data()->trackCount());
    query.addBindValue(playlist.data()->duration());
    query.addBindValue(Utils::currentMSecsSinceEpoch());
    success = query.exec();
    m_database.close();

    if (success) {
        this->removeStoredPlaylistDownload(playlist.data()->id());
        emit playlistsUpdated();
    }

    return success;
}

QSharedPointer<PlaylistItem> Database::getPlaylist(const QString &id) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    QSqlRecord record;
    query.prepare("SELECT * FROM playlists WHERE id = ?");
    query.addBindValue(id);
    query.exec();
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            QSharedPointer<PlaylistItem> playlist = QSharedPointer<PlaylistItem>(new PlaylistItem);
            playlist.data()->setId(query.value(0).toString());
            playlist.data()->setArtist(query.value(1).toString());
            playlist.data()->setTitle(query.value(2).toString());
            playlist.data()->setDescription(query.value(3).toString());
            playlist.data()->setThumbnailUrl(query.value(4).toString());
            playlist.data()->setTrackCount(query.value(5).toInt());
            playlist.data()->setDuration(query.value(6).toInt());
            playlist.data()->setDate(Utils::dateFromMSecs(query.value(7).toLongLong()));
            m_database.close();

            return playlist;
        }
    }

    return QSharedPointer<PlaylistItem>();
}

bool Database::updatePlaylist(const QString &id, const QString &key, const QVariant &value) {    
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare(QString("UPDATE playlists SET %1 = ? WHERE id = ?").arg(key));
        query.addBindValue(value);
        query.addBindValue(id);
        bool success = query.exec();
        m_database.close();

        if (success) {
            emit playlistsUpdated();
        }

        return success;
    }

    return false;
}

bool Database::updatePlaylist(const QString &id, int trackCountChange, int durationChange) {
    QSharedPointer<PlaylistItem> playlist = this->getPlaylist(id);

    if (playlist.isNull()) {
        return false;
    }

    if (!m_database.isOpen()) {
        m_database.open();
    }

    int newTrackCount = playlist.data()->trackCount() + trackCountChange;
    int newDuration = playlist.data()->duration() + durationChange;

    bool success = false;
    QSqlQuery query;
    query.prepare("UPDATE playlists SET trackCount = ?, duration = ?, lastModified = ? WHERE id = ?");
    query.addBindValue(newTrackCount);
    query.addBindValue(newDuration);
    query.addBindValue(Utils::currentMSecsSinceEpoch());
    query.addBindValue(id);
    success = query.exec();
    m_database.close();

    if (success) {
        emit playlistsUpdated();

        if (newTrackCount == 0) {
            this->deletePlaylist(playlist.data()->id());
        }
    }

    return success;
}

bool Database::deletePlaylist(const QString &id, bool deleteTracksFromFS) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare("DELETE FROM playlists WHERE id = ?");
    query.addBindValue(id);
    success = query.exec();
    m_database.close();

    if (success) {

        if (deleteTracksFromFS) {
            QList< QSharedPointer<TrackItem> > tracks = this->getTracks(id);

            while (!tracks.isEmpty()) {
                this->deleteTrack(tracks.takeFirst(), true, false);
            }

            emit tracksUpdated();
        }

        emit alert(tr("Playlist deleted"));
        emit playlistsUpdated();
    }
    else {
        emit error(tr("Playlist could not be deleted"));
    }

    return success;
}

bool Database::addTrack(QSharedPointer<TrackItem> track) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare("INSERT OR IGNORE INTO tracks VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    query.addBindValue(track.data()->url().toLocalFile());
    query.addBindValue(track.data()->id());
    query.addBindValue(track.data()->artist());
    query.addBindValue(track.data()->title());
    query.addBindValue(track.data()->description());
    query.addBindValue(track.data()->genre());
    query.addBindValue(track.data()->duration());
    query.addBindValue(track.data()->thumbnailUrl());
    query.addBindValue(track.data()->waveformUrl());
    query.addBindValue(track.data()->playlistId());
    query.addBindValue(track.data()->trackNumber());
    query.addBindValue(QVariant::Int);
    query.addBindValue(QVariant::Int);
    query.addBindValue(track.data()->size());
    query.addBindValue(track.data()->url().toString().section('.', -1));
    query.addBindValue(Utils::currentMSecsSinceEpoch());
    success = query.exec();
    m_database.close();

    if (success) {
        this->removeStoredTrackDownload(track.data()->id());
        emit tracksUpdated();
    }

    return success;
}

bool Database::updateTrack(const QUrl &url, const QString &key, const QVariant &value) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare(QString("UPDATE tracks SET %1 = ? WHERE filePath = ?").arg(key));
        query.addBindValue(value);
        query.addBindValue(url.toLocalFile());
        bool success = query.exec();
        m_database.close();

        if (success) {
            emit tracksUpdated();
        }

        return success;
    }

    return false;
}

bool Database::deleteTrack(QSharedPointer<TrackItem> track, bool deleteFromFS, bool updatePlaylistTrackCount) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare("DELETE FROM tracks WHERE filePath = ?");
    query.addBindValue(track.data()->url().toLocalFile());
    success = query.exec();
    m_database.close();

    if (success) {
        if (deleteFromFS) {
            QFile::remove(track.data()->url().toLocalFile());
            QFile::remove(track.data()->thumbnailUrl().toLocalFile());
            QFile::remove(track.data()->waveformUrl().toLocalFile());
        }

        if (updatePlaylistTrackCount) {
            this->updatePlaylist(track.data()->playlistId(), -1, -track.data()->duration());
        }
    }

    return success;
}

bool Database::deleteTracks(QList< QSharedPointer<TrackItem> > tracks, bool deleteFromFS, bool updatePlaylistTrackCount) {
    bool success = true;

    while ((success) && (!tracks.isEmpty())) {
        success = this->deleteTrack(tracks.takeFirst(), deleteFromFS, updatePlaylistTrackCount);
    }

    if (success) {
        emit tracksUpdated();
        emit alert(tr("Tracks(s) deleted"));
    }
    else {
        emit error(tr("Some tracks could not be deleted"));
    }

    return success;
}

QList< QSharedPointer<TrackItem> > Database::getTracks(const QString &playlistId) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    QSqlRecord record;
    QList< QSharedPointer<TrackItem> > tracks;

    if (!playlistId.isEmpty()) {
        query.prepare("SELECT * FROM tracks WHERE playlistId = ?");
        query.addBindValue(playlistId);
    }
    else {
        query.prepare("SELECT * FROM tracks ORDER BY lastModified DESC");
    }

    query.exec();
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            QSharedPointer<TrackItem> track = QSharedPointer<TrackItem>(new TrackItem);
            track.data()->setUrl(QUrl::fromLocalFile(query.value(0).toString().remove("file://")));
            track.data()->setStreamUrl(track.data()->url());
            track.data()->setId(query.value(1).toString());
            track.data()->setArtist(query.value(2).toString());
            track.data()->setTitle(query.value(3).toString());
            track.data()->setDescription(query.value(4).toString());
            track.data()->setGenre(query.value(5).toString());
            track.data()->setDuration(query.value(6).toInt());
            track.data()->setThumbnailUrl(query.value(7).toString());
            track.data()->setLargeThumbnailUrl(query.value(7).toString());
            track.data()->setWaveformUrl(query.value(8).toString());
            track.data()->setPlaylistId(query.value(9).toString());
            track.data()->setTrackNumber(query.value(10).toInt());
            track.data()->setPlayCount(query.value(11).toInt());
            track.data()->setLastPlayed(query.value(12).toLongLong());
            track.data()->setSize(query.value(13).toLongLong());
            track.data()->setFormat(query.value(14).toString());
            track.data()->setDate(Utils::dateFromMSecs(query.value(15).toLongLong()));
            track.data()->setBpm(0);
            track.data()->setService(Services::NoService);
            track.data()->setDownloaded(false);
            track.data()->setFavourite(false);
            tracks.append(track);
        }
    }

    m_database.close();

    return tracks;
}

QList< QSharedPointer<PlaylistItem> > Database::getPlaylists() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    QSqlRecord record;
    QList< QSharedPointer<PlaylistItem> > playlists;
    query.prepare("SELECT * FROM playlists ORDER BY lastModified DESC");
    query.exec();
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            QSharedPointer<PlaylistItem> playlist = QSharedPointer<PlaylistItem>(new PlaylistItem);
            playlist.data()->setId(query.value(0).toString());
            playlist.data()->setArtist(query.value(1).toString());
            playlist.data()->setTitle(query.value(2).toString());
            playlist.data()->setDescription(query.value(3).toString());
            playlist.data()->setThumbnailUrl(query.value(4).toString());
            playlist.data()->setTrackCount(query.value(5).toInt());
            playlist.data()->setDuration(query.value(6).toInt());
            playlist.data()->setDate(Utils::dateFromMSecs(query.value(7).toLongLong()));
            playlist.data()->setService(Services::NoService);
            playlists.append(playlist);
        }
    }

    m_database.close();

    return playlists;
}
