#include "archivetrackmodel.h"
#include "tracklistmodel.h"
#include "database.h"
#if (defined (QML_USER_INTERFACE) && (!defined (Q_WS_MAEMO_5)))
#include "settings.h"
#endif

ArchiveTrackModel::ArchiveTrackModel(QObject *parent) :
    QSortFilterProxyModel(parent),
    m_model(new TrackListModel(this)),
    m_loading(true)
{
#if (defined (QML_USER_INTERFACE) && (!defined (Q_WS_MAEMO_5)))
    this->setHighlightColor(Settings::instance()->activeColor());
#endif
    this->setSourceModel(m_model);
    this->setFilterRole(TrackListModel::TitleRole);

    this->connect(m_model, SIGNAL(itemsSelectedChanged(bool)), this, SIGNAL(itemsSelectedChanged(bool)));
    this->connect(Database::instance(), SIGNAL(tracksUpdated()), this, SLOT(reset()));
}

ArchiveTrackModel::~ArchiveTrackModel() {}

void ArchiveTrackModel::clear() {
    m_model->clear();
}

void ArchiveTrackModel::reset() {
    this->clear();
    this->getTracks();
}

QSharedPointer<TrackItem> ArchiveTrackModel::get(int row) const {
    return m_model->get(this->mapToSource(this->index(row, 0)).row());
}

#ifdef QML_USER_INTERFACE
TrackItem* ArchiveTrackModel::getFromQML(int row) const {
    return m_model->getFromQML(this->mapToSource(this->index(row, 0)).row());
}
#endif

void ArchiveTrackModel::getTracks(const QString &playlistId) {
    this->setLoading(true);

    if (!playlistId.isEmpty()) {
        this->setPlaylistId(playlistId);
    }

    QList< QSharedPointer<TrackItem> > tracks = Database::instance()->getTracks(this->playlistId());

    while (!tracks.isEmpty()) {
        m_model->appendTrack(tracks.takeFirst());
    }

    emit countChanged(this->rowCount());
    this->setLoading(false);
}

void ArchiveTrackModel::setPlaylistId(const QString &id) {
    if (id != this->playlistId()) {
        m_playlistId = id;
    }
}

void ArchiveTrackModel::setQuery(const QString &query) {
    if (query != this->query()) {
        m_query = query;
        this->setFilterRegExp(QRegExp(query, Qt::CaseInsensitive, QRegExp::FixedString));
        emit dataChanged(this->index(0, 0), this->index(this->rowCount() - 1, 0));
        emit queryChanged(query);
    }
}

QString ArchiveTrackModel::regexMatch(QString str) const {
    QRegExp re(this->filterRegExp());

    if ((!re.isEmpty()) && (re.indexIn(str) > -1)) {
#if (defined (QML_USER_INTERFACE) && (!defined (Q_WS_MAEMO_5)))
        str.replace(re, "<u><font color=\"" + this->highlightColor() + "\">" + re.cap() + "</font></u>");
#else
        str.replace(re, "<b>" + re.cap() + "</b>");
#endif
    }

    return str;
}

QVariant ArchiveTrackModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case TrackListModel::TitleRole:
        return !this->query().isEmpty() ? this->regexMatch(QSortFilterProxyModel::data(index, role).toString())
                                        : QSortFilterProxyModel::data(index, role);
#ifndef QML_USER_INTERFACE
    case TrackListModel::ThumbnailPressedRole:
        return m_model->data(m_model->index(index.row()), role);
#endif
    default:
        return QSortFilterProxyModel::data(index, role);
    }
}

#if QT_VERSION >= 0x040600
QVariant ArchiveTrackModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}
#endif

void ArchiveTrackModel::toggleSelected(int row) {
    m_model->toggleSelected(this->mapToSource(this->index(row, 0)).row());
}

void ArchiveTrackModel::selectAll() {
    m_model->selectAll();
}

void ArchiveTrackModel::selectNone() {
    m_model->selectNone();
}

bool ArchiveTrackModel::itemsSelected() const {
    return m_model->itemsSelected();
}

QList<int> ArchiveTrackModel::selectedRows() const {
    return m_model->selectedRows();
}

QList< QSharedPointer<TrackItem> > ArchiveTrackModel::selectedItems() const {
    return m_model->selectedItems();
}

QList< QSharedPointer<TrackItem> > ArchiveTrackModel::allItems() const {
    return m_model->allItems();
}

#ifdef QML_USER_INTERFACE
QList<TrackItem*> ArchiveTrackModel::selectedItemsFromQML() const {
    return m_model->selectedItemsFromQML();
}

QList<TrackItem*> ArchiveTrackModel::allItemsFromQML() const {
    return m_model->allItemsFromQML();
}
#endif

void ArchiveTrackModel::queueSelectedTracks() {
    m_model->queueSelectedTracks();
}

void ArchiveTrackModel::deleteSelectedTracks(bool deleteFromFS) {
    QMetaObject::invokeMethod(Database::instance(), "deleteTracks", Q_ARG(QList< QSharedPointer<TrackItem> >, this->selectedItems()), Q_ARG(bool, deleteFromFS), Q_ARG(bool, true));
    this->selectNone();
}

void ArchiveTrackModel::deleteTrack(int row, bool deleteFromFS) {
    QMetaObject::invokeMethod(Database::instance(), "deleteTracks", Q_ARG(QList< QSharedPointer<TrackItem> >, QList< QSharedPointer<TrackItem> >() << this->get(row)), Q_ARG(bool, deleteFromFS), Q_ARG(bool, true));
}
