#ifndef MEDIAPLAYER_H
#define MEDIAPLAYER_H

#include "trackitem.h"
#include <QMediaPlayer>
#include <QSharedPointer>

class MediaPlayer : public QMediaPlayer
{
    Q_OBJECT

    Q_PROPERTY(bool playing
               READ playing
               NOTIFY playingChanged)
    Q_PROPERTY(bool paused
               READ paused
               NOTIFY pausedChanged)
    Q_PROPERTY(PlaybackMode::Mode playbackMode
               READ playbackMode
               WRITE setPlaybackMode
               NOTIFY playbackModeChanged)
    Q_PROPERTY(int currentIndex
               READ currentIndex
               WRITE setCurrentIndex
               NOTIFY currentIndexChanged)
#ifdef QML_USER_INTERFACE
    Q_PROPERTY(TrackItem* currentTrack
               READ currentTrackFromQML
               NOTIFY currentTrackChanged)
#else
    Q_PROPERTY(QSharedPointer<TrackItem> currentTrack
               READ currentTrack
               NOTIFY currentTrackChanged)
#endif

public:
    explicit MediaPlayer(QObject *parent = 0);
    ~MediaPlayer();
#ifdef QML_USER_INTERFACE
    inline TrackItem* currentTrackFromQML() const { return m_track.data(); }
#endif
    inline QSharedPointer<TrackItem> currentTrack() const { return m_track; }
    inline int currentIndex() const { return m_index; }
    inline PlaybackMode::Mode playbackMode() const { return m_playbackMode; }
    bool playing() const;
    bool paused() const;

    static MediaPlayer* instance();
        
public slots:
#ifdef QML_USER_INTERFACE
    void playTrack(TrackItem *track, bool playImmediately = true);
    void playTracks(QList<TrackItem*> tracks, bool playImmediately = true);
#else
    void playTrack(QSharedPointer<TrackItem> track, bool playImmediately = true);
    void playTracks(QList< QSharedPointer<TrackItem> > tracks, bool playImmediately = true);
#endif
    void removeTrack(int row);
    void clearTracks();
    void setPlaybackMode(PlaybackMode::Mode mode);
    void togglePlaybackMode();
    void togglePlayPause();
    void play(bool resetSource = false);
    void next();
    void previous();
    void setCurrentIndex(int index, bool playImmediately = true);

private:
    void setCurrentTrack(QSharedPointer<TrackItem> track);
    void onTrackPlayed(QSharedPointer<TrackItem> track);

private slots:
    void onStateChanged(QMediaPlayer::State state);
    void onMediaStatusChanged(QMediaPlayer::MediaStatus status);
    void onError(QMediaPlayer::Error error);
#ifdef MEDIA_PLAYERS_ENABLED
    void onMediaPlayerChanged(const QString &mediaPlayer);
#endif
    void onCountChanged(int count);

signals:
#ifdef QML_USER_INTERFACE
    void currentTrackChanged(TrackItem *track);
#else
    void currentTrackChanged(QSharedPointer<TrackItem> track);
#endif
    void currentIndexChanged(int index);
    void playbackModeChanged(PlaybackMode::Mode mode);
    void playingChanged(bool playing);
    void pausedChanged(bool paused);

private:
    static MediaPlayer* m_instance;
    QSharedPointer<TrackItem> m_track;
    int m_index;
    PlaybackMode::Mode m_playbackMode;
};

#endif // MEDIAPLAYER_H
