#ifndef DAILYMOTIONGROUPLISTMODEL_H
#define DAILYMOTIONGROUPLISTMODEL_H

#include "groupitem.h"
#include <QAbstractListModel>
#include <QSharedPointer>

#ifndef QML_USER_INTERFACE
class ThumbnailCache;
#endif

class GroupListModel : public QAbstractListModel
{
    Q_OBJECT

    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY loadingChanged)
    Q_PROPERTY(QUrl feed
               READ feed
               WRITE setFeed)

public:
    enum Roles {
        IdRole = Qt::UserRole + 1,
        TitleRole,
        ArtistRole,
        DateRole,
#ifndef QML_USER_INTERFACE
        ThumbnailRole,
#endif
        ThumbnailUrlRole,
        DescriptionRole
    };

public:
    explicit GroupListModel(QObject *parent = 0);
    ~GroupListModel();
    inline QUrl feed() const { return m_feed; }
    void setFeed(const QUrl &feed);
    inline bool loading() const { return m_loading; }
    inline bool moreResults() const { return m_moreResults; }
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant data(const QModelIndex &index, int role) const;
    Q_INVOKABLE QVariant data(int row, const QByteArray &role) const;
    QSharedPointer<GroupItem> get(int row) const;
#ifdef QML_USER_INTERFACE
    Q_INVOKABLE GroupItem* getFromQML(int row) const;
#endif

public slots:
    void clear();
    void reset();
    void getGroups(QUrl feed = QUrl());
    void getMoreGroups();
    void search(const QString &query = QString(), int order = Queries::Hotness);
    void insertGroup(int row, QSharedPointer<GroupItem> group);
    void appendGroup(QSharedPointer<GroupItem> group);
    bool removeGroup(const QVariant &value, int role = IdRole);

private:
    inline QString query() const { return m_query; }
    inline void setQuery(const QString &query) { m_query = query; }
    inline int order() const { return m_order; }
    inline void setOrder(int order) { m_order = order; }
    inline void setLoading(bool loading) { m_loading = loading; emit loadingChanged(loading); }
    inline void setMoreResults(bool more) { m_moreResults = more; }
    inline int offset() const { return m_offset; }
    inline void setOffset(int offset) { m_offset = offset; }
    inline bool error() const { return m_error; }
    inline void setError(bool error) { m_error = error; }
    bool removeRow(int row, const QModelIndex &parent = QModelIndex());

private slots:
    void addGroups();
    inline void onError() { setError(true); setLoading(false); }
    inline void stopLoading() { this->setLoading(false); emit countChanged(this->rowCount()); }
#ifndef QML_USER_INTERFACE
    void onThumbnailReady();
#endif

signals:
    void loadingChanged(bool loading);
    void countChanged(int count);

private:
#ifndef QML_USER_INTERFACE
    ThumbnailCache *m_cache;
#endif
    QUrl m_feed;
    QString m_query;
    int m_order;
    bool m_loading;
    bool m_moreResults;
    bool m_error;
    int m_offset;
    QList< QSharedPointer<GroupItem> > m_list;
};

#endif // DAILYMOTIONGROUPLISTMODEL_H
