/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "controlloader.h"
#include "keyboard.h"
#include "timer.h"
#include "volumekeys.h"
#include "buttonroles.h"
#include "../../base/scriptengine.h"
#include <QDomElement>
#include <QMetaMethod>
#include <QDeclarativeComponent>
#include <QDeclarativeEngine>
#include <QDeclarativeContext>
#include <QDeclarativeItem>
#include <QDeclarativeInfo>
#include <qdeclarative.h>
#include <QDebug>

QObject* ControlLoader::loadControl(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QString name = el.tagName().toLower();

    if (name == "action") {
        return loadAction(el, registerHandlers, parent);
    }

    if (name == "button") {
        return loadButton(el, registerHandlers, parent);
    }

    if (name == "dialog") {
        return loadDialog(el, registerHandlers, parent);
    }

    if (name == "dpad") {
        return loadDpad(el, registerHandlers, parent);
    }

    if (name == "image") {
        return loadImage(el, registerHandlers, parent);
    }

    if (name == "keyboard") {
        return loadKeyboard(el, registerHandlers, parent);
    }

    if (name == "label") {
        return loadLabel(el, registerHandlers, parent);
    }

    if (name == "listselector") {
        return loadListSelector(el, registerHandlers, parent);
    }

    if (name == "listview") {
        return loadListView(el, registerHandlers, parent);
    }

    if (name == "numberpad") {
        return loadNumberpad(el, registerHandlers, parent);
    }

    if (name == "page") {
        return loadPage(el, registerHandlers, parent);
    }

    if (name == "pagestack") {
        return loadPageStack(el, registerHandlers, parent);
    }

    if (name == "sheet") {
        return loadSheet(el, registerHandlers, parent);
    }

    if (name == "slider") {
        return loadSlider(el, registerHandlers, parent);
    }

    if (name == "textfield") {
        return loadTextField(el, registerHandlers, parent);
    }

    if (name == "timer") {
        return loadTimer(el, registerHandlers, parent);
    }

    if (name == "toolbar") {
        return loadToolBar(el, registerHandlers, parent);
    }

    if (name == "toolbutton") {
        return loadToolButton(el, registerHandlers, parent);
    }

    if (name == "touchpad") {
        return loadTouchpad(el, registerHandlers, parent);
    }

    if (name == "volumekeys") {
        return loadVolumeKeys(el, registerHandlers, parent);
    }

    return 0;
}

QDeclarativeItem* ControlLoader::loadPageStack(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeEngine *engine = qmlEngine(parent);

    if (!engine) {
        return 0;
    }

    QDeclarativeComponent *component = new QDeclarativeComponent(engine, QUrl("qrc:/controls/PageStackControl.qml"), parent);
    QObject *obj = component->create(engine->rootContext());

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *stack = qobject_cast<QDeclarativeItem*>(obj);

    if (!stack) {
        return 0;
    }

    stack->setObjectName("pageStack");
    loadProperties(stack, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(stack, el);
    }

    QDomNodeList pages = el.elementsByTagName("page");

    for (int i = 0; i < pages.size(); i++) {
        if (QDeclarativeItem *page = loadPage(pages.at(i).toElement(), registerHandlers, stack)) {
            QMetaObject::invokeMethod(stack, "addPage", Q_ARG(QVariant, QVariant::fromValue(page)));
        }
    }

    return stack;
}

QDeclarativeItem* ControlLoader::loadPage(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/PageControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *page = qobject_cast<QDeclarativeItem*>(obj);

    if (!page) {
        return 0;
    }

    page->setObjectName(el.attribute("name"));
    loadProperties(page, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(page, el);
    }

    QDomNodeList controls = el.childNodes();

    for (int i = 0; i < controls.size(); i++) {
        if (QObject* control = loadControl(controls.at(i).toElement(), registerHandlers, page)) {
            if (QDeclarativeItem *item = qobject_cast<QDeclarativeItem*>(control)) {
                QMetaObject::invokeMethod(page, "addItem", Q_ARG(QVariant, QVariant::fromValue(item)));
            }
        }
    }

    return page;
}

QDeclarativeItem* ControlLoader::loadDialog(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/DialogControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *dialog = qobject_cast<QDeclarativeItem*>(obj);

    if (!dialog) {
        return 0;
    }

    dialog->setObjectName(el.attribute("name"));
    loadProperties(dialog, el);
    loadItemLayout(dialog, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(dialog, el);
    }

    QDomNodeList controls = el.childNodes();

    for (int i = 0; i < controls.size(); i++) {
        if (controls.at(i).nodeName() == "buttons") {
            QDomNodeList buttons = controls.at(i).childNodes();

            for (int ii = 0; ii < buttons.size(); ii++) {
                QMetaObject::invokeMethod(dialog, "addButton", Q_ARG(QDeclarativeItem*, loadButton(buttons.at(ii).toElement(), registerHandlers, dialog)));
            }
        }
        else if (QObject* control = loadControl(controls.at(i).toElement(), registerHandlers, dialog)) {
            if (QDeclarativeItem *item = qobject_cast<QDeclarativeItem*>(control)) {
                QMetaObject::invokeMethod(dialog, "addItem", Q_ARG(QDeclarativeItem*, item));
            }
        }
    }

    return dialog;
}

Keyboard* ControlLoader::loadKeyboard(const QDomElement &el, bool registerHandlers, QObject *parent) {
    Keyboard *keyboard = new Keyboard(parent);
    keyboard->setObjectName(el.attribute("name"));

    if (parent) {
        parent->installEventFilter(keyboard);
    }

    if (registerHandlers) {
        registerHandlersWithScriptEngine(keyboard, el);
    }

    return keyboard;
}

VolumeKeys* ControlLoader::loadVolumeKeys(const QDomElement &el, bool registerHandlers, QObject *parent) {
    VolumeKeys *keys = new VolumeKeys(parent);
    keys->setObjectName(el.attribute("name"));

    if (parent) {
        parent->installEventFilter(keys);
    }

    if (registerHandlers) {
        registerHandlersWithScriptEngine(keys, el);
    }

    return keys;
}

QDeclarativeItem* ControlLoader::loadAction(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/ActionControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *action = qobject_cast<QDeclarativeItem*>(obj);

    if (!action) {
        return 0;
    }

    action->setObjectName(el.attribute("name"));
    loadProperties(action, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(action, el);
    }

    return action;
}

QDeclarativeItem* ControlLoader::loadButton(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/ButtonControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *button = qobject_cast<QDeclarativeItem*>(obj);

    if (!button) {
        return 0;
    }

    button->setObjectName(el.attribute("name"));
    loadProperties(button, el);
    loadItemLayout(button, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(button, el);
    }

    return button;
}

QDeclarativeItem* ControlLoader::loadDpad(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/DpadControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *dpad = qobject_cast<QDeclarativeItem*>(obj);

    if (!dpad) {
        return 0;
    }

    dpad->setObjectName(el.attribute("name"));
    loadItemLayout(dpad, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(dpad, el);
    }

    QHash<QString, Dpad::ButtonRole> roles;
    roles["left"] = Dpad::Left;
    roles["right"] = Dpad::Right;
    roles["up"] = Dpad::Up;
    roles["down"] = Dpad::Down;
    roles["enter"] = Dpad::Enter;

    QDomNodeList buttons = el.childNodes();

    for (int i = 0; i < buttons.size(); i++) {
        QDomElement key = buttons.at(i).toElement();
        QString role = key.tagName().toLower();

        if (roles.contains(role)) {
            if (QDeclarativeItem *button = loadButton(key, registerHandlers, dpad)) {
                QMetaObject::invokeMethod(dpad, "addButton", Q_ARG(QDeclarativeItem*, button), Q_ARG(Dpad::ButtonRole, roles.value(role)));
            }
        }
    }

    return dpad;
}

QDeclarativeItem* ControlLoader::loadImage(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/ImageControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *image = qobject_cast<QDeclarativeItem*>(obj);

    if (!image) {
        return 0;
    }

    image->setObjectName(el.attribute("name"));
    loadProperties(image, el);
    loadItemLayout(image, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(image, el);
    }

    return image;
}

QDeclarativeItem* ControlLoader::loadLabel(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/LabelControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *label = qobject_cast<QDeclarativeItem*>(obj);

    if (!label) {
        return 0;
    }

    label->setObjectName(el.attribute("name"));
    loadProperties(label, el);
    loadItemLayout(label, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(label, el);
    }

    return label;
}

QDeclarativeItem* ControlLoader::loadListSelector(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/ListSelectorControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *selector = qobject_cast<QDeclarativeItem*>(obj);

    if (!selector) {
        return 0;
    }

    selector->setObjectName(el.attribute("name"));
    loadProperties(selector, el);
    loadItemLayout(selector, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(selector, el);
    }

    QDomNodeList nodes = el.elementsByTagName("element");

    for (int i = 0; i < nodes.size(); i++) {
        QDomElement element = nodes.at(i).toElement();
        QMetaObject::invokeMethod(selector, "addItem", Q_ARG(QString, element.attribute("name")), Q_ARG(QVariant, element.attribute("value")));
    }

    return selector;
}

QDeclarativeItem* ControlLoader::loadListView(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/ListViewControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *view = qobject_cast<QDeclarativeItem*>(obj);

    if (!view) {
        return 0;
    }

    view->setObjectName(el.attribute("name"));
    loadProperties(view, el);
    loadItemLayout(view, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(view, el);
    }

    QDomNodeList nodes = el.elementsByTagName("element");

    for (int i = 0; i < nodes.size(); i++) {
        QDomElement element = nodes.at(i).toElement();
        QMetaObject::invokeMethod(view, "addItem", Q_ARG(QString, element.attribute("name")), Q_ARG(QVariant, element.attribute("value")));
    }

    return view;
}

QDeclarativeItem* ControlLoader::loadNumberpad(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/NumberpadControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *pad = qobject_cast<QDeclarativeItem*>(obj);

    if (!pad) {
        return 0;
    }

    pad->setObjectName(el.attribute("name"));
    loadItemLayout(pad, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(pad, el);
    }

    QHash<QString, Numberpad::ButtonRole> roles;
    roles["one"] = Numberpad::One;
    roles["two"] = Numberpad::Two;
    roles["three"] = Numberpad::Three;
    roles["four"] = Numberpad::Four;
    roles["five"] = Numberpad::Five;
    roles["six"] = Numberpad::Six;
    roles["seven"] = Numberpad::Seven;
    roles["eight"] = Numberpad::Eight;
    roles["nine"] = Numberpad::Nine;
    roles["zero"] = Numberpad::Zero;
    roles["back"] = Numberpad::Back;
    roles["forward"] = Numberpad::Forward;

    QDomNodeList buttons = el.childNodes();

    for (int i = 0; i < buttons.size(); i++) {
        QDomElement key = buttons.at(i).toElement();
        QString role = key.tagName().toLower();

        if (roles.contains(role)) {
            if (QDeclarativeItem *button = loadButton(key, registerHandlers, pad)) {
                QMetaObject::invokeMethod(pad, "addButton", Q_ARG(QDeclarativeItem*, button), Q_ARG(Numberpad::ButtonRole, roles.value(role)));
            }
        }
    }

    return pad;
}

QDeclarativeItem* ControlLoader::loadSheet(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/SheetControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *sheet = qobject_cast<QDeclarativeItem*>(obj);

    if (!sheet) {
        return 0;
    }

    sheet->setObjectName(el.attribute("name"));
    loadProperties(sheet, el);
    loadItemLayout(sheet, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(sheet, el);
    }

    QDomNodeList controls = el.childNodes();

    for (int i = 0; i < controls.size(); i++) {
        if (controls.at(i).nodeName() == "buttons") {
            QDomNodeList buttons = controls.at(i).childNodes();

            for (int ii = 0; ii < buttons.size(); ii++) {
                QMetaObject::invokeMethod(sheet, "addButton", Q_ARG(QDeclarativeItem*, loadButton(buttons.at(ii).toElement(), registerHandlers, sheet)));
            }
        }
        else if (QObject* control = loadControl(controls.at(i).toElement(), registerHandlers, sheet)) {
            if (QDeclarativeItem *item = qobject_cast<QDeclarativeItem*>(control)) {
                QMetaObject::invokeMethod(sheet, "addItem", Q_ARG(QDeclarativeItem*, item));
            }
        }
    }

    return sheet;
}

QDeclarativeItem* ControlLoader::loadSlider(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/SliderControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *slider = qobject_cast<QDeclarativeItem*>(obj);

    if (!slider) {
        return 0;
    }

    slider->setObjectName(el.attribute("name"));
    loadProperties(slider, el);
    loadItemLayout(slider, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(slider, el);
    }

    return slider;
}

QDeclarativeItem* ControlLoader::loadTextField(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/TextFieldControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *field = qobject_cast<QDeclarativeItem*>(obj);

    if (!field) {
        return 0;
    }

    field->setObjectName(el.attribute("name"));
    loadProperties(field, el);
    loadItemLayout(field, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(field, el);
    }

    return field;
}

Timer* ControlLoader::loadTimer(const QDomElement &el, bool registerHandlers, QObject *parent) {
    Timer *timer = new Timer(parent);
    timer->setObjectName(el.attribute("name"));
    loadProperties(timer, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(timer, el);
    }

    return timer;
}

QDeclarativeItem* ControlLoader::loadToolBar(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/ToolBarControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *toolbar = qobject_cast<QDeclarativeItem*>(obj);

    if (!toolbar) {
        return 0;
    }

    toolbar->setObjectName(el.attribute("name"));
    loadItemLayout(toolbar, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(toolbar, el);
    }

    QDomNodeList controls = el.childNodes();

    for (int i = 0; i < controls.size(); i++) {
        loadControl(controls.at(i).toElement(), registerHandlers, toolbar);
    }

    return toolbar;
}

QDeclarativeItem* ControlLoader::loadToolButton(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/ToolButtonControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *button = qobject_cast<QDeclarativeItem*>(obj);

    if (!button) {
        return 0;
    }

    button->setObjectName(el.attribute("name"));
    loadProperties(button, el);
    loadItemLayout(button, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(button, el);
    }

    return button;
}

QDeclarativeItem* ControlLoader::loadTouchpad(const QDomElement &el, bool registerHandlers, QObject *parent) {
    QDeclarativeComponent *component = new QDeclarativeComponent(qmlEngine(parent), QUrl("qrc:/controls/TouchpadControl.qml"), parent);
    QObject *obj = component->create(QDeclarativeEngine::contextForObject(parent));

    if (!obj) {
        return 0;
    }

    QDeclarativeItem *pad = qobject_cast<QDeclarativeItem*>(obj);

    if (!pad) {
        return 0;
    }

    pad->setObjectName(el.attribute("name"));
    loadItemLayout(pad, el);

    if (registerHandlers) {
        registerHandlersWithScriptEngine(pad, el);
    }

    return pad;
}

void ControlLoader::loadProperties(QObject *obj, const QDomElement &el) {
    QDomNodeList properties = el.childNodes();

    for (int i = 0; i < properties.size(); i++) {
        QDomElement property = properties.at(i).toElement();
        obj->setProperty(property.tagName().toUtf8(), property.text());
    }
}

void ControlLoader::loadItemLayout(QDeclarativeItem *item, const QDomElement &el) {
    QDomElement widthEl = el.firstChildElement("width");

    if (!widthEl.firstChild().isNull()) {
        item->setProperty("widthPortrait", qMax(0, widthEl.firstChildElement("portrait").text().toInt()));
        item->setProperty("widthLandscape", qMax(0, widthEl.firstChildElement("landscape").text().toInt()));
    }

    QDomElement heightEl = el.firstChildElement("height");

    if (!heightEl.firstChild().isNull()) {
        item->setProperty("heightPortrait", qMax(0, heightEl.firstChildElement("portrait").text().toInt()));
        item->setProperty("heightLandscape", qMax(0, heightEl.firstChildElement("landscape").text().toInt()));
    }

    QDomElement rowEl = el.firstChildElement("row");

    if (rowEl.firstChild().isText()) {
        item->setProperty("rowPortrait", qMax(0, rowEl.text().toInt()));
        item->setProperty("rowLandscape", qMax(0, rowEl.text().toInt()));
    }
    else {
        item->setProperty("rowPortrait", qMax(0, rowEl.firstChildElement("portrait").text().toInt()));
        item->setProperty("rowLandscape", qMax(0, rowEl.firstChildElement("landscape").text().toInt()));
    }

    QDomElement colEl = el.firstChildElement("column");

    if (colEl.firstChild().isText()) {
        item->setProperty("columnPortrait", qMax(0, colEl.text().toInt()));
        item->setProperty("columnLandscape", qMax(0, colEl.text().toInt()));
    }
    else {
        item->setProperty("columnPortrait", qMax(0, colEl.firstChildElement("portrait").text().toInt()));
        item->setProperty("columnLandscape", qMax(0, colEl.firstChildElement("landscape").text().toInt()));
    }

    QDomElement rowSpanEl = el.firstChildElement("rowSpan");

    if (rowSpanEl.firstChild().isText()) {
        item->setProperty("rowSpanPortrait", qMax(1, rowSpanEl.text().toInt()));
        item->setProperty("rowSpanLandscape", qMax(1, rowSpanEl.text().toInt()));
    }
    else {
        item->setProperty("rowSpanPortrait", qMax(1, rowSpanEl.firstChildElement("portrait").text().toInt()));
        item->setProperty("rowSpanLandscape", qMax(1, rowSpanEl.firstChildElement("landscape").text().toInt()));
    }

    QDomElement colSpanEl = el.firstChildElement("colSpan");

    if (colSpanEl.firstChild().isText()) {
        item->setProperty("colSpanPortrait", qMax(1, colSpanEl.text().toInt()));
        item->setProperty("colSpanLandscape", qMax(1, colSpanEl.text().toInt()));
    }
    else {
        item->setProperty("colSpanPortrait", qMax(1, colSpanEl.firstChildElement("portrait").text().toInt()));
        item->setProperty("colSpanLandscape", qMax(1, colSpanEl.firstChildElement("landscape").text().toInt()));
    }

    QDomElement alignEl = el.firstChildElement("alignment");

    if (alignEl.firstChild().isText()) {
        item->setProperty("alignmentPortrait", alignEl.text());
        item->setProperty("alignmentLandscape", alignEl.text());
    }
    else if (!alignEl.firstChild().isNull()) {
        item->setProperty("alignmentPortrait", alignEl.firstChildElement("portrait").text());
        item->setProperty("alignmentLandscape", alignEl.firstChildElement("landscape").text());
    }
}

void ControlLoader::registerHandlersWithScriptEngine(QObject *obj, const QDomElement &el) {
    ScriptEngine::instance()->globalObject().setProperty(obj->objectName(), ScriptEngine::instance()->newQObject(obj));

    QDomNodeList nodes = el.childNodes();

    for (int i = 0; i < nodes.size(); i++) {
        QDomElement elem = nodes.at(i).toElement();
        QString tagName = elem.tagName();

        if ((tagName.startsWith("on")) && (tagName.size() > 3)) {
            QString handler = elem.text();

            if (!handler.isEmpty()) {
                QString signal = tagName.at(2).toLower() + tagName.mid(3);

                for (int i = 0; i < obj->metaObject()->methodCount(); i++) {
                    QMetaMethod method = obj->metaObject()->method(i);
                    QString signature = method.signature();

                    if (signature.left(signature.indexOf('(')) == signal) {
                        QStringList params;

                        foreach (QByteArray param, method.parameterNames()) {
                            params << param;
                        }

                        QScriptValue val = ScriptEngine::instance()->evaluate(QString("%1.%2.connect( %3 );").arg(obj->objectName()).arg(signal)
                                                                       .arg(handler.startsWith("function") ? handler : QString("function (%1) { %2 }").arg(params.join(",")).arg(handler)));

                        if (val.isError()) {
                            qWarning() << QObject::tr("Cannnot connect to signal %1: %2").arg(signal).arg(val.toString());
                        }

                        break;
                    }
                }
            }
        }
    }
}
