#include "profilebrowserpanel.h"
#include "ui_profilebrowserpanel.h"

#include <QTimer>
#include <QToolBar>
#include <QActionGroup>
#include <QToolButton>

#include "filter/servicefilter.h"
#include "filter/filtermanager.h"

#include "config.h"

ProfileBrowserPanel::ProfileBrowserPanel(QFont usedFont, QWidget *parent) :
        QWidget(parent),
        ui(new Ui::ProfileBrowserPanel),
        isPortraitMode(false),
        state(WindowFriendList)
{
    this->usedFont = usedFont;
    this->sm = new ServiceMgr(this);
    ui->setupUi(this);

    init();
}

ProfileBrowserPanel::ProfileBrowserPanel(QFont usedFont, ServiceMgr *mgr, QWidget *parent) :
    QWidget(parent),
    ui(new Ui::ProfileBrowserPanel),
    isPortraitMode(false),
    state(WindowFriendList)
{
    this->usedFont = usedFont;

    if (mgr != NULL)
        this->sm = mgr;
    else
        this->sm = new ServiceMgr(this);
    ui->setupUi(this);

    init();
}

ProfileBrowserPanel::~ProfileBrowserPanel()
{
    delete ui;
}

void ProfileBrowserPanel::init()
{
    isFriendsUpdate = false;

    pb = new ProfileBrowser(this->sm, true, this);
    ui->layoutProfile->addWidget(pb, 0);

    // Title labels
    QLabel* friendLabel = new QLabel();
    friendLabel->setPixmap(QIcon(":/res/general_contacts.png").pixmap(ICON_SIZE));
    ui->layoutFriendsTop->addWidget(friendLabel);
    friendLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

    friendsCountLabel = new QLabel(tr("People (%1)").arg(0));
    ui->layoutFriendsTop->addWidget(friendsCountLabel);
    //ui->layoutFriendsTop->addStretch();

    // Buttons
    buttonBack = new QToolButton(this);
    buttonBack->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonBack->setFixedHeight(ICON_SIZE);
    buttonBack->setIcon(QIcon::fromTheme("general_back",
                                         QIcon(":/res/general_back.png")));
    ui->layoutProfileTop->addWidget(buttonBack, 1);
    buttonBack->setVisible(FALSE);

#ifndef Q_WS_MAEMO_5
    buttonAlbums = new BTN_TYPE(tr("Albums"));
    buttonAlbums->setFlat(true);
#else
    buttonAlbums = new BTN_TYPE(this);
    buttonAlbums->setFixedHeight(ICON_SIZE);
#endif
    buttonAlbums->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonAlbums->setIcon(QIcon(":/res/tasklaunch_photos.png"));
    buttonAlbums->setEnabled(FALSE);
    ui->layoutProfileTop->addWidget(buttonAlbums, 1);

#ifndef Q_WS_MAEMO_5
    buttonMessages = new BTN_TYPE(tr("Messages"));
    buttonMessages->setFlat(true);
#else
    buttonMessages = new BTN_TYPE(this);
    buttonMessages->setFixedHeight(ICON_SIZE);
#endif
    buttonMessages->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonMessages->setIcon(QIcon(":/res/messages.png"));
    buttonMessages->setEnabled(FALSE);
    ui->layoutProfileTop->addWidget(buttonMessages, 1);

#ifndef Q_WS_MAEMO_5
    buttonSendMsg = new BTN_TYPE(tr("Send message"));
    buttonSendMsg->setFlat(true);
#else
    buttonSendMsg = new BTN_TYPE(this);
    buttonSendMsg->setFixedHeight(ICON_SIZE);
#endif
    buttonSendMsg->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonSendMsg->setIcon(QIcon(":/res/message.png"));
    buttonSendMsg->setEnabled(FALSE);
    ui->layoutProfileTop->addWidget(buttonSendMsg, 1);

    ui->layoutProfileTop->addWidget(buttonRefreshProfile = new QToolButton(this), Qt::AlignRight);
    buttonRefreshProfile->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonRefreshProfile->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    buttonRefreshProfile->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    // friends list
    ui->layoutFriendsTop->addWidget(buttonRefreshFriends = new QToolButton(this), Qt::AlignRight);
    buttonRefreshFriends->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    buttonRefreshFriends->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    this->friendsPanel = new QFriendListWidget(this->usedFont, sm, this);
    this->ui->layoutFriends->addWidget(this->friendsPanel);

    // Filters
    QList<QActionGroup*> actionsList = sm->filterManager()->createActions<Friend>();
    foreach (QActionGroup* actions, actionsList) {
#if !defined(Q_WS_MAEMO_5) && !defined(MEEGO_EDITION_HARMATTAN)
        friendsPanel->filtersToolBar()->addSeparator();
#endif
        friendsPanel->filtersToolBar()->addActions(actions->actions());
    }

    connect(buttonBack, SIGNAL(clicked()), this, SLOT(hideProfile()));
    connect(buttonAlbums, SIGNAL(clicked()), this, SLOT(showFriendAlbums()));
    connect(buttonMessages, SIGNAL(clicked()), this, SLOT(showFriendMessages()));
    connect(buttonSendMsg, SIGNAL(clicked()), this, SLOT(sendMessageToFriend()));
    connect(buttonRefreshFriends, SIGNAL(clicked()), this,
            SLOT(refreshFriends()));
    connect(buttonRefreshProfile, SIGNAL(clicked()), this, SLOT(refreshProfile()));
    connect(this->friendsPanel, SIGNAL(ownerSelected(Friend)),
            this, SLOT(ownrSelected(Friend)));

    // device rotation
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

    this->isFriendsUpdate = false;

// update margins
#ifdef Q_WS_MAEMO_5
    buttonRefreshProfile->setFixedHeight(ICON_SIZE);
    buttonRefreshFriends->setFixedHeight(ICON_SIZE);
    this->setContentsMargins(0,0,0,0);
    this->ui->layoutFriendsTop->setSpacing(0);
    this->ui->layoutFriendsTop->setMargin(0);
    this->ui->layoutProfileTop->setSpacing(0);
    this->ui->layoutProfileTop->setMargin(0);
    this->ui->horizontalLayout->setMargin(0);
    this->ui->layoutFriends->setMargin(0);
    this->ui->layoutFilters->setSpacing(0);
#endif

    // connect to signals from service manager
    connect(sm, SIGNAL(updateFriends(FriendList, bool)), this,
            SLOT(updateListFriends(FriendList, bool)));
    connect(sm, SIGNAL(updateProfile(Friend)), this, SLOT(updateProfile(Friend)));

    // check that one acount mast be setup
    if (this->sm->getAccounts().length() != 0)
        this->updateListFriends(sm->getFriends(false, false), true);

    this->setState();
}

void ProfileBrowserPanel::ownrSelected(Friend fd, bool isNeedUpdateState)
{
    qDebug() << __FILE__ << ":" << __LINE__ << ":" << __FUNCTION__ << ": ownerId=" << fd.ownerId();

    if (!fd.isEmpty()) {
        updateProfile(fd);
        emit openProfile(fd);
        this->sm->getProfile(fd.accountId(), fd.ownerId(), false, true, !fd.hasFullProfile());
        if (this->sm->canSendMessage(fd.accountId()))
            this->buttonSendMsg->setEnabled(true);
        else
            this->buttonSendMsg->setEnabled(false);
    } else {
        // load my profile
       Friend mFd = this->sm->getMyProfile(false);
       this->updateProfile(mFd);
       emit openProfile(mFd);
       this->buttonSendMsg->setEnabled(false);
    }

    if (isNeedUpdateState)
        this->setState(WindowProfile);
}

void ProfileBrowserPanel::updateProfile(Friend profile)
{
    Friend curFr = this->friendsPanel->getSelectedFriend();

    if (curFr.isEmpty())
        curFr = this->sm->getMyProfile(false);

    if (profile.ownerId() == curFr.ownerId()) {
        pb->setProfile(profile);
        this->buttonAlbums->setEnabled(TRUE);
        this->buttonMessages->setEnabled(TRUE);
        this->buttonSendMsg->setEnabled(TRUE);
    }
}

void ProfileBrowserPanel::ownrSelected(Friend item)
{
    this->ownrSelected(item, true);
}

void ProfileBrowserPanel::updateListFriends(FriendList list, bool isLastUpdate)
{
    Q_UNUSED(list);

    if (isLastUpdate) {
        this->isFriendsUpdate = false;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png")));
    } else {
        this->isFriendsUpdate = true;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
    }

    this->friendsCountLabel->setText(QString(tr("People (%1)").arg(list.length())));


    Friend curFr = this->friendsPanel->getSelectedFriend();

    if (curFr.isEmpty())
    {
        curFr = sm->getMyProfile(false);
    } else {
        if (curFr.hasFullProfile())
            curFr = this->sm->getProfile(curFr.accountId(), curFr.ownerId(), false, false, true);
    }
    this->updateProfile(curFr);
}

void ProfileBrowserPanel::refreshFriends()
{
    this->refreshFriends(true);
}

void ProfileBrowserPanel::refreshFriends(bool isNeedUpdate)
{
    if (!this->isFriendsUpdate) {
        this->isFriendsUpdate = true;
        buttonRefreshFriends->setIcon(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")));
        sm->getFriends(isNeedUpdate, true);
    } else {
        // TODO: add stopping friends update
    }

}

void ProfileBrowserPanel::refreshProfile()
{
    Friend fr = this->friendsPanel->getSelectedFriend();
    if (fr.isEmpty())
        this->sm->getMyProfile(true);
    else
        this->sm->getProfile(fr.accountId(), fr.ownerId(), true, true, true);
}

void ProfileBrowserPanel::setState()
{
    this->setState(this->state);
}

void ProfileBrowserPanel::setState(WindowState state) {
    this->state = state;
    bool isMultiWindow = this->sm->settings->value(SETTINGS_MULTIPANEL).toBool() && !this->isPortraitMode;

    switch(this->state) {
    case WindowFriendList:
        setFriendsVisible(true);

        if (isMultiWindow) {
            showAll(ui->layoutProfiles);
            buttonBack->setVisible(FALSE);
        } else {
            hideAll(ui->layoutProfiles);
        }
        break;
    case WindowProfile:
        showAll(ui->layoutProfiles);

        if (isMultiWindow) {
            setFriendsVisible(true);
            buttonBack->setVisible(FALSE);
        } else {
            setFriendsVisible(false);
            buttonBack->setVisible(TRUE);
        }
        break;
    default:
        QMessageBox::warning(this, tr("Error occured"), tr("Unknown window state"));
        break;
    }
}

void ProfileBrowserPanel::hideAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            hideAll(layout->itemAt(i)->layout());
        } else if (layout->itemAt(i)->widget()) {
            layout->itemAt(i)->widget()->setVisible(FALSE);
        }
    }
}
void ProfileBrowserPanel::showAll(QLayout *layout)
{
    for (int i = 0; i < layout->count(); i++) {
        if(layout->itemAt(i)->layout()) {
            showAll(layout->itemAt(i)->layout());
        } else if (layout->itemAt(i)->widget()) {
            layout->itemAt(i)->widget()->setVisible(TRUE);
        }
    }
}

void ProfileBrowserPanel::hideProfile()
{
    this->setState(WindowFriendList);
}

void ProfileBrowserPanel::showFriendInfo(Friend fd)
{
    this->friendsPanel->selectFriend(fd);
    ownrSelected(fd);
}

void ProfileBrowserPanel::showFriendAlbums()
{
    emit showFriendAlbums(this->friendsPanel->getSelectedFriend());
}

void ProfileBrowserPanel::showFriendMessages()
{
    emit showFriendMessages(this->friendsPanel->getSelectedFriend());
}

void ProfileBrowserPanel::sendMessageToFriend()
{
    Friend fd = this->friendsPanel->getSelectedFriend();
    emit sendMessage(fd);
}

void ProfileBrowserPanel::orientationChanged()
{
    QRect geom = QApplication::desktop()->screenGeometry();
    if (geom.width() > geom.height())
        this->isPortraitMode = false;
    else
        this->isPortraitMode = true;
}

void ProfileBrowserPanel::setFriendsVisible(const bool isShow)
{
    if (isShow) {
        this->showAll(this->ui->layoutFriends);
    } else {
        this->hideAll(this->ui->layoutFriends);
        this->ui->layoutFriends->invalidate();
    }
}
