#include "newsbrowserpanel.h"

#include <cassert>

#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QListView>
#include <QtCore/QTimer>
#include <QtGui/QPushButton>
#include <QtCore/QDebug>
#include <QtGui/QToolBar>
#include <QtGui/QLabel>
#include <QtGui/QActionGroup>

#include "config.h"
#include "servicemgr.h"
#include "widgets/feed/feedmodel.h"
#include "widgets/feed/feeddelegate.h"
#include "widgets/feed/feedview.h"

NewsBrowserPanel::NewsBrowserPanel(ServiceMgr* mgr, QWidget* parent) :
    QWidget(parent), mServiceMgr(mgr),
    buttonIconRefresh(QIcon::fromTheme("general_refresh", QIcon(":/res/general_refresh.png"))),
    buttonIconStop(QIcon::fromTheme("general_stop", QIcon(":/res/general_stop.png")))
{
    createLayouts();
    createListView();
    createFilterPanel();
    initializeModel();
    createButtons();

    reloadFeeds(false);
}

NewsBrowserPanel::~NewsBrowserPanel()
{

}

void NewsBrowserPanel::createLayouts()
{
    mMainLayout = new QVBoxLayout;
    setLayout(mMainLayout);

    mTopLayout = new QHBoxLayout;
    mMainLayout->addLayout(mTopLayout);

    mBottomLayout = new QHBoxLayout;
    mMainLayout->addLayout(mBottomLayout);

    mFiltersLayout = new QVBoxLayout;
    mBottomLayout->addLayout(mFiltersLayout);

    mNewsLayout = new QVBoxLayout;
    mBottomLayout->addLayout(mNewsLayout);
}

void NewsBrowserPanel::createListView()
{
    FeedView* fv = new FeedView;
    fv->setVerticalScrollMode(QListView::ScrollPerPixel);
    fv->setResizeMode(QListView::Adjust);
    mNewsLayout->addWidget(fv);
    mNewsList = fv;

    connect(fv, SIGNAL(imageClicked(Photo)), SIGNAL(openImage(Photo)));
}

void NewsBrowserPanel::createButtons()
{
    // Reload button
    mReloadButton = new QPushButton(this);
    mReloadButton->setIcon(buttonIconRefresh);
    mReloadButton->setFlat(true);
    mReloadButton->setIconSize(QSize(ICON_SIZE, ICON_SIZE));


    // Label
    QLabel* iconLabel = new QLabel();
    iconLabel->setPixmap(QIcon(":/res/general_rss.png").pixmap(ICON_SIZE));


    mCountLabel = new QLabel();
    setNewsCount(0);

    mTopLayout->addWidget(iconLabel);
    mTopLayout->addWidget(mCountLabel);
    mTopLayout->addStretch();
    mTopLayout->addWidget(mReloadButton);

    connect(mServiceMgr, SIGNAL(updateFeed(QEventFeedList,QEventFeed::FeedType,bool)),
            this, SLOT(updateButtonIcon(QEventFeedList,QEventFeed::FeedType,bool)));
    connect(mReloadButton, SIGNAL(clicked()),
            this, SLOT(reloadFeeds()));
}

void NewsBrowserPanel::initializeModel()
{
    mModel = new FeedModel(mServiceMgr, this);
    mNewsList->setModel(mModel);

    FeedDelegate* delegate = new FeedDelegate(mServiceMgr, mNewsList);
    mNewsList->setItemDelegate(delegate);

    connect(mServiceMgr, SIGNAL(updateFeed(QEventFeedList,QEventFeed::FeedType,bool)),
            mModel, SLOT(setFeed(QEventFeedList,QEventFeed::FeedType,bool)));
    connect(mModel, SIGNAL(newFeedType(QEventFeed::FeedType)),
             this, SLOT(addFeedType(QEventFeed::FeedType)));
    connect(mModel, SIGNAL(feedCountChanged(int)),
            this, SLOT(setNewsCount(int)));
}

void NewsBrowserPanel::reloadFeeds(bool update)
{
    mReloadButton->setIcon(buttonIconStop);
    mServiceMgr->getFeed(QEventFeed::messageFeed, update, true);
    mServiceMgr->getFeed(QEventFeed::photoFeed, update, true);
}

void NewsBrowserPanel::updateButtonIcon(QEventFeedList, QEventFeed::FeedType, bool isLastUpdate)
{
    if (isLastUpdate)
        mReloadButton->setIcon(buttonIconRefresh);
}

void NewsBrowserPanel::createFilterPanel()
{
    mFiltersToolbar = new QToolBar;
    mFiltersToolbar->setOrientation(Qt::Vertical);
    mFiltersToolbar->setIconSize(QSize(TOOLBAR_ICON_SIZE, TOOLBAR_ICON_SIZE));
    mFiltersLayout->addWidget(mFiltersToolbar);

    mFilterActions = new QActionGroup(this);
    mFilterActions->setExclusive(false);
    connect(mFilterActions, SIGNAL(triggered(QAction*)),
            this, SLOT(setFeedTypeState(QAction*)));
}

void NewsBrowserPanel::addFeedType(QEventFeed::FeedType feedType)
{
    QAction* action = new QAction(getIconByFeedType(feedType), QString(), mFilterActions);
    action->setCheckable(true);
    action->setChecked(true);
    action->setData(QVariant::fromValue(feedType));
    mFiltersToolbar->addAction(action);
}

void NewsBrowserPanel::setFeedTypeState(QAction *action)
{
    assert(action->data().canConvert<QEventFeed::FeedType>());

    QEventFeed::FeedType feedType = action->data().value<QEventFeed::FeedType>();
    mModel->setFeedTypeState(feedType, action->isChecked());
}

QIcon NewsBrowserPanel::getIconByFeedType(QEventFeed::FeedType feedType)
{
    QIcon icon;
    switch (feedType) {
    case QEventFeed::messageFeed:
        icon = QIcon(":/res/message.png");
        break;
    case QEventFeed::photoFeed:
        icon = QIcon(":/res/general_image.png");
        break;

    default:
        ;
    }

    return icon;
}

void NewsBrowserPanel::setNewsCount(int count)
{
    mCountLabel->setText(tr("News (%1)").arg(count));
}
